package org.apache.torque.om.mapper;

import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.lang3.function.FailableBiConsumer;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.commons.lang3.tuple.Pair;
import org.apache.torque.TorqueException;

/**
 * Mapping strategy used in processRow method of generated mappers.
 * 
 * @param <T>
 */
public class MappingStrategy<T> {
    
    /**
     * The {@link Pair#getLeft()} is to allow lazy sorting, {@link Pair#getRight()} contains the object to be mapped 
     */
    private final List<Pair<Integer, FailableBiConsumer<ResultSet, T, TorqueException>>> tasks;
    
    private boolean allSet;

    public MappingStrategy()
    {
        this.tasks = new ArrayList<>();
        this.allSet = false;
    }

    public void addColumn(int offset, FailableBiConsumer<ResultSet, T, TorqueException> setter)
    {
        this.tasks.add(Pair.of(offset, setter));
    }

    /**
     * Last finishing steps before execute.
     * 
     * @param num_fields the total column size of the object
     * @param sort <code>true</code> explicitely sort with {@link Pair#getLeft()} of the {@link #tasks}.
     */
    public void finish(int num_fields, boolean sort)
    {
        // The list should already be in the correct order because Criteria loops through the columns
        // in the same order in which they are added to the SQL statement but just in case something weird
        // is being done this gets us closer to the desired contract of ResultSet of looping over monotonically
        // increasing indices of columns only.
        if (sort) {
            this.tasks.sort(Comparator.comparing(Pair::getLeft));
        }
        this.allSet = this.tasks.size() == num_fields;
    }

    public boolean isEmpty()
    {
        return this.tasks.isEmpty();
    }

    public boolean isAllSet() {
        return this.allSet;
    }
    
    public void reset()
    {
        // to use this and to use only a single strategy we might need to Collections.sync(ArrayList).
        this.tasks.clear();
    }

    /**
     * Iterates through the {@link #tasks} list and executes each task.
     * 
     * @param result Resultset
     * @param instance target object
     * 
     * @throws TorqueException
     */
    public void execute(ResultSet result, T instance) throws TorqueException
    {
        for (Pair<Integer, FailableBiConsumer<ResultSet, T, TorqueException>> strategy : this.tasks)
        {
            strategy.getRight().accept(result, instance);
        }
    }
}