/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.analysis.interpolation;

import java.util.Random;
import org.apache.commons.math3.analysis.differentiation.DerivativeStructure;
import org.apache.commons.math3.analysis.interpolation.HermiteInterpolator;
import org.apache.commons.math3.analysis.polynomials.PolynomialFunction;
import org.apache.commons.math3.exception.NoDataException;
import org.apache.commons.math3.util.FastMath;
import org.junit.Assert;
import org.junit.Test;

public class HermiteInterpolatorTest {
    @Test
    public void testZero() {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        interpolator.addSamplePoint(0.0, (double[][])new double[][]{{0.0}});
        for (double x = -10.0; x < 10.0; x += 1.0) {
            DerivativeStructure y = interpolator.value(new DerivativeStructure(1, 1, 0, x))[0];
            Assert.assertEquals((double)0.0, (double)y.getValue(), (double)1.0E-15);
            Assert.assertEquals((double)0.0, (double)y.getPartialDerivative(new int[]{1}), (double)1.0E-15);
        }
        this.checkPolynomial(new PolynomialFunction(new double[]{0.0}), interpolator.getPolynomials()[0]);
    }

    @Test
    public void testQuadratic() {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        interpolator.addSamplePoint(0.0, (double[][])new double[][]{{2.0}});
        interpolator.addSamplePoint(1.0, (double[][])new double[][]{{0.0}});
        interpolator.addSamplePoint(2.0, (double[][])new double[][]{{0.0}});
        for (double x = -10.0; x < 10.0; x += 1.0) {
            DerivativeStructure y = interpolator.value(new DerivativeStructure(1, 1, 0, x))[0];
            Assert.assertEquals((double)((x - 1.0) * (x - 2.0)), (double)y.getValue(), (double)1.0E-15);
            Assert.assertEquals((double)(2.0 * x - 3.0), (double)y.getPartialDerivative(new int[]{1}), (double)1.0E-15);
        }
        this.checkPolynomial(new PolynomialFunction(new double[]{2.0, -3.0, 1.0}), interpolator.getPolynomials()[0]);
    }

    @Test
    public void testMixedDerivatives() {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        interpolator.addSamplePoint(0.0, (double[][])new double[][]{{1.0}, {2.0}});
        interpolator.addSamplePoint(1.0, (double[][])new double[][]{{4.0}});
        interpolator.addSamplePoint(2.0, (double[][])new double[][]{{5.0}, {2.0}});
        Assert.assertEquals((long)4L, (long)interpolator.getPolynomials()[0].degree());
        DerivativeStructure y0 = interpolator.value(new DerivativeStructure(1, 1, 0, 0.0))[0];
        Assert.assertEquals((double)1.0, (double)y0.getValue(), (double)1.0E-15);
        Assert.assertEquals((double)2.0, (double)y0.getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)4.0, (double)interpolator.value(1.0)[0], (double)1.0E-15);
        DerivativeStructure y2 = interpolator.value(new DerivativeStructure(1, 1, 0, 2.0))[0];
        Assert.assertEquals((double)5.0, (double)y2.getValue(), (double)1.0E-15);
        Assert.assertEquals((double)2.0, (double)y2.getPartialDerivative(new int[]{1}), (double)1.0E-15);
        this.checkPolynomial(new PolynomialFunction(new double[]{1.0, 2.0, 4.0, -4.0, 1.0}), interpolator.getPolynomials()[0]);
    }

    @Test
    public void testRandomPolynomialsValuesOnly() {
        Random random = new Random(4805877208711276850L);
        for (int i = 0; i < 100; ++i) {
            int maxDegree = 0;
            PolynomialFunction[] p = new PolynomialFunction[5];
            for (int k = 0; k < p.length; ++k) {
                int degree = random.nextInt(7);
                p[k] = this.randomPolynomial(degree, random);
                maxDegree = FastMath.max((int)maxDegree, (int)degree);
            }
            HermiteInterpolator interpolator = new HermiteInterpolator();
            for (int j = 0; j < 1 + maxDegree; ++j) {
                double x = 0.1 * (double)j;
                double[] values = new double[p.length];
                for (int k = 0; k < p.length; ++k) {
                    values[k] = p[k].value(x);
                }
                interpolator.addSamplePoint(x, (double[][])new double[][]{values});
            }
            for (double x = 0.0; x < 2.0; x += 0.1) {
                double[] values = interpolator.value(x);
                Assert.assertEquals((long)p.length, (long)values.length);
                for (int k = 0; k < p.length; ++k) {
                    Assert.assertEquals((double)p[k].value(x), (double)values[k], (double)(1.0E-8 * FastMath.abs((double)p[k].value(x))));
                }
            }
            PolynomialFunction[] result = interpolator.getPolynomials();
            for (int k = 0; k < p.length; ++k) {
                this.checkPolynomial(p[k], result[k]);
            }
        }
    }

    @Test
    public void testRandomPolynomialsFirstDerivative() {
        Random random = new Random(6271266645131025723L);
        for (int i = 0; i < 100; ++i) {
            int maxDegree = 0;
            PolynomialFunction[] p = new PolynomialFunction[5];
            PolynomialFunction[] pPrime = new PolynomialFunction[5];
            for (int k = 0; k < p.length; ++k) {
                int degree = random.nextInt(7);
                p[k] = this.randomPolynomial(degree, random);
                pPrime[k] = p[k].polynomialDerivative();
                maxDegree = FastMath.max((int)maxDegree, (int)degree);
            }
            HermiteInterpolator interpolator = new HermiteInterpolator();
            for (int j = 0; j < 1 + maxDegree / 2; ++j) {
                double x = 0.1 * (double)j;
                double[] values = new double[p.length];
                double[] derivatives = new double[p.length];
                for (int k = 0; k < p.length; ++k) {
                    values[k] = p[k].value(x);
                    derivatives[k] = pPrime[k].value(x);
                }
                interpolator.addSamplePoint(x, (double[][])new double[][]{values, derivatives});
            }
            for (double x = 0.0; x < 2.0; x += 0.1) {
                DerivativeStructure[] y = interpolator.value(new DerivativeStructure(1, 1, 0, x));
                Assert.assertEquals((long)p.length, (long)y.length);
                for (int k = 0; k < p.length; ++k) {
                    Assert.assertEquals((double)p[k].value(x), (double)y[k].getValue(), (double)(1.0E-8 * FastMath.abs((double)p[k].value(x))));
                    Assert.assertEquals((double)pPrime[k].value(x), (double)y[k].getPartialDerivative(new int[]{1}), (double)(4.0E-8 * FastMath.abs((double)p[k].value(x))));
                }
            }
            PolynomialFunction[] result = interpolator.getPolynomials();
            for (int k = 0; k < p.length; ++k) {
                this.checkPolynomial(p[k], result[k]);
            }
        }
    }

    @Test
    public void testSine() {
        double x;
        HermiteInterpolator interpolator = new HermiteInterpolator();
        for (x = 0.0; x < Math.PI; x += 0.5) {
            interpolator.addSamplePoint(x, (double[][])new double[][]{{FastMath.sin((double)x)}});
        }
        for (x = 0.1; x <= 2.9; x += 0.01) {
            DerivativeStructure y = interpolator.value(new DerivativeStructure(1, 2, 0, x))[0];
            Assert.assertEquals((double)FastMath.sin((double)x), (double)y.getValue(), (double)3.5E-5);
            Assert.assertEquals((double)FastMath.cos((double)x), (double)y.getPartialDerivative(new int[]{1}), (double)1.3E-4);
            Assert.assertEquals((double)(-FastMath.sin((double)x)), (double)y.getPartialDerivative(new int[]{2}), (double)0.0029);
        }
    }

    @Test
    public void testSquareRoot() {
        double x;
        HermiteInterpolator interpolator = new HermiteInterpolator();
        for (x = 1.0; x < 3.6; x += 0.5) {
            interpolator.addSamplePoint(x, (double[][])new double[][]{{FastMath.sqrt((double)x)}});
        }
        for (x = 1.1; x < 3.5; x += 0.01) {
            DerivativeStructure y = interpolator.value(new DerivativeStructure(1, 1, 0, x))[0];
            Assert.assertEquals((double)FastMath.sqrt((double)x), (double)y.getValue(), (double)1.5E-4);
            Assert.assertEquals((double)(0.5 / FastMath.sqrt((double)x)), (double)y.getPartialDerivative(new int[]{1}), (double)8.5E-4);
        }
    }

    @Test
    public void testWikipedia() {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        interpolator.addSamplePoint(-1.0, (double[][])new double[][]{{2.0}, {-8.0}, {56.0}});
        interpolator.addSamplePoint(0.0, (double[][])new double[][]{{1.0}, {0.0}, {0.0}});
        interpolator.addSamplePoint(1.0, (double[][])new double[][]{{2.0}, {8.0}, {56.0}});
        for (double x = -1.0; x <= 1.0; x += 0.125) {
            DerivativeStructure y = interpolator.value(new DerivativeStructure(1, 1, 0, x))[0];
            double x2 = x * x;
            double x4 = x2 * x2;
            double x8 = x4 * x4;
            Assert.assertEquals((double)(x8 + 1.0), (double)y.getValue(), (double)1.0E-15);
            Assert.assertEquals((double)(8.0 * x4 * x2 * x), (double)y.getPartialDerivative(new int[]{1}), (double)1.0E-15);
        }
        this.checkPolynomial(new PolynomialFunction(new double[]{1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0}), interpolator.getPolynomials()[0]);
    }

    @Test
    public void testOnePointParabola() {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        interpolator.addSamplePoint(0.0, (double[][])new double[][]{{1.0}, {1.0}, {2.0}});
        for (double x = -1.0; x <= 1.0; x += 0.125) {
            DerivativeStructure y = interpolator.value(new DerivativeStructure(1, 1, 0, x))[0];
            Assert.assertEquals((double)(1.0 + x * (1.0 + x)), (double)y.getValue(), (double)1.0E-15);
            Assert.assertEquals((double)(1.0 + 2.0 * x), (double)y.getPartialDerivative(new int[]{1}), (double)1.0E-15);
        }
        this.checkPolynomial(new PolynomialFunction(new double[]{1.0, 1.0, 1.0}), interpolator.getPolynomials()[0]);
    }

    private PolynomialFunction randomPolynomial(int degree, Random random) {
        double[] coeff = new double[1 + degree];
        for (int j = 0; j < degree; ++j) {
            coeff[j] = random.nextDouble();
        }
        return new PolynomialFunction(coeff);
    }

    @Test(expected=NoDataException.class)
    public void testEmptySample() {
        new HermiteInterpolator().value(0.0);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testDuplicatedAbscissa() {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        interpolator.addSamplePoint(1.0, (double[][])new double[][]{{0.0}});
        interpolator.addSamplePoint(1.0, (double[][])new double[][]{{1.0}});
    }

    private void checkPolynomial(PolynomialFunction expected, PolynomialFunction result) {
        int i;
        Assert.assertTrue((result.degree() >= expected.degree() ? 1 : 0) != 0);
        double[] cE = expected.getCoefficients();
        double[] cR = result.getCoefficients();
        for (i = 0; i < cE.length; ++i) {
            Assert.assertEquals((double)cE[i], (double)cR[i], (double)(1.0E-8 * FastMath.abs((double)cE[i])));
        }
        for (i = cE.length; i < cR.length; ++i) {
            Assert.assertEquals((double)0.0, (double)cR[i], (double)1.0E-9);
        }
    }
}

