/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.archivers.zip;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.CRC32;
import org.apache.commons.compress.AbstractTestCase;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.zip.ResourceAlignmentExtraField;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipExtraField;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.compress.archivers.zip.ZipSplitReadOnlySeekableByteChannel;
import org.apache.commons.compress.utils.ByteUtils;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.commons.compress.utils.SeekableInMemoryByteChannel;
import org.apache.commons.lang3.SystemUtils;
import org.junit.Assume;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;

public class ZipFileTest
extends AbstractTestCase {
    private static final int OUT_OF_MEMORY = 137;
    private ZipFile zf = null;

    private static void assertEntryName(ArrayList<ZipArchiveEntry> entries, int index, String expectedName) {
        ZipArchiveEntry ze = entries.get(index);
        Assertions.assertEquals((Object)("src/main/java/org/apache/commons/compress/archivers/zip/" + expectedName + ".java"), (Object)ze.getName());
    }

    private static void nameSource(String archive, String entry, ZipArchiveEntry.NameSource expected) throws Exception {
        try (ZipFile zf = new ZipFile(ZipFileTest.getFile(archive));){
            ZipArchiveEntry ze = zf.getEntry(entry);
            Assertions.assertEquals((Object)entry, (Object)ze.getName());
            Assertions.assertEquals((Object)expected, (Object)ze.getNameSource());
        }
    }

    private void assertAllReadMethods(byte[] expected, ZipFile zipFile, ZipArchiveEntry entry) {
        byte[] full;
        InputStream stream;
        try {
            stream = this.zf.getInputStream(entry);
            try {
                full = IOUtils.toByteArray((InputStream)stream);
                Assertions.assertArrayEquals((byte[])expected, (byte[])full);
            }
            finally {
                if (stream != null) {
                    stream.close();
                }
            }
        }
        catch (IOException ex) {
            throw new AssertionError((Object)ex);
        }
        try {
            stream = this.zf.getInputStream(entry);
            try {
                byte[] bytes = new byte[262144];
                int read = stream.read(bytes);
                full = read < 0 ? ByteUtils.EMPTY_BYTE_ARRAY : this.readStreamRest(bytes, read, stream);
                Assertions.assertArrayEquals((byte[])expected, (byte[])full);
            }
            finally {
                if (stream != null) {
                    stream.close();
                }
            }
        }
        catch (IOException ex) {
            throw new AssertionError((Object)ex);
        }
        try {
            stream = this.zf.getInputStream(entry);
            try {
                int single = stream.read();
                if (single < 0) {
                    full = ByteUtils.EMPTY_BYTE_ARRAY;
                } else {
                    byte[] big = new byte[262144];
                    big[0] = (byte)single;
                    int read = stream.read(big, 1, big.length - 1);
                    full = read < 0 ? new byte[]{(byte)single} : this.readStreamRest(big, read + 1, stream);
                }
                Assertions.assertArrayEquals((byte[])expected, (byte[])full);
            }
            finally {
                if (stream != null) {
                    stream.close();
                }
            }
        }
        catch (IOException ex) {
            throw new AssertionError((Object)ex);
        }
    }

    private void assertFileEqualIgnoreEndOfLine(File file1, File file2) throws IOException {
        List<String> linesOfFile1 = Files.readAllLines(Paths.get(file1.getCanonicalPath(), new String[0]), StandardCharsets.UTF_8);
        List<String> linesOfFile2 = Files.readAllLines(Paths.get(file2.getCanonicalPath(), new String[0]), StandardCharsets.UTF_8);
        if (linesOfFile1.size() != linesOfFile2.size()) {
            Assertions.fail((String)("files not equal : " + file1.getName() + " , " + file2.getName()));
        }
        for (int i = 0; i < linesOfFile1.size(); ++i) {
            String tempLineInFile1 = linesOfFile1.get(i).replace("\r\n", "\n");
            String tempLineInFile2 = linesOfFile2.get(i).replace("\r\n", "\n");
            Assertions.assertEquals((Object)tempLineInFile1, (Object)tempLineInFile2);
        }
    }

    private void assertFileEqualsToEntry(File fileToCompare, ZipArchiveEntry entry, ZipFile zipFile) throws IOException {
        byte[] buffer = new byte[10240];
        File tempFile = File.createTempFile("temp", "txt");
        try (OutputStream outputStream = Files.newOutputStream(tempFile.toPath(), new OpenOption[0]);
             InputStream inputStream = zipFile.getInputStream(entry);){
            int readLen;
            while ((readLen = inputStream.read(buffer)) > 0) {
                outputStream.write(buffer, 0, readLen);
            }
        }
        this.assertFileEqualIgnoreEndOfLine(fileToCompare, tempFile);
    }

    private long calculateCrc32(byte[] content) {
        CRC32 crc = new CRC32();
        crc.update(content);
        return crc.getValue();
    }

    @Test
    public void extractFileLiesAcrossSplitZipSegmentsCreatedByWinrar() throws Exception {
        File lastFile = ZipFileTest.getFile("COMPRESS-477/split_zip_created_by_winrar/split_zip_created_by_winrar.zip");
        try (SeekableByteChannel channel = ZipSplitReadOnlySeekableByteChannel.buildFromLastSplitSegment((File)lastFile);){
            this.zf = new ZipFile(channel);
            ZipArchiveEntry zipEntry = this.zf.getEntry("commons-compress/src/main/java/org/apache/commons/compress/archivers/zip/ZipArchiveInputStream.java");
            File fileToCompare = ZipFileTest.getFile("COMPRESS-477/split_zip_created_by_winrar/file_to_compare_1");
            this.assertFileEqualsToEntry(fileToCompare, zipEntry, this.zf);
        }
    }

    @Test
    public void extractFileLiesAcrossSplitZipSegmentsCreatedByZip() throws Exception {
        File lastFile = ZipFileTest.getFile("COMPRESS-477/split_zip_created_by_zip/split_zip_created_by_zip.zip");
        try (SeekableByteChannel channel = ZipSplitReadOnlySeekableByteChannel.buildFromLastSplitSegment((File)lastFile);){
            this.zf = new ZipFile(channel);
            ZipArchiveEntry zipEntry = this.zf.getEntry("commons-compress/src/main/java/org/apache/commons/compress/archivers/dump/UnsupportedCompressionAlgorithmException.java");
            File fileToCompare = ZipFileTest.getFile("COMPRESS-477/split_zip_created_by_zip/file_to_compare_1");
            this.assertFileEqualsToEntry(fileToCompare, zipEntry, this.zf);
            zipEntry = this.zf.getEntry("commons-compress/src/main/java/org/apache/commons/compress/compressors/deflate/DeflateParameters.java");
            fileToCompare = ZipFileTest.getFile("COMPRESS-477/split_zip_created_by_zip/file_to_compare_2");
            this.assertFileEqualsToEntry(fileToCompare, zipEntry, this.zf);
        }
    }

    @Test
    public void extractFileLiesAcrossSplitZipSegmentsCreatedByZipOfZip64() throws Exception {
        File lastFile = ZipFileTest.getFile("COMPRESS-477/split_zip_created_by_zip/split_zip_created_by_zip_zip64.zip");
        try (SeekableByteChannel channel = ZipSplitReadOnlySeekableByteChannel.buildFromLastSplitSegment((File)lastFile);){
            this.zf = new ZipFile(channel);
            ZipArchiveEntry zipEntry = this.zf.getEntry("commons-compress/src/main/java/org/apache/commons/compress/archivers/dump/UnsupportedCompressionAlgorithmException.java");
            File fileToCompare = ZipFileTest.getFile("COMPRESS-477/split_zip_created_by_zip/file_to_compare_1");
            this.assertFileEqualsToEntry(fileToCompare, zipEntry, this.zf);
            zipEntry = this.zf.getEntry("commons-compress/src/main/java/org/apache/commons/compress/compressors/deflate/DeflateParameters.java");
            fileToCompare = ZipFileTest.getFile("COMPRESS-477/split_zip_created_by_zip/file_to_compare_2");
            this.assertFileEqualsToEntry(fileToCompare, zipEntry, this.zf);
        }
    }

    private void multiByteReadConsistentlyReturnsMinusOneAtEof(File file) throws Exception {
        byte[] buf = new byte[2];
        try (ZipFile archive = new ZipFile(file);){
            ZipArchiveEntry e = (ZipArchiveEntry)archive.getEntries().nextElement();
            try (InputStream is = archive.getInputStream(e);){
                IOUtils.toByteArray((InputStream)is);
                Assertions.assertEquals((int)-1, (int)is.read(buf));
                Assertions.assertEquals((int)-1, (int)is.read(buf));
            }
        }
    }

    @Test
    public void multiByteReadConsistentlyReturnsMinusOneAtEofUsingBzip2() throws Exception {
        this.multiByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("bzip2-zip.zip"));
    }

    @Test
    public void multiByteReadConsistentlyReturnsMinusOneAtEofUsingDeflate() throws Exception {
        this.multiByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("bla.zip"));
    }

    @Test
    public void multiByteReadConsistentlyReturnsMinusOneAtEofUsingDeflate64() throws Exception {
        this.multiByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("COMPRESS-380/COMPRESS-380.zip"));
    }

    @Test
    public void multiByteReadConsistentlyReturnsMinusOneAtEofUsingExplode() throws Exception {
        this.multiByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("imploding-8Kdict-3trees.zip"));
    }

    @Test
    public void multiByteReadConsistentlyReturnsMinusOneAtEofUsingStore() throws Exception {
        this.multiByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("COMPRESS-264.zip"));
    }

    @Test
    public void multiByteReadConsistentlyReturnsMinusOneAtEofUsingUnshrink() throws Exception {
        this.multiByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("SHRUNK.ZIP"));
    }

    @Test
    public void nameSourceDefaultsToName() throws Exception {
        ZipFileTest.nameSource("bla.zip", "test1.xml", ZipArchiveEntry.NameSource.NAME);
    }

    @Test
    public void nameSourceIsSetToEFS() throws Exception {
        ZipFileTest.nameSource("utf8-7zip-test.zip", "\u20ac_for_Dollar.txt", ZipArchiveEntry.NameSource.NAME_WITH_EFS_FLAG);
    }

    @Test
    public void nameSourceIsSetToUnicodeExtraField() throws Exception {
        ZipFileTest.nameSource("utf8-winzip-test.zip", "\u20ac_for_Dollar.txt", ZipArchiveEntry.NameSource.UNICODE_EXTRA_FIELD);
    }

    @Test
    public void readDeflate64CompressedStream() throws Exception {
        File input = ZipFileTest.getFile("COMPRESS-380/COMPRESS-380-input");
        File archive = ZipFileTest.getFile("COMPRESS-380/COMPRESS-380.zip");
        try (InputStream in = Files.newInputStream(input.toPath(), new OpenOption[0]);
             ZipFile zf = new ZipFile(archive);){
            byte[] orig = IOUtils.toByteArray((InputStream)in);
            ZipArchiveEntry e = zf.getEntry("input2");
            try (InputStream s = zf.getInputStream(e);){
                byte[] fromZip = IOUtils.toByteArray((InputStream)s);
                Assertions.assertArrayEquals((byte[])orig, (byte[])fromZip);
            }
        }
    }

    private void readOrderTest() throws Exception {
        File archive = ZipFileTest.getFile("ordertest.zip");
        this.zf = new ZipFile(archive);
    }

    private byte[] readStreamRest(byte[] beginning, int length, InputStream stream) throws IOException {
        byte[] rest = IOUtils.toByteArray((InputStream)stream);
        byte[] full = new byte[length + rest.length];
        System.arraycopy(beginning, 0, full, 0, length);
        System.arraycopy(rest, 0, full, length, rest.length);
        return full;
    }

    private void singleByteReadConsistentlyReturnsMinusOneAtEof(File file) throws Exception {
        try (ZipFile archive = new ZipFile(file);){
            ZipArchiveEntry e = (ZipArchiveEntry)archive.getEntries().nextElement();
            try (InputStream is = archive.getInputStream(e);){
                IOUtils.toByteArray((InputStream)is);
                Assertions.assertEquals((int)-1, (int)is.read());
                Assertions.assertEquals((int)-1, (int)is.read());
            }
        }
    }

    @Test
    public void singleByteReadConsistentlyReturnsMinusOneAtEofUsingBzip2() throws Exception {
        this.singleByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("bzip2-zip.zip"));
    }

    @Test
    public void singleByteReadConsistentlyReturnsMinusOneAtEofUsingDeflate() throws Exception {
        this.singleByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("bla.zip"));
    }

    @Test
    public void singleByteReadConsistentlyReturnsMinusOneAtEofUsingDeflate64() throws Exception {
        this.singleByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("COMPRESS-380/COMPRESS-380.zip"));
    }

    @Test
    public void singleByteReadConsistentlyReturnsMinusOneAtEofUsingExplode() throws Exception {
        this.singleByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("imploding-8Kdict-3trees.zip"));
    }

    @Test
    public void singleByteReadConsistentlyReturnsMinusOneAtEofUsingStore() throws Exception {
        this.singleByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("COMPRESS-264.zip"));
    }

    @Test
    public void singleByteReadConsistentlyReturnsMinusOneAtEofUsingUnshrink() throws Exception {
        this.singleByteReadConsistentlyReturnsMinusOneAtEof(ZipFileTest.getFile("SHRUNK.ZIP"));
    }

    @AfterEach
    public void tearDownClose() {
        ZipFile.closeQuietly((ZipFile)this.zf);
    }

    @Test
    public void testCDOrder() throws Exception {
        this.readOrderTest();
        ArrayList<ZipArchiveEntry> l = Collections.list(this.zf.getEntries());
        ZipFileTest.assertEntryName(l, 0, "AbstractUnicodeExtraField");
        ZipFileTest.assertEntryName(l, 1, "AsiExtraField");
        ZipFileTest.assertEntryName(l, 2, "ExtraFieldUtils");
        ZipFileTest.assertEntryName(l, 3, "FallbackZipEncoding");
        ZipFileTest.assertEntryName(l, 4, "GeneralPurposeBit");
        ZipFileTest.assertEntryName(l, 5, "JarMarker");
        ZipFileTest.assertEntryName(l, 6, "NioZipEncoding");
        ZipFileTest.assertEntryName(l, 7, "Simple8BitZipEncoding");
        ZipFileTest.assertEntryName(l, 8, "UnicodeCommentExtraField");
        ZipFileTest.assertEntryName(l, 9, "UnicodePathExtraField");
        ZipFileTest.assertEntryName(l, 10, "UnixStat");
        ZipFileTest.assertEntryName(l, 11, "UnparseableExtraFieldData");
        ZipFileTest.assertEntryName(l, 12, "UnrecognizedExtraField");
        ZipFileTest.assertEntryName(l, 13, "ZipArchiveEntry");
        ZipFileTest.assertEntryName(l, 14, "ZipArchiveInputStream");
        ZipFileTest.assertEntryName(l, 15, "ZipArchiveOutputStream");
        ZipFileTest.assertEntryName(l, 16, "ZipEncoding");
        ZipFileTest.assertEntryName(l, 17, "ZipEncodingHelper");
        ZipFileTest.assertEntryName(l, 18, "ZipExtraField");
        ZipFileTest.assertEntryName(l, 19, "ZipUtil");
        ZipFileTest.assertEntryName(l, 20, "ZipLong");
        ZipFileTest.assertEntryName(l, 21, "ZipShort");
        ZipFileTest.assertEntryName(l, 22, "ZipFile");
    }

    @Test
    public void testCDOrderInMemory() throws Exception {
        byte[] data;
        try (InputStream fis = ZipFileTest.newInputStream("ordertest.zip");){
            data = IOUtils.toByteArray((InputStream)fis);
        }
        try (SeekableInMemoryByteChannel channel = new SeekableInMemoryByteChannel(data);){
            this.zf = new ZipFile((SeekableByteChannel)channel, "UTF8");
            ArrayList<ZipArchiveEntry> l = Collections.list(this.zf.getEntries());
            ZipFileTest.assertEntryName(l, 0, "AbstractUnicodeExtraField");
            ZipFileTest.assertEntryName(l, 1, "AsiExtraField");
            ZipFileTest.assertEntryName(l, 2, "ExtraFieldUtils");
            ZipFileTest.assertEntryName(l, 3, "FallbackZipEncoding");
            ZipFileTest.assertEntryName(l, 4, "GeneralPurposeBit");
            ZipFileTest.assertEntryName(l, 5, "JarMarker");
            ZipFileTest.assertEntryName(l, 6, "NioZipEncoding");
            ZipFileTest.assertEntryName(l, 7, "Simple8BitZipEncoding");
            ZipFileTest.assertEntryName(l, 8, "UnicodeCommentExtraField");
            ZipFileTest.assertEntryName(l, 9, "UnicodePathExtraField");
            ZipFileTest.assertEntryName(l, 10, "UnixStat");
            ZipFileTest.assertEntryName(l, 11, "UnparseableExtraFieldData");
            ZipFileTest.assertEntryName(l, 12, "UnrecognizedExtraField");
            ZipFileTest.assertEntryName(l, 13, "ZipArchiveEntry");
            ZipFileTest.assertEntryName(l, 14, "ZipArchiveInputStream");
            ZipFileTest.assertEntryName(l, 15, "ZipArchiveOutputStream");
            ZipFileTest.assertEntryName(l, 16, "ZipEncoding");
            ZipFileTest.assertEntryName(l, 17, "ZipEncodingHelper");
            ZipFileTest.assertEntryName(l, 18, "ZipExtraField");
            ZipFileTest.assertEntryName(l, 19, "ZipUtil");
            ZipFileTest.assertEntryName(l, 20, "ZipLong");
            ZipFileTest.assertEntryName(l, 21, "ZipShort");
            ZipFileTest.assertEntryName(l, 22, "ZipFile");
        }
    }

    @Test
    public void testConcurrentReadFile() throws Exception {
        File archive = ZipFileTest.getFile("mixed.zip");
        this.zf = new ZipFile(archive);
        HashMap<String, byte[]> content = new HashMap<String, byte[]>();
        for (ZipArchiveEntry entry : Collections.list(this.zf.getEntries())) {
            InputStream inputStream = this.zf.getInputStream(entry);
            try {
                content.put(entry.getName(), IOUtils.toByteArray((InputStream)inputStream));
            }
            finally {
                if (inputStream == null) continue;
                inputStream.close();
            }
        }
        AtomicInteger passedCount = new AtomicInteger();
        Runnable run = () -> {
            for (ZipArchiveEntry entry : Collections.list(this.zf.getEntries())) {
                this.assertAllReadMethods((byte[])content.get(entry.getName()), this.zf, entry);
            }
            passedCount.incrementAndGet();
        };
        Thread t0 = new Thread(run);
        Thread t1 = new Thread(run);
        t0.start();
        t1.start();
        t0.join();
        t1.join();
        Assertions.assertEquals((int)2, (int)passedCount.get());
    }

    @Test
    public void testConcurrentReadSeekable() throws Exception {
        byte[] data;
        try (InputStream fis = ZipFileTest.newInputStream("mixed.zip");){
            data = IOUtils.toByteArray((InputStream)fis);
        }
        try (SeekableInMemoryByteChannel channel = new SeekableInMemoryByteChannel(data);){
            this.zf = new ZipFile((SeekableByteChannel)channel, "UTF8");
            HashMap<String, byte[]> content = new HashMap<String, byte[]>();
            for (ZipArchiveEntry entry : Collections.list(this.zf.getEntries())) {
                InputStream inputStream = this.zf.getInputStream(entry);
                try {
                    content.put(entry.getName(), IOUtils.toByteArray((InputStream)inputStream));
                }
                finally {
                    if (inputStream == null) continue;
                    inputStream.close();
                }
            }
            AtomicInteger passedCount = new AtomicInteger();
            Runnable run = () -> {
                for (ZipArchiveEntry entry : Collections.list(this.zf.getEntries())) {
                    this.assertAllReadMethods((byte[])content.get(entry.getName()), this.zf, entry);
                }
                passedCount.incrementAndGet();
            };
            Thread t0 = new Thread(run);
            Thread t1 = new Thread(run);
            t0.start();
            t1.start();
            t0.join();
            t1.join();
            Assertions.assertEquals((int)2, (int)passedCount.get());
        }
    }

    @Test
    public void testDelayedOffsetsAndSizes() throws Exception {
        ZipArchiveEntry inflatedEntry;
        ByteArrayOutputStream zipContent = new ByteArrayOutputStream();
        try (ZipArchiveOutputStream zipOutput = new ZipArchiveOutputStream((OutputStream)zipContent);){
            inflatedEntry = new ZipArchiveEntry("inflated.txt");
            inflatedEntry.setMethod(8);
            zipOutput.putArchiveEntry((ArchiveEntry)inflatedEntry);
            zipOutput.write("Hello Deflated\n".getBytes());
            zipOutput.closeArchiveEntry();
            byte[] storedContent = "Hello Stored\n".getBytes();
            ZipArchiveEntry storedEntry = new ZipArchiveEntry("stored.txt");
            storedEntry.setMethod(0);
            storedEntry.setSize((long)storedContent.length);
            storedEntry.setCrc(this.calculateCrc32(storedContent));
            zipOutput.putArchiveEntry((ArchiveEntry)storedEntry);
            zipOutput.write("Hello Stored\n".getBytes());
            zipOutput.closeArchiveEntry();
        }
        try (ZipFile zf = new ZipFile((SeekableByteChannel)new SeekableInMemoryByteChannel(zipContent.toByteArray()));){
            inflatedEntry = zf.getEntry("inflated.txt");
            Assertions.assertNotEquals((long)-1L, (long)inflatedEntry.getLocalHeaderOffset());
            Assertions.assertNotEquals((long)-1L, (long)inflatedEntry.getDataOffset());
            Assertions.assertTrue((boolean)inflatedEntry.isStreamContiguous());
            Assertions.assertNotEquals((long)-1L, (long)inflatedEntry.getCompressedSize());
            Assertions.assertNotEquals((long)-1L, (long)inflatedEntry.getSize());
            ZipArchiveEntry storedEntry = zf.getEntry("stored.txt");
            Assertions.assertNotEquals((long)-1L, (long)storedEntry.getLocalHeaderOffset());
            Assertions.assertNotEquals((long)-1L, (long)storedEntry.getDataOffset());
            Assertions.assertTrue((boolean)inflatedEntry.isStreamContiguous());
            Assertions.assertNotEquals((long)-1L, (long)storedEntry.getCompressedSize());
            Assertions.assertNotEquals((long)-1L, (long)storedEntry.getSize());
        }
    }

    @Test
    public void testDoubleClose() throws Exception {
        this.readOrderTest();
        this.zf.close();
        Assertions.assertDoesNotThrow(() -> ((ZipFile)this.zf).close(), (String)"Caught exception of second close");
    }

    @Test
    public void testDuplicateEntry() throws Exception {
        File archive = ZipFileTest.getFile("COMPRESS-227.zip");
        this.zf = new ZipFile(archive);
        ZipArchiveEntry ze = this.zf.getEntry("test1.txt");
        Assertions.assertNotNull((Object)ze);
        try (InputStream inputStream = this.zf.getInputStream(ze);){
            Assertions.assertNotNull((Object)inputStream);
            int numberOfEntries = 0;
            for (ZipArchiveEntry entry : this.zf.getEntries("test1.txt")) {
                ++numberOfEntries;
                InputStream inputStream2 = this.zf.getInputStream(entry);
                try {
                    Assertions.assertNotNull((Object)inputStream2);
                }
                finally {
                    if (inputStream2 == null) continue;
                    inputStream2.close();
                }
            }
            Assertions.assertEquals((int)2, (int)numberOfEntries);
        }
    }

    @Test
    public void testEntryAlignment() throws Exception {
        try (SeekableInMemoryByteChannel zipContent = new SeekableInMemoryByteChannel();){
            ZipArchiveEntry inflatedEntry;
            try (ZipArchiveOutputStream zipOutput = new ZipArchiveOutputStream((SeekableByteChannel)zipContent);){
                inflatedEntry = new ZipArchiveEntry("inflated.txt");
                inflatedEntry.setMethod(8);
                inflatedEntry.setAlignment(1024);
                zipOutput.putArchiveEntry((ArchiveEntry)inflatedEntry);
                zipOutput.write("Hello Deflated\n".getBytes(StandardCharsets.UTF_8));
                zipOutput.closeArchiveEntry();
                ZipArchiveEntry storedEntry = new ZipArchiveEntry("stored.txt");
                storedEntry.setMethod(0);
                storedEntry.setAlignment(1024);
                zipOutput.putArchiveEntry((ArchiveEntry)storedEntry);
                zipOutput.write("Hello Stored\n".getBytes(StandardCharsets.UTF_8));
                zipOutput.closeArchiveEntry();
                ZipArchiveEntry storedEntry2 = new ZipArchiveEntry("stored2.txt");
                storedEntry2.setMethod(0);
                storedEntry2.setAlignment(1024);
                storedEntry2.addExtraField((ZipExtraField)new ResourceAlignmentExtraField(1));
                zipOutput.putArchiveEntry((ArchiveEntry)storedEntry2);
                zipOutput.write("Hello overload-alignment Stored\n".getBytes(StandardCharsets.UTF_8));
                zipOutput.closeArchiveEntry();
                ZipArchiveEntry storedEntry3 = new ZipArchiveEntry("stored3.txt");
                storedEntry3.setMethod(0);
                storedEntry3.addExtraField((ZipExtraField)new ResourceAlignmentExtraField(1024));
                zipOutput.putArchiveEntry((ArchiveEntry)storedEntry3);
                zipOutput.write("Hello copy-alignment Stored\n".getBytes(StandardCharsets.UTF_8));
                zipOutput.closeArchiveEntry();
            }
            try (ZipFile zf = new ZipFile((SeekableByteChannel)new SeekableInMemoryByteChannel(Arrays.copyOfRange(zipContent.array(), 0, (int)zipContent.size())));){
                inflatedEntry = zf.getEntry("inflated.txt");
                ResourceAlignmentExtraField inflatedAlignmentEx = (ResourceAlignmentExtraField)inflatedEntry.getExtraField(ResourceAlignmentExtraField.ID);
                Assertions.assertNotEquals((long)-1L, (long)inflatedEntry.getCompressedSize());
                Assertions.assertNotEquals((long)-1L, (long)inflatedEntry.getSize());
                Assertions.assertEquals((long)0L, (long)(inflatedEntry.getDataOffset() % 1024L));
                Assertions.assertNotNull((Object)inflatedAlignmentEx);
                Assertions.assertEquals((int)1024, (int)inflatedAlignmentEx.getAlignment());
                Assertions.assertFalse((boolean)inflatedAlignmentEx.allowMethodChange());
                try (InputStream stream = zf.getInputStream(inflatedEntry);){
                    Assertions.assertEquals((Object)"Hello Deflated\n", (Object)new String(IOUtils.toByteArray((InputStream)stream), StandardCharsets.UTF_8));
                }
                ZipArchiveEntry storedEntry = zf.getEntry("stored.txt");
                ResourceAlignmentExtraField storedAlignmentEx = (ResourceAlignmentExtraField)storedEntry.getExtraField(ResourceAlignmentExtraField.ID);
                Assertions.assertNotEquals((long)-1L, (long)storedEntry.getCompressedSize());
                Assertions.assertNotEquals((long)-1L, (long)storedEntry.getSize());
                Assertions.assertEquals((long)0L, (long)(storedEntry.getDataOffset() % 1024L));
                Assertions.assertNotNull((Object)storedAlignmentEx);
                Assertions.assertEquals((int)1024, (int)storedAlignmentEx.getAlignment());
                Assertions.assertFalse((boolean)storedAlignmentEx.allowMethodChange());
                try (InputStream stream = zf.getInputStream(storedEntry);){
                    Assertions.assertEquals((Object)"Hello Stored\n", (Object)new String(IOUtils.toByteArray((InputStream)stream), StandardCharsets.UTF_8));
                }
                ZipArchiveEntry storedEntry2 = zf.getEntry("stored2.txt");
                ResourceAlignmentExtraField stored2AlignmentEx = (ResourceAlignmentExtraField)storedEntry2.getExtraField(ResourceAlignmentExtraField.ID);
                Assertions.assertNotEquals((long)-1L, (long)storedEntry2.getCompressedSize());
                Assertions.assertNotEquals((long)-1L, (long)storedEntry2.getSize());
                Assertions.assertEquals((long)0L, (long)(storedEntry2.getDataOffset() % 1024L));
                Assertions.assertNotNull((Object)stored2AlignmentEx);
                Assertions.assertEquals((int)1024, (int)stored2AlignmentEx.getAlignment());
                Assertions.assertFalse((boolean)stored2AlignmentEx.allowMethodChange());
                try (InputStream stream = zf.getInputStream(storedEntry2);){
                    Assertions.assertEquals((Object)"Hello overload-alignment Stored\n", (Object)new String(IOUtils.toByteArray((InputStream)stream), StandardCharsets.UTF_8));
                }
                ZipArchiveEntry storedEntry3 = zf.getEntry("stored3.txt");
                ResourceAlignmentExtraField stored3AlignmentEx = (ResourceAlignmentExtraField)storedEntry3.getExtraField(ResourceAlignmentExtraField.ID);
                Assertions.assertNotEquals((long)-1L, (long)storedEntry3.getCompressedSize());
                Assertions.assertNotEquals((long)-1L, (long)storedEntry3.getSize());
                Assertions.assertEquals((long)0L, (long)(storedEntry3.getDataOffset() % 1024L));
                Assertions.assertNotNull((Object)stored3AlignmentEx);
                Assertions.assertEquals((int)1024, (int)stored3AlignmentEx.getAlignment());
                Assertions.assertFalse((boolean)stored3AlignmentEx.allowMethodChange());
                try (InputStream stream = zf.getInputStream(storedEntry3);){
                    Assertions.assertEquals((Object)"Hello copy-alignment Stored\n", (Object)new String(IOUtils.toByteArray((InputStream)stream), StandardCharsets.UTF_8));
                }
            }
        }
    }

    @Test
    public void testEntryAlignmentExceed() throws Exception {
        try (SeekableInMemoryByteChannel zipContent = new SeekableInMemoryByteChannel();
             ZipArchiveOutputStream zipOutput = new ZipArchiveOutputStream((SeekableByteChannel)zipContent);){
            ZipArchiveEntry inflatedEntry = new ZipArchiveEntry("inflated.txt");
            inflatedEntry.setMethod(0);
            Assertions.assertThrows(IllegalArgumentException.class, () -> inflatedEntry.setAlignment(131072));
        }
    }

    @Test
    public void testExcessDataInZip64ExtraField() throws Exception {
        File archive = ZipFileTest.getFile("COMPRESS-228.zip");
        this.zf = new ZipFile(archive);
        ZipArchiveEntry ze = this.zf.getEntry("src/main/java/org/apache/commons/compress/archivers/zip/ZipFile.java");
        Assertions.assertEquals((long)26101L, (long)ze.getSize());
    }

    @Test
    public void testInvalidAlignment() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> new ZipArchiveEntry("dummy").setAlignment(3));
    }

    @Test
    public void testOffsets() throws Exception {
        File archive = ZipFileTest.getFile("mixed.zip");
        try (ZipFile zf = new ZipFile(archive);){
            ZipArchiveEntry inflatedEntry = zf.getEntry("inflated.txt");
            Assertions.assertEquals((long)0L, (long)inflatedEntry.getLocalHeaderOffset());
            Assertions.assertEquals((long)70L, (long)inflatedEntry.getDataOffset());
            Assertions.assertTrue((boolean)inflatedEntry.isStreamContiguous());
            ZipArchiveEntry storedEntry = zf.getEntry("stored.txt");
            Assertions.assertEquals((long)22674L, (long)storedEntry.getLocalHeaderOffset());
            Assertions.assertEquals((long)22742L, (long)storedEntry.getDataOffset());
            Assertions.assertTrue((boolean)inflatedEntry.isStreamContiguous());
        }
    }

    @Test
    public void testPhysicalOrder() throws Exception {
        this.readOrderTest();
        ArrayList<ZipArchiveEntry> l = Collections.list(this.zf.getEntriesInPhysicalOrder());
        ZipFileTest.assertEntryName(l, 0, "AbstractUnicodeExtraField");
        ZipFileTest.assertEntryName(l, 1, "AsiExtraField");
        ZipFileTest.assertEntryName(l, 2, "ExtraFieldUtils");
        ZipFileTest.assertEntryName(l, 3, "FallbackZipEncoding");
        ZipFileTest.assertEntryName(l, 4, "GeneralPurposeBit");
        ZipFileTest.assertEntryName(l, 5, "JarMarker");
        ZipFileTest.assertEntryName(l, 6, "NioZipEncoding");
        ZipFileTest.assertEntryName(l, 7, "Simple8BitZipEncoding");
        ZipFileTest.assertEntryName(l, 8, "UnicodeCommentExtraField");
        ZipFileTest.assertEntryName(l, 9, "UnicodePathExtraField");
        ZipFileTest.assertEntryName(l, 10, "UnixStat");
        ZipFileTest.assertEntryName(l, 11, "UnparseableExtraFieldData");
        ZipFileTest.assertEntryName(l, 12, "UnrecognizedExtraField");
        ZipFileTest.assertEntryName(l, 13, "ZipArchiveEntry");
        ZipFileTest.assertEntryName(l, 14, "ZipArchiveInputStream");
        ZipFileTest.assertEntryName(l, 15, "ZipArchiveOutputStream");
        ZipFileTest.assertEntryName(l, 16, "ZipEncoding");
        ZipFileTest.assertEntryName(l, 17, "ZipEncodingHelper");
        ZipFileTest.assertEntryName(l, 18, "ZipExtraField");
        ZipFileTest.assertEntryName(l, 19, "ZipFile");
        ZipFileTest.assertEntryName(l, 20, "ZipLong");
        ZipFileTest.assertEntryName(l, 21, "ZipShort");
        ZipFileTest.assertEntryName(l, 22, "ZipUtil");
    }

    @Test
    public void testPhysicalOrderOfSpecificFile() throws Exception {
        this.readOrderTest();
        String entryName = "src/main/java/org/apache/commons/compress/archivers/zip/ZipExtraField.java";
        Iterable entries = this.zf.getEntriesInPhysicalOrder("src/main/java/org/apache/commons/compress/archivers/zip/ZipExtraField.java");
        Iterator iter = entries.iterator();
        ZipArchiveEntry entry = (ZipArchiveEntry)iter.next();
        Assertions.assertEquals((Object)"src/main/java/org/apache/commons/compress/archivers/zip/ZipExtraField.java", (Object)entry.getName());
        Assertions.assertFalse((boolean)iter.hasNext());
    }

    @Test
    public void testReadingOfExtraDataBeforeZip() throws IOException {
        byte[] fileHeader = "Before Zip file".getBytes(StandardCharsets.UTF_8);
        String entryName = "COMPRESS-621.txt";
        byte[] entryContent = "https://issues.apache.org/jira/browse/COMPRESS-621".getBytes(StandardCharsets.UTF_8);
        try (ZipFile archive = new ZipFile(ZipFileTest.getFile("COMPRESS-621.zip"));){
            Assertions.assertEquals((long)fileHeader.length, (long)archive.getFirstLocalFileHeaderOffset());
            try (InputStream input = archive.getContentBeforeFirstLocalFileHeader();){
                Assertions.assertArrayEquals((byte[])fileHeader, (byte[])IOUtils.toByteArray((InputStream)input));
            }
            ZipArchiveEntry e = archive.getEntry("COMPRESS-621.txt");
            Assertions.assertEquals((long)entryContent.length, (long)e.getSize());
            try (InputStream input = archive.getInputStream(e);){
                Assertions.assertArrayEquals((byte[])entryContent, (byte[])IOUtils.toByteArray((InputStream)input));
            }
        }
    }

    @Test
    public void testReadingOfFirstStoredEntry() throws Exception {
        File archive = ZipFileTest.getFile("COMPRESS-264.zip");
        this.zf = new ZipFile(archive);
        ZipArchiveEntry ze = this.zf.getEntry("test.txt");
        Assertions.assertEquals((long)5L, (long)ze.getSize());
        try (InputStream inputStream = this.zf.getInputStream(ze);){
            Assertions.assertArrayEquals((byte[])new byte[]{100, 97, 116, 97, 10}, (byte[])IOUtils.toByteArray((InputStream)inputStream));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testReadingOfStoredEntry() throws Exception {
        File f = File.createTempFile("commons-compress-zipfiletest", ".zip");
        try {
            ZipArchiveEntry ze;
            f.deleteOnExit();
            try (OutputStream o = Files.newOutputStream(f.toPath(), new OpenOption[0]);
                 ZipArchiveOutputStream zo = new ZipArchiveOutputStream(o);){
                ze = new ZipArchiveEntry("foo");
                ze.setMethod(0);
                ze.setSize(4L);
                ze.setCrc(3057449933L);
                zo.putArchiveEntry((ArchiveEntry)ze);
                zo.write(new byte[]{1, 2, 3, 4});
                zo.closeArchiveEntry();
            }
            this.zf = new ZipFile(f);
            ze = this.zf.getEntry("foo");
            Assertions.assertNotNull((Object)ze);
            try (InputStream i = this.zf.getInputStream(ze);){
                byte[] b = new byte[4];
                Assertions.assertEquals((int)4, (int)i.read(b));
                Assertions.assertEquals((int)-1, (int)i.read());
            }
        }
        finally {
            f.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSelfExtractingZipUsingUnzipsfx() throws IOException, InterruptedException {
        File unzipsfx = new File("/usr/bin/unzipsfx");
        Assumptions.assumeTrue((boolean)unzipsfx.exists());
        File testZip = File.createTempFile("commons-compress-selfExtractZipTest", ".zip");
        testZip.deleteOnExit();
        String testEntryName = "test_self_extract_zip/foo";
        File extractedFile = new File(testZip.getParentFile(), "test_self_extract_zip/foo");
        extractedFile.deleteOnExit();
        byte[] testData = new byte[]{1, 2, 3, 4};
        byte[] buffer = new byte[512];
        try (InputStream unzipsfxInputStream = Files.newInputStream(unzipsfx.toPath(), new OpenOption[0]);){
            int bytesRead;
            try (OutputStream outputStream = Files.newOutputStream(testZip.toPath(), new OpenOption[0]);
                 ZipArchiveOutputStream zo = new ZipArchiveOutputStream(outputStream);){
                while ((bytesRead = unzipsfxInputStream.read(buffer)) > 0) {
                    zo.writePreamble(buffer, 0, bytesRead);
                }
                ZipArchiveEntry ze = new ZipArchiveEntry("test_self_extract_zip/foo");
                ze.setMethod(0);
                ze.setSize(4L);
                ze.setCrc(3057449933L);
                zo.putArchiveEntry((ArchiveEntry)ze);
                zo.write(testData);
                zo.closeArchiveEntry();
            }
            ProcessBuilder pbChmod = new ProcessBuilder("chmod", "+x", testZip.getPath());
            pbChmod.redirectErrorStream(true);
            Process processChmod = pbChmod.start();
            Assertions.assertEquals((int)0, (int)processChmod.waitFor(), (String)new String(IOUtils.toByteArray((InputStream)processChmod.getInputStream())));
            ProcessBuilder pb = new ProcessBuilder(testZip.getPath());
            pb.redirectOutput(ProcessBuilder.Redirect.PIPE);
            pb.directory(testZip.getParentFile());
            pb.redirectErrorStream(true);
            Process process = pb.start();
            int rc = process.waitFor();
            if (rc == 137 && SystemUtils.IS_OS_MAC) {
                Assume.assumeTrue((boolean)Boolean.getBoolean("skipReturnCode137"));
                return;
            }
            Assertions.assertEquals((int)0, (int)rc, (String)new String(IOUtils.toByteArray((InputStream)process.getInputStream())));
            if (!extractedFile.exists()) {
                Assertions.fail((String)"Can not find the extracted file");
            }
            try (InputStream inputStream = Files.newInputStream(extractedFile.toPath(), new OpenOption[0]);){
                bytesRead = IOUtils.readFully((InputStream)inputStream, (byte[])buffer);
                Assertions.assertEquals((int)testData.length, (int)bytesRead);
                Assertions.assertArrayEquals((byte[])testData, (byte[])Arrays.copyOfRange(buffer, 0, bytesRead));
            }
        }
        finally {
            testZip.delete();
            extractedFile.delete();
            extractedFile.getParentFile().delete();
        }
    }

    @Test
    public void testSetLevelTooBigForZipArchiveOutputStream() {
        ZipArchiveOutputStream fixture = new ZipArchiveOutputStream((OutputStream)new ByteArrayOutputStream());
        Assertions.assertThrows(IllegalArgumentException.class, () -> fixture.setLevel(10));
    }

    @Test
    public void testSetLevelTooSmallForZipArchiveOutputStream() {
        ZipArchiveOutputStream fixture = new ZipArchiveOutputStream((OutputStream)new ByteArrayOutputStream());
        Assertions.assertThrows(IllegalArgumentException.class, () -> fixture.setLevel(-2));
    }

    @Test
    public void testSkipsPK00Prefix() throws Exception {
        File archive = ZipFileTest.getFile("COMPRESS-208.zip");
        this.zf = new ZipFile(archive);
        Assertions.assertNotNull((Object)this.zf.getEntry("test1.xml"));
        Assertions.assertNotNull((Object)this.zf.getEntry("test2.xml"));
    }

    @Test
    public void testUnixSymlinkSampleFile() throws Exception {
        String entryPrefix = "COMPRESS-214_unix_symlinks/";
        TreeMap<String, String> expectedVals = new TreeMap<String, String>();
        expectedVals.put("COMPRESS-214_unix_symlinks/link1", "../COMPRESS-214_unix_symlinks/./a/b/c/../../../\uf999");
        expectedVals.put("COMPRESS-214_unix_symlinks/link2", "../COMPRESS-214_unix_symlinks/./a/b/c/../../../g");
        expectedVals.put("COMPRESS-214_unix_symlinks/link3", "../COMPRESS-214_unix_symlinks/././a/b/c/../../../\u76f4\u6a39");
        expectedVals.put("COMPRESS-214_unix_symlinks/link4", "\u82b1\u5b50/\u745b\u5b50");
        expectedVals.put("COMPRESS-214_unix_symlinks/\uf999", "./\u82b1\u5b50/\u745b\u5b50/\u5897\u8c37/\uf999");
        expectedVals.put("COMPRESS-214_unix_symlinks/g", "./a/b/c/d/e/f/g");
        expectedVals.put("COMPRESS-214_unix_symlinks/\u76f4\u6a39", "./g");
        expectedVals.put("COMPRESS-214_unix_symlinks/link5", "../COMPRESS-214_unix_symlinks/././a/b");
        expectedVals.put("COMPRESS-214_unix_symlinks/link6", "../COMPRESS-214_unix_symlinks/././a/b/");
        File archive = ZipFileTest.getFile("COMPRESS-214_unix_symlinks.zip");
        this.zf = new ZipFile(archive);
        Enumeration en = this.zf.getEntries();
        while (en.hasMoreElements()) {
            ZipArchiveEntry zae = (ZipArchiveEntry)en.nextElement();
            String link = this.zf.getUnixSymlink(zae);
            if (zae.isUnixSymlink()) {
                String name = zae.getName();
                String expected = (String)expectedVals.get(name);
                Assertions.assertEquals((Object)expected, (Object)link);
                continue;
            }
            Assertions.assertNull((Object)link);
        }
    }

    @Test
    public void testUnshrinking() throws Exception {
        InputStream inputStream;
        this.zf = new ZipFile(ZipFileTest.getFile("SHRUNK.ZIP"));
        ZipArchiveEntry test = this.zf.getEntry("TEST1.XML");
        try (InputStream original = ZipFileTest.newInputStream("test1.xml");){
            inputStream = this.zf.getInputStream(test);
            try {
                Assertions.assertArrayEquals((byte[])IOUtils.toByteArray((InputStream)original), (byte[])IOUtils.toByteArray((InputStream)inputStream));
            }
            finally {
                if (inputStream != null) {
                    inputStream.close();
                }
            }
        }
        test = this.zf.getEntry("TEST2.XML");
        original = ZipFileTest.newInputStream("test2.xml");
        try {
            inputStream = this.zf.getInputStream(test);
            try {
                Assertions.assertArrayEquals((byte[])IOUtils.toByteArray((InputStream)original), (byte[])IOUtils.toByteArray((InputStream)inputStream));
            }
            finally {
                if (inputStream != null) {
                    inputStream.close();
                }
            }
        }
        finally {
            if (original != null) {
                original.close();
            }
        }
    }

    @Test
    public void testUnzipBZip2CompressedEntry() throws Exception {
        File archive = ZipFileTest.getFile("bzip2-zip.zip");
        this.zf = new ZipFile(archive);
        ZipArchiveEntry ze = this.zf.getEntry("lots-of-as");
        Assertions.assertEquals((long)42L, (long)ze.getSize());
        byte[] expected = new byte[42];
        Arrays.fill(expected, (byte)97);
        try (InputStream inputStream = this.zf.getInputStream(ze);){
            Assertions.assertArrayEquals((byte[])expected, (byte[])IOUtils.toByteArray((InputStream)inputStream));
        }
    }

    @Test
    public void testWinzipBackSlashWorkaround() throws Exception {
        File archive = ZipFileTest.getFile("test-winzip.zip");
        this.zf = new ZipFile(archive);
        Assertions.assertNull((Object)this.zf.getEntry("\u00e4\\\u00fc.txt"));
        Assertions.assertNotNull((Object)this.zf.getEntry("\u00e4/\u00fc.txt"));
    }

    @Test
    public void throwsExceptionWhenWritingPreamble() throws IOException {
        ZipArchiveOutputStream outputStream = new ZipArchiveOutputStream((OutputStream)new ByteArrayOutputStream());
        outputStream.putArchiveEntry((ArchiveEntry)new ZipArchiveEntry());
        Assertions.assertThrows(IllegalStateException.class, () -> outputStream.writePreamble(ByteUtils.EMPTY_BYTE_ARRAY));
    }
}

