/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.codec.digest;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.Locale;
import java.util.Random;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.binary.StringUtils;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.RandomAccessFileMode;
import org.apache.commons.lang3.JavaVersion;
import org.apache.commons.lang3.SystemUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class DigestUtilsTest {
    private static final String EMPTY_STRING = "";
    private final byte[] testData = new byte[0x100000];
    private Path testFile;
    private Path testRandomAccessFile;
    private RandomAccessFile testRandomAccessFileWrapper;

    DigestUtilsTest() {
    }

    private void assumeJava8() {
        Assumptions.assumeTrue((boolean)SystemUtils.isJavaVersionAtLeast((JavaVersion)JavaVersion.JAVA_1_8));
    }

    private void assumeJava9() {
        Assumptions.assumeTrue((boolean)SystemUtils.isJavaVersionAtLeast((JavaVersion)JavaVersion.JAVA_9));
    }

    byte[] getTestData() {
        return this.testData;
    }

    Path getTestPath() {
        return this.testFile;
    }

    RandomAccessFile getTestRandomAccessFile() {
        return this.testRandomAccessFileWrapper;
    }

    @BeforeEach
    public void setUp() throws Exception {
        new Random().nextBytes(this.testData);
        this.testFile = Files.createTempFile(DigestUtilsTest.class.getName(), ".dat", new FileAttribute[0]);
        try (OutputStream fos = Files.newOutputStream(this.testFile, new OpenOption[0]);){
            fos.write(this.testData);
        }
        this.testRandomAccessFile = Files.createTempFile(DigestUtilsTest.class.getName(), ".dat", new FileAttribute[0]);
        fos = Files.newOutputStream(this.testRandomAccessFile, new OpenOption[0]);
        try {
            fos.write(this.testData);
        }
        finally {
            if (fos != null) {
                fos.close();
            }
        }
        this.testRandomAccessFileWrapper = RandomAccessFileMode.READ_WRITE.create(this.testRandomAccessFile);
    }

    @AfterEach
    public void tearDown() throws IOException {
        if (this.testRandomAccessFileWrapper != null) {
            this.testRandomAccessFileWrapper.close();
        }
        Files.deleteIfExists(this.testFile);
        Files.deleteIfExists(this.testRandomAccessFile);
    }

    @Test
    void testDigestFileAs() throws IOException {
        String expected = "d41d8cd98f00b204e9800998ecf8427e";
        String pathName = "src/test/resources/org/apache/commons/codec/empty.bin";
        String algo = "MD5";
        Path path = Paths.get("src/test/resources/org/apache/commons/codec/empty.bin", new String[0]);
        Assertions.assertEquals((Object)"d41d8cd98f00b204e9800998ecf8427e", (Object)new DigestUtils("MD5").digestAsHex(path.toFile()));
        try (InputStream inputStream = Files.newInputStream(path, new OpenOption[0]);){
            Assertions.assertEquals((Object)"d41d8cd98f00b204e9800998ecf8427e", (Object)new DigestUtils("MD5").digestAsHex(inputStream));
        }
        byte[] allBytes = Files.readAllBytes(path);
        Assertions.assertEquals((Object)"d41d8cd98f00b204e9800998ecf8427e", (Object)new DigestUtils("MD5").digestAsHex(allBytes));
        Assertions.assertEquals((Object)"d41d8cd98f00b204e9800998ecf8427e", (Object)new DigestUtils("MD5").digestAsHex(ByteBuffer.wrap(allBytes)));
    }

    @Test
    void testGetMessageDigest() {
        DigestUtils digestUtils = new DigestUtils("MD5");
        Assertions.assertNotNull((Object)digestUtils.getMessageDigest());
        Assertions.assertEquals((Object)"MD5", (Object)digestUtils.getMessageDigest().getAlgorithm());
    }

    @Test
    void testInternalNoSuchAlgorithmException() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> DigestUtils.getDigest((String)"Bogus Bogus"));
    }

    @Test
    void testIsAvailable() {
        Assertions.assertTrue((boolean)DigestUtils.isAvailable((String)"MD5"));
        Assertions.assertFalse((boolean)DigestUtils.isAvailable((String)"FOO"));
        Assertions.assertFalse((boolean)DigestUtils.isAvailable(null));
    }

    @Test
    void testMd2Hex() throws IOException {
        Assertions.assertEquals((Object)"8350e5a3e24c153df2275c9f80692773", (Object)DigestUtils.md2Hex((String)EMPTY_STRING));
        Assertions.assertEquals((Object)"32ec01ec4a6dac72c0ab96fb34c0b5d1", (Object)DigestUtils.md2Hex((String)"a"));
        Assertions.assertEquals((Object)"da853b0d3f88d99b30283a69e6ded6bb", (Object)DigestUtils.md2Hex((String)"abc"));
        Assertions.assertEquals((Object)"ab4f496bfb2a530b219ff33031fe06b0", (Object)DigestUtils.md2Hex((String)"message digest"));
        Assertions.assertEquals((Object)"4e8ddff3650292ab5a4108c3aa47940b", (Object)DigestUtils.md2Hex((String)"abcdefghijklmnopqrstuvwxyz"));
        Assertions.assertEquals((Object)"da33def2a42df13975352846c30338cd", (Object)DigestUtils.md2Hex((String)"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"));
        Assertions.assertEquals((Object)"d5976f79d83d3a0dc9806c3c66f3efd8", (Object)DigestUtils.md2Hex((String)"12345678901234567890123456789012345678901234567890123456789012345678901234567890"));
        Assertions.assertEquals((Object)DigestUtils.md2Hex((byte[])this.testData), (Object)DigestUtils.md2Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testMd2HexLength() {
        String hashMe = "this is some string that is longer than 32 characters";
        String hash = DigestUtils.md2Hex((byte[])StringUtils.getBytesUtf8((String)hashMe));
        Assertions.assertEquals((int)32, (int)hash.length());
        hashMe = "length < 32";
        hash = DigestUtils.md2Hex((byte[])StringUtils.getBytesUtf8((String)hashMe));
        Assertions.assertEquals((int)32, (int)hash.length());
    }

    @Test
    void testMd2Length() {
        String hashMe = "this is some string that is longer than 16 characters";
        byte[] hash = DigestUtils.md2((byte[])StringUtils.getBytesUtf8((String)hashMe));
        Assertions.assertEquals((int)16, (int)hash.length);
        hashMe = "length < 16";
        hash = DigestUtils.md2((byte[])StringUtils.getBytesUtf8((String)hashMe));
        Assertions.assertEquals((int)16, (int)hash.length);
    }

    @Test
    void testMd5Hex() throws IOException {
        Assertions.assertEquals((Object)"d41d8cd98f00b204e9800998ecf8427e", (Object)DigestUtils.md5Hex((String)EMPTY_STRING));
        Assertions.assertEquals((Object)"0cc175b9c0f1b6a831c399e269772661", (Object)DigestUtils.md5Hex((String)"a"));
        Assertions.assertEquals((Object)"900150983cd24fb0d6963f7d28e17f72", (Object)DigestUtils.md5Hex((String)"abc"));
        Assertions.assertEquals((Object)"f96b697d7cb7938d525a2f31aaf161d0", (Object)DigestUtils.md5Hex((String)"message digest"));
        Assertions.assertEquals((Object)"c3fcd3d76192e4007dfb496cca67e13b", (Object)DigestUtils.md5Hex((String)"abcdefghijklmnopqrstuvwxyz"));
        Assertions.assertEquals((Object)"d174ab98d277d9f5a5611c2c9f419d9f", (Object)DigestUtils.md5Hex((String)"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"));
        Assertions.assertEquals((Object)"57edf4a22be3c955ac49da2e2107b67a", (Object)DigestUtils.md5Hex((String)"12345678901234567890123456789012345678901234567890123456789012345678901234567890"));
        Assertions.assertEquals((Object)DigestUtils.md5Hex((byte[])this.testData), (Object)DigestUtils.md5Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testMd5HexLengthForBytes() {
        String hashMe = "this is some string that is longer than 32 characters";
        String hash = DigestUtils.md5Hex((byte[])StringUtils.getBytesUtf8((String)hashMe));
        Assertions.assertEquals((int)32, (int)hash.length());
        hashMe = "length < 32";
        hash = DigestUtils.md5Hex((byte[])StringUtils.getBytesUtf8((String)hashMe));
        Assertions.assertEquals((int)32, (int)hash.length());
    }

    @Test
    void testMd5LengthForBytes() {
        String hashMe = "this is some string that is longer than 16 characters";
        byte[] hash = DigestUtils.md5((byte[])StringUtils.getBytesUtf8((String)hashMe));
        Assertions.assertEquals((int)16, (int)hash.length);
        hashMe = "length < 16";
        hash = DigestUtils.md5((byte[])StringUtils.getBytesUtf8((String)hashMe));
        Assertions.assertEquals((int)16, (int)hash.length);
    }

    @Test
    void testSha1Hex() throws IOException {
        Assertions.assertEquals((Object)"a9993e364706816aba3e25717850c26c9cd0d89d", (Object)DigestUtils.sha1Hex((String)"abc"));
        Assertions.assertEquals((Object)"a9993e364706816aba3e25717850c26c9cd0d89d", (Object)DigestUtils.sha1Hex((byte[])StringUtils.getBytesUtf8((String)"abc")));
        Assertions.assertEquals((Object)"84983e441c3bd26ebaae4aa1f95129e5e54670f1", (Object)DigestUtils.sha1Hex((String)"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"));
        Assertions.assertEquals((Object)DigestUtils.sha1Hex((byte[])this.testData), (Object)DigestUtils.sha1Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha1UpdateWithByteArray() {
        String d1 = "C'est un homme qui rentre dans un caf\u00e9, et plouf";
        String d2 = "C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'";
        MessageDigest messageDigest = DigestUtils.getSha1Digest();
        messageDigest.update("C'est un homme qui rentre dans un caf\u00e9, et plouf".getBytes());
        messageDigest.update("C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'".getBytes());
        String expectedResult = Hex.encodeHexString((byte[])messageDigest.digest());
        messageDigest = DigestUtils.getSha1Digest();
        DigestUtils.updateDigest((MessageDigest)messageDigest, (byte[])"C'est un homme qui rentre dans un caf\u00e9, et plouf".getBytes());
        DigestUtils.updateDigest((MessageDigest)messageDigest, (byte[])"C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'".getBytes());
        String actualResult = Hex.encodeHexString((byte[])messageDigest.digest());
        Assertions.assertEquals((Object)expectedResult, (Object)actualResult);
    }

    @Test
    void testSha1UpdateWithByteBuffer() {
        String d1 = "C'est un homme qui rentre dans un caf\u00e9, et plouf";
        String d2 = "C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'";
        MessageDigest messageDigest = DigestUtils.getSha1Digest();
        messageDigest.update("C'est un homme qui rentre dans un caf\u00e9, et plouf".getBytes());
        messageDigest.update("C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'".getBytes());
        String expectedResult = Hex.encodeHexString((byte[])messageDigest.digest());
        messageDigest = DigestUtils.getSha1Digest();
        DigestUtils.updateDigest((MessageDigest)messageDigest, (ByteBuffer)ByteBuffer.wrap("C'est un homme qui rentre dans un caf\u00e9, et plouf".getBytes()));
        DigestUtils.updateDigest((MessageDigest)messageDigest, (ByteBuffer)ByteBuffer.wrap("C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'".getBytes()));
        String actualResult = Hex.encodeHexString((byte[])messageDigest.digest());
        Assertions.assertEquals((Object)expectedResult, (Object)actualResult);
    }

    @Test
    void testSha1UpdateWithString() {
        String d1 = "C'est un homme qui rentre dans un caf\u00e9, et plouf";
        String d2 = "C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'";
        MessageDigest messageDigest = DigestUtils.getSha1Digest();
        messageDigest.update(StringUtils.getBytesUtf8((String)"C'est un homme qui rentre dans un caf\u00e9, et plouf"));
        messageDigest.update(StringUtils.getBytesUtf8((String)"C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'"));
        String expectedResult = Hex.encodeHexString((byte[])messageDigest.digest());
        messageDigest = DigestUtils.getSha1Digest();
        DigestUtils.updateDigest((MessageDigest)messageDigest, (String)"C'est un homme qui rentre dans un caf\u00e9, et plouf");
        DigestUtils.updateDigest((MessageDigest)messageDigest, (String)"C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'");
        String actualResult = Hex.encodeHexString((byte[])messageDigest.digest());
        Assertions.assertEquals((Object)expectedResult, (Object)actualResult);
    }

    @Test
    void testSha224_FileAsHex() throws IOException {
        this.assumeJava8();
        String expected = "d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f";
        String pathName = "src/test/resources/org/apache/commons/codec/empty.bin";
        String algo = "SHA-224";
        DigestUtils digestUtils = new DigestUtils("SHA-224");
        Path path = Paths.get("src/test/resources/org/apache/commons/codec/empty.bin", new String[0]);
        Assertions.assertEquals((Object)"d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f", (Object)digestUtils.digestAsHex(path.toFile()));
        try (InputStream inputStream = Files.newInputStream(path, new OpenOption[0]);){
            Assertions.assertEquals((Object)"d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f", (Object)digestUtils.digestAsHex(inputStream));
        }
        byte[] allBytes = Files.readAllBytes(path);
        Assertions.assertEquals((Object)"d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f", (Object)digestUtils.digestAsHex(allBytes));
        Assertions.assertEquals((Object)"d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f", (Object)digestUtils.digestAsHex(ByteBuffer.wrap(allBytes)));
    }

    @Test
    void testSha224_PathAsHex() throws IOException {
        this.assumeJava8();
        Assertions.assertEquals((Object)"d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f", (Object)new DigestUtils("SHA-224").digestAsHex(Paths.get("src/test/resources/org/apache/commons/codec/empty.bin", new String[0]), new OpenOption[0]));
    }

    @Test
    void testSha224_StringAsHex() {
        this.assumeJava8();
        Assertions.assertEquals((Object)"d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f", (Object)new DigestUtils("SHA-224").digestAsHex(EMPTY_STRING));
        Assertions.assertEquals((Object)"730e109bd7a8a32b1cb9d9a09aa2325d2430587ddbc0c38bad911525", (Object)new DigestUtils("SHA-224").digestAsHex("The quick brown fox jumps over the lazy dog"));
    }

    @Test
    void testSha256() throws IOException {
        Assertions.assertEquals((Object)"ba7816bf8f01cfea414140de5dae2223b00361a396177a9cb410ff61f20015ad", (Object)DigestUtils.sha256Hex((String)"abc"));
        Assertions.assertEquals((Object)"ba7816bf8f01cfea414140de5dae2223b00361a396177a9cb410ff61f20015ad", (Object)DigestUtils.sha256Hex((byte[])StringUtils.getBytesUtf8((String)"abc")));
        Assertions.assertEquals((Object)"248d6a61d20638b8e5c026930c3e6039a33ce45964ff2167f6ecedd419db06c1", (Object)DigestUtils.sha256Hex((String)"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"));
        Assertions.assertEquals((Object)DigestUtils.sha256Hex((byte[])this.testData), (Object)DigestUtils.sha256Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha256HexInputStream() throws IOException {
        Assertions.assertEquals((Object)DigestUtils.sha256Hex((byte[])this.testData), (Object)DigestUtils.sha256Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha3_224() {
        this.assumeJava9();
        Assertions.assertEquals((Object)"6b4e03423667dbb73b6e15454f0eb1abd4597f9a1b078e3f5b5a6bc7", (Object)DigestUtils.sha3_224Hex((String)EMPTY_STRING));
    }

    @Test
    void testSha3_224HexInputStream() throws IOException {
        this.assumeJava9();
        Assertions.assertEquals((Object)DigestUtils.sha3_224Hex((byte[])this.testData), (Object)DigestUtils.sha3_224Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha3_256() {
        this.assumeJava9();
        Assertions.assertEquals((Object)"a7ffc6f8bf1ed76651c14756a061d662f580ff4de43b49fa82d80a4b80f8434a", (Object)DigestUtils.sha3_256Hex((String)EMPTY_STRING));
    }

    @Test
    void testSha3_256HexInputStream() throws IOException {
        this.assumeJava9();
        Assertions.assertEquals((Object)DigestUtils.sha3_256Hex((byte[])this.testData), (Object)DigestUtils.sha3_256Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha3_384() {
        this.assumeJava9();
        Assertions.assertEquals((Object)"0c63a75b845e4f7d01107d852e4c2485c51a50aaaa94fc61995e71bbee983a2ac3713831264adb47fb6bd1e058d5f004", (Object)DigestUtils.sha3_384Hex((String)EMPTY_STRING));
    }

    @Test
    void testSha3_384HexInputStream() throws IOException {
        this.assumeJava9();
        Assertions.assertEquals((Object)DigestUtils.sha3_384Hex((byte[])this.testData), (Object)DigestUtils.sha3_384Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha3_512() {
        this.assumeJava9();
        Assertions.assertEquals((Object)"a69f73cca23a9ac5c8b567dc185a756e97c982164fe25859e0d1dcc1475c80a615b2123af1f5f94c11e3e9402c3ac558f500199d95b6d3e301758586281dcd26", (Object)DigestUtils.sha3_512Hex((String)EMPTY_STRING));
    }

    @Test
    void testSha3_512HexInputStream() throws IOException {
        this.assumeJava9();
        Assertions.assertEquals((Object)DigestUtils.sha3_512Hex((byte[])this.testData), (Object)DigestUtils.sha3_512Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha384() throws IOException {
        Assertions.assertEquals((Object)"cb00753f45a35e8bb5a03d699ac65007272c32ab0eded1631a8b605a43ff5bed8086072ba1e7cc2358baeca134c825a7", (Object)DigestUtils.sha384Hex((String)"abc"));
        Assertions.assertEquals((Object)"cb00753f45a35e8bb5a03d699ac65007272c32ab0eded1631a8b605a43ff5bed8086072ba1e7cc2358baeca134c825a7", (Object)DigestUtils.sha384Hex((byte[])StringUtils.getBytesUtf8((String)"abc")));
        Assertions.assertEquals((Object)"09330c33f71147e83d192fc782cd1b4753111b173b3b05d22fa08086e3b0f712fcc7c71a557e2db966c3e9fa91746039", (Object)DigestUtils.sha384Hex((String)"abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"));
        Assertions.assertEquals((Object)DigestUtils.sha384Hex((byte[])this.testData), (Object)DigestUtils.sha384Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha384HexInputStream() throws IOException {
        Assertions.assertEquals((Object)DigestUtils.sha384Hex((byte[])this.testData), (Object)DigestUtils.sha384Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testSha512() {
        Assertions.assertEquals((Object)"ddaf35a193617abacc417349ae20413112e6fa4e89a97ea20a9eeee64b55d39a2192992a274fc1a836ba3c23a3feebbd454d4423643ce80e2a9ac94fa54ca49f", (Object)DigestUtils.sha512Hex((String)"abc"));
        Assertions.assertEquals((Object)"ddaf35a193617abacc417349ae20413112e6fa4e89a97ea20a9eeee64b55d39a2192992a274fc1a836ba3c23a3feebbd454d4423643ce80e2a9ac94fa54ca49f", (Object)DigestUtils.sha512Hex((byte[])StringUtils.getBytesUtf8((String)"abc")));
        Assertions.assertEquals((Object)"8e959b75dae313da8cf4f72814fc143f8f7779c6eb9f7fa17299aeadb6889018501d289e4900f7e4331b99dec4b5433ac7d329eeb6dd26545e96e55b874be909", (Object)DigestUtils.sha512Hex((String)"abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"));
    }

    @Test
    void testSha512_224() throws Exception {
        this.assumeJava9();
        String stringInput = "abc";
        byte[] bytesInput = StringUtils.getBytesUtf8((String)"abc");
        String resultString = "4634270F707B6A54DAAE7530460842E20E37ED265CEEE9A43E8924AA".toLowerCase(Locale.ROOT);
        byte[] resultBytes = Hex.decodeHex((String)resultString);
        Assertions.assertArrayEquals((byte[])resultBytes, (byte[])DigestUtils.sha512_224((byte[])bytesInput));
        Assertions.assertArrayEquals((byte[])resultBytes, (byte[])DigestUtils.sha512_224((InputStream)new ByteArrayInputStream(bytesInput)));
        Assertions.assertArrayEquals((byte[])resultBytes, (byte[])DigestUtils.sha512_224((String)"abc"));
        Assertions.assertEquals((Object)resultString, (Object)DigestUtils.sha512_224Hex((byte[])bytesInput));
        Assertions.assertEquals((Object)resultString, (Object)DigestUtils.sha512_224Hex((InputStream)new ByteArrayInputStream(bytesInput)));
        Assertions.assertEquals((Object)resultString, (Object)DigestUtils.sha512_224Hex((String)"abc"));
        Assertions.assertEquals((Object)"23FEC5BB94D60B23308192640B0C453335D664734FE40E7268674AF9".toLowerCase(Locale.ROOT), (Object)DigestUtils.sha512_224Hex((String)"abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"));
    }

    @Test
    void testSha512_256() throws Exception {
        this.assumeJava9();
        String stringInput = "abc";
        byte[] bytesInput = StringUtils.getBytesUtf8((String)"abc");
        String resultString = "53048E2681941EF99B2E29B76B4C7DABE4C2D0C634FC6D46E0E2F13107E7AF23".toLowerCase(Locale.ROOT);
        byte[] resultBytes = Hex.decodeHex((String)resultString);
        Assertions.assertArrayEquals((byte[])resultBytes, (byte[])DigestUtils.sha512_256((byte[])bytesInput));
        Assertions.assertArrayEquals((byte[])resultBytes, (byte[])DigestUtils.sha512_256((InputStream)new ByteArrayInputStream(bytesInput)));
        Assertions.assertArrayEquals((byte[])resultBytes, (byte[])DigestUtils.sha512_256((String)"abc"));
        Assertions.assertEquals((Object)resultString, (Object)DigestUtils.sha512_256Hex((byte[])bytesInput));
        Assertions.assertEquals((Object)resultString, (Object)DigestUtils.sha512_256Hex((InputStream)new ByteArrayInputStream(bytesInput)));
        Assertions.assertEquals((Object)resultString, (Object)DigestUtils.sha512_256Hex((String)"abc"));
        Assertions.assertEquals((Object)"3928E184FB8690F840DA3988121D31BE65CB9D3EF83EE6146FEAC861E19B563A".toLowerCase(Locale.ROOT), (Object)DigestUtils.sha512_256Hex((String)"abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"));
    }

    @Test
    void testSha512HexInputStream() throws IOException {
        Assertions.assertEquals((Object)DigestUtils.sha512Hex((byte[])this.testData), (Object)DigestUtils.sha512Hex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testShaHex() throws IOException {
        Assertions.assertEquals((Object)"a9993e364706816aba3e25717850c26c9cd0d89d", (Object)DigestUtils.shaHex((String)"abc"));
        Assertions.assertEquals((Object)"a9993e364706816aba3e25717850c26c9cd0d89d", (Object)DigestUtils.shaHex((byte[])StringUtils.getBytesUtf8((String)"abc")));
        Assertions.assertEquals((Object)"84983e441c3bd26ebaae4aa1f95129e5e54670f1", (Object)DigestUtils.shaHex((String)"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"));
        Assertions.assertEquals((Object)DigestUtils.shaHex((byte[])this.testData), (Object)DigestUtils.shaHex((InputStream)new ByteArrayInputStream(this.testData)));
    }

    @Test
    void testShaUpdateWithByteArray() {
        String d1 = "C'est un homme qui rentre dans un caf\u00e9, et plouf";
        String d2 = "C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'";
        MessageDigest messageDigest = DigestUtils.getShaDigest();
        messageDigest.update("C'est un homme qui rentre dans un caf\u00e9, et plouf".getBytes());
        messageDigest.update("C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'".getBytes());
        String expectedResult = Hex.encodeHexString((byte[])messageDigest.digest());
        messageDigest = DigestUtils.getShaDigest();
        DigestUtils.updateDigest((MessageDigest)messageDigest, (byte[])"C'est un homme qui rentre dans un caf\u00e9, et plouf".getBytes());
        DigestUtils.updateDigest((MessageDigest)messageDigest, (byte[])"C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'".getBytes());
        String actualResult = Hex.encodeHexString((byte[])messageDigest.digest());
        Assertions.assertEquals((Object)expectedResult, (Object)actualResult);
    }

    @Test
    void testShaUpdateWithString() {
        String d1 = "C'est un homme qui rentre dans un caf\u00e9, et plouf";
        String d2 = "C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'";
        MessageDigest messageDigest = DigestUtils.getShaDigest();
        messageDigest.update(StringUtils.getBytesUtf8((String)"C'est un homme qui rentre dans un caf\u00e9, et plouf"));
        messageDigest.update(StringUtils.getBytesUtf8((String)"C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'"));
        String expectedResult = Hex.encodeHexString((byte[])messageDigest.digest());
        messageDigest = DigestUtils.getShaDigest();
        DigestUtils.updateDigest((MessageDigest)messageDigest, (String)"C'est un homme qui rentre dans un caf\u00e9, et plouf");
        DigestUtils.updateDigest((MessageDigest)messageDigest, (String)"C'est un homme, c'est qu'une t\u00eate, on lui offre un cadeau: 'oh... encore un chapeau!'");
        String actualResult = Hex.encodeHexString((byte[])messageDigest.digest());
        Assertions.assertEquals((Object)expectedResult, (Object)actualResult);
    }
}

