/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service.reads.range;

import com.google.common.annotations.VisibleForTesting;
import java.util.ArrayList;
import java.util.Collections;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.PartitionRangeReadCommand;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.filter.DataLimits;
import org.apache.cassandra.db.partitions.PartitionIterator;
import org.apache.cassandra.db.rows.RowIterator;
import org.apache.cassandra.exceptions.ReadAbortException;
import org.apache.cassandra.exceptions.ReadFailureException;
import org.apache.cassandra.exceptions.ReadTimeoutException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.locator.EndpointsForRange;
import org.apache.cassandra.locator.Replica;
import org.apache.cassandra.locator.ReplicaPlan;
import org.apache.cassandra.metrics.ClientRangeRequestMetrics;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.reads.DataResolver;
import org.apache.cassandra.service.reads.ReadCallback;
import org.apache.cassandra.service.reads.range.SingleRangeResponse;
import org.apache.cassandra.service.reads.repair.ReadRepair;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.utils.AbstractIterator;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.CloseableIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@VisibleForTesting
public class RangeCommandIterator
extends AbstractIterator<RowIterator>
implements PartitionIterator {
    private static final Logger logger = LoggerFactory.getLogger(RangeCommandIterator.class);
    public static final ClientRangeRequestMetrics rangeMetrics = new ClientRangeRequestMetrics("RangeSlice");
    final CloseableIterator<ReplicaPlan.ForRangeRead> replicaPlans;
    final int totalRangeCount;
    final PartitionRangeReadCommand command;
    final boolean enforceStrictLiveness;
    final long queryStartNanoTime;
    int rangesQueried;
    int batchesRequested = 0;
    private final long startTime;
    private DataLimits.Counter counter;
    private PartitionIterator sentQueryIterator;
    private final int maxConcurrencyFactor;
    private int concurrencyFactor;
    private int liveReturned;

    RangeCommandIterator(CloseableIterator<ReplicaPlan.ForRangeRead> replicaPlans, PartitionRangeReadCommand command, int concurrencyFactor, int maxConcurrencyFactor, int totalRangeCount, long queryStartNanoTime) {
        this.replicaPlans = replicaPlans;
        this.command = command;
        this.concurrencyFactor = concurrencyFactor;
        this.maxConcurrencyFactor = maxConcurrencyFactor;
        this.totalRangeCount = totalRangeCount;
        this.queryStartNanoTime = queryStartNanoTime;
        this.startTime = Clock.Global.nanoTime();
        this.enforceStrictLiveness = command.metadata().enforceStrictLiveness();
    }

    @Override
    protected RowIterator computeNext() {
        try {
            while (this.sentQueryIterator == null || !this.sentQueryIterator.hasNext()) {
                if (!this.replicaPlans.hasNext()) {
                    return (RowIterator)this.endOfData();
                }
                if (this.sentQueryIterator != null) {
                    this.liveReturned += this.counter.counted();
                    this.sentQueryIterator.close();
                    this.updateConcurrencyFactor();
                }
                this.sentQueryIterator = this.sendNextRequests();
            }
            return (RowIterator)this.sentQueryIterator.next();
        }
        catch (UnavailableException e) {
            RangeCommandIterator.rangeMetrics.unavailables.mark();
            StorageProxy.logRequestException(e, Collections.singleton(this.command));
            throw e;
        }
        catch (ReadTimeoutException e) {
            RangeCommandIterator.rangeMetrics.timeouts.mark();
            StorageProxy.logRequestException(e, Collections.singleton(this.command));
            throw e;
        }
        catch (ReadAbortException e) {
            rangeMetrics.markAbort(e);
            throw e;
        }
        catch (ReadFailureException e) {
            RangeCommandIterator.rangeMetrics.failures.mark();
            throw e;
        }
    }

    private void updateConcurrencyFactor() {
        this.liveReturned += this.counter.counted();
        this.concurrencyFactor = RangeCommandIterator.computeConcurrencyFactor(this.totalRangeCount, this.rangesQueried, this.maxConcurrencyFactor, this.command.limits().count(), this.liveReturned);
    }

    @VisibleForTesting
    static int computeConcurrencyFactor(int totalRangeCount, int rangesQueried, int maxConcurrencyFactor, int limit, int liveReturned) {
        maxConcurrencyFactor = Math.max(1, Math.min(maxConcurrencyFactor, totalRangeCount - rangesQueried));
        if (liveReturned == 0) {
            Tracing.trace("Didn't get any response rows; new concurrent requests: {}", (Object)maxConcurrencyFactor);
            return maxConcurrencyFactor;
        }
        int remainingRows = limit - liveReturned;
        float rowsPerRange = (float)liveReturned / (float)rangesQueried;
        int concurrencyFactor = Math.max(1, Math.min(maxConcurrencyFactor, Math.round((float)remainingRows / rowsPerRange)));
        logger.trace("Didn't get enough response rows; actual rows per range: {}; remaining rows: {}, new concurrent requests: {}", new Object[]{Float.valueOf(rowsPerRange), remainingRows, concurrencyFactor});
        return concurrencyFactor;
    }

    private SingleRangeResponse query(ReplicaPlan.ForRangeRead replicaPlan, boolean isFirst) {
        PartitionRangeReadCommand rangeCommand = this.command.forSubRange(replicaPlan.range(), isFirst);
        boolean trackRepairedStatus = DatabaseDescriptor.getRepairedDataTrackingForRangeReadsEnabled() && ((EndpointsForRange)((EndpointsForRange)replicaPlan.contacts()).filter(Replica::isFull)).size() > 1;
        ReplicaPlan.SharedForRangeRead sharedReplicaPlan = ReplicaPlan.shared(replicaPlan);
        ReadRepair<EndpointsForRange, ReplicaPlan.ForRangeRead> readRepair = ReadRepair.create(this.command, sharedReplicaPlan, this.queryStartNanoTime);
        DataResolver<EndpointsForRange, ReplicaPlan.ForRangeRead> resolver = new DataResolver<EndpointsForRange, ReplicaPlan.ForRangeRead>(rangeCommand, sharedReplicaPlan, readRepair, this.queryStartNanoTime, trackRepairedStatus);
        ReadCallback<EndpointsForRange, ReplicaPlan.ForRangeRead> handler = new ReadCallback<EndpointsForRange, ReplicaPlan.ForRangeRead>(resolver, rangeCommand, sharedReplicaPlan, this.queryStartNanoTime);
        if (((EndpointsForRange)replicaPlan.contacts()).size() == 1 && ((EndpointsForRange)replicaPlan.contacts()).get(0).isSelf()) {
            Stage.READ.execute(new StorageProxy.LocalReadRunnable(rangeCommand, handler, trackRepairedStatus));
        } else {
            for (Replica replica : (EndpointsForRange)replicaPlan.contacts()) {
                Tracing.trace("Enqueuing request to {}", (Object)replica);
                PartitionRangeReadCommand command = replica.isFull() ? rangeCommand : rangeCommand.copyAsTransientQuery(replica);
                Message<ReadCommand> message = command.createMessage(trackRepairedStatus && replica.isFull());
                MessagingService.instance().sendWithCallback(message, replica.endpoint(), handler);
            }
        }
        return new SingleRangeResponse(resolver, handler, readRepair);
    }

    PartitionIterator sendNextRequests() {
        ArrayList<PartitionIterator> concurrentQueries = new ArrayList<PartitionIterator>(this.concurrencyFactor);
        ArrayList readRepairs = new ArrayList(this.concurrencyFactor);
        try {
            ReplicaPlan.ForRangeRead replicaPlan;
            for (int i = 0; i < this.concurrencyFactor && this.replicaPlans.hasNext(); i += replicaPlan.vnodeCount()) {
                replicaPlan = (ReplicaPlan.ForRangeRead)this.replicaPlans.next();
                SingleRangeResponse response = this.query(replicaPlan, i == 0);
                concurrentQueries.add(response);
                readRepairs.add(response.getReadRepair());
                this.rangesQueried += replicaPlan.vnodeCount();
            }
            ++this.batchesRequested;
        }
        catch (Throwable t) {
            for (PartitionIterator response : concurrentQueries) {
                response.close();
            }
            throw t;
        }
        Tracing.trace("Submitted {} concurrent range requests", (Object)concurrentQueries.size());
        this.counter = DataLimits.NONE.newCounter(this.command.nowInSec(), true, this.command.selectsFullPartition(), this.enforceStrictLiveness);
        return this.counter.applyTo(StorageProxy.concatAndBlockOnRepair(concurrentQueries, readRepairs));
    }

    @Override
    public void close() {
        try {
            if (this.sentQueryIterator != null) {
                this.sentQueryIterator.close();
            }
            this.replicaPlans.close();
        }
        finally {
            long latency = Clock.Global.nanoTime() - this.startTime;
            rangeMetrics.addNano(latency);
            RangeCommandIterator.rangeMetrics.roundTrips.update(this.batchesRequested);
            Keyspace.openAndGetStore((TableMetadata)this.command.metadata()).metric.coordinatorScanLatency.update(latency, TimeUnit.NANOSECONDS);
        }
    }

    @VisibleForTesting
    int rangesQueried() {
        return this.rangesQueried;
    }

    @VisibleForTesting
    int batchesRequested() {
        return this.batchesRequested;
    }

    @VisibleForTesting
    int maxConcurrencyFactor() {
        return this.maxConcurrencyFactor;
    }

    @VisibleForTesting
    int concurrencyFactor() {
        return this.concurrencyFactor;
    }
}

