/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.repair.consistent;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.cassandra.concurrent.ImmediateExecutor;
import org.apache.cassandra.exceptions.RepairException;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.repair.CoordinatedRepairResult;
import org.apache.cassandra.repair.SharedContext;
import org.apache.cassandra.repair.SomeRepairFailedException;
import org.apache.cassandra.repair.consistent.ConsistentSession;
import org.apache.cassandra.repair.messages.FailSession;
import org.apache.cassandra.repair.messages.FinalizeCommit;
import org.apache.cassandra.repair.messages.FinalizePropose;
import org.apache.cassandra.repair.messages.PrepareConsistentRequest;
import org.apache.cassandra.repair.messages.RepairMessage;
import org.apache.cassandra.utils.concurrent.AsyncPromise;
import org.apache.cassandra.utils.concurrent.Future;
import org.apache.cassandra.utils.concurrent.ImmediateFuture;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CoordinatorSession
extends ConsistentSession {
    private static final Logger logger = LoggerFactory.getLogger(CoordinatorSession.class);
    private final SharedContext ctx;
    private final Map<InetAddressAndPort, ConsistentSession.State> participantStates = new HashMap<InetAddressAndPort, ConsistentSession.State>();
    private final AsyncPromise<Void> prepareFuture = AsyncPromise.uncancellable();
    private final AsyncPromise<Void> finalizeProposeFuture = AsyncPromise.uncancellable();
    private volatile long sessionStart = Long.MIN_VALUE;
    private volatile long repairStart = Long.MIN_VALUE;
    private volatile long finalizeStart = Long.MIN_VALUE;

    public CoordinatorSession(Builder builder) {
        super(builder);
        this.ctx = builder.ctx == null ? SharedContext.Global.instance : builder.ctx;
        for (InetAddressAndPort participant : this.participants) {
            this.participantStates.put(participant, ConsistentSession.State.PREPARING);
        }
    }

    public static Builder builder(SharedContext ctx) {
        return new Builder(ctx);
    }

    @Override
    public void setState(ConsistentSession.State state) {
        logger.trace("Setting coordinator state to {} for repair {}", (Object)state, (Object)this.sessionID);
        super.setState(state);
    }

    @VisibleForTesting
    synchronized ConsistentSession.State getParticipantState(InetAddressAndPort participant) {
        return this.participantStates.get(participant);
    }

    public synchronized void setParticipantState(InetAddressAndPort participant, ConsistentSession.State state) {
        logger.trace("Setting participant {} to state {} for repair {}", new Object[]{participant, state, this.sessionID});
        Preconditions.checkArgument((boolean)this.participantStates.containsKey(participant), (String)"Session %s doesn't include %s", (Object)this.sessionID, (Object)participant);
        Preconditions.checkArgument((boolean)this.participantStates.get(participant).canTransitionTo(state), (String)"Invalid state transition %s -> %s", (Object)((Object)this.participantStates.get(participant)), (Object)((Object)state));
        this.participantStates.put(participant, state);
        if (Iterables.all(this.participantStates.values(), s -> s == state)) {
            this.setState(state);
        }
    }

    synchronized void setAll(ConsistentSession.State state) {
        for (InetAddressAndPort participant : this.participants) {
            this.setParticipantState(participant, state);
        }
    }

    synchronized boolean allStates(ConsistentSession.State state) {
        return this.getState() == state && Iterables.all(this.participantStates.values(), v -> v == state);
    }

    synchronized boolean hasFailed() {
        return this.getState() == ConsistentSession.State.FAILED || Iterables.any(this.participantStates.values(), v -> v == ConsistentSession.State.FAILED);
    }

    protected void sendMessage(InetAddressAndPort destination, Message<RepairMessage> message) {
        logger.trace("Sending {} to {}", message.payload, (Object)destination);
        this.ctx.messaging().send(message, destination);
    }

    public Future<Void> prepare() {
        Preconditions.checkArgument((boolean)this.allStates(ConsistentSession.State.PREPARING));
        logger.info("Beginning prepare phase of incremental repair session {}", (Object)this.sessionID);
        Message<RepairMessage> message = Message.out(Verb.PREPARE_CONSISTENT_REQ, new PrepareConsistentRequest(this.sessionID, this.coordinator, (Set<InetAddressAndPort>)this.participants));
        for (InetAddressAndPort participant : this.participants) {
            this.sendMessage(participant, message);
        }
        return this.prepareFuture;
    }

    public synchronized void handlePrepareResponse(InetAddressAndPort participant, boolean success) {
        if (this.getState() == ConsistentSession.State.FAILED) {
            logger.trace("Incremental repair {} has failed, ignoring prepare response from {}", (Object)this.sessionID, (Object)participant);
            return;
        }
        if (!success) {
            logger.warn("{} failed the prepare phase for incremental repair session {}", (Object)participant, (Object)this.sessionID);
            this.sendFailureMessageToParticipants();
            this.setParticipantState(participant, ConsistentSession.State.FAILED);
        } else {
            logger.trace("Successful prepare response received from {} for repair session {}", (Object)participant, (Object)this.sessionID);
            this.setParticipantState(participant, ConsistentSession.State.PREPARED);
        }
        if (Iterables.any(this.participantStates.values(), v -> v == ConsistentSession.State.PREPARING)) {
            return;
        }
        if (this.getState() == ConsistentSession.State.PREPARED) {
            logger.info("Incremental repair session {} successfully prepared.", (Object)this.sessionID);
            this.prepareFuture.trySuccess(null);
        } else {
            this.fail();
        }
    }

    public synchronized void setRepairing() {
        this.setAll(ConsistentSession.State.REPAIRING);
    }

    public synchronized Future<Void> finalizePropose() {
        Preconditions.checkArgument((boolean)this.allStates(ConsistentSession.State.REPAIRING));
        logger.info("Proposing finalization of repair session {}", (Object)this.sessionID);
        Message<RepairMessage> message = Message.out(Verb.FINALIZE_PROPOSE_MSG, new FinalizePropose(this.sessionID));
        for (InetAddressAndPort participant : this.participants) {
            this.sendMessage(participant, message);
        }
        return this.finalizeProposeFuture;
    }

    public synchronized void handleFinalizePromise(InetAddressAndPort participant, boolean success) {
        if (this.getState() == ConsistentSession.State.FAILED) {
            logger.trace("Incremental repair {} has failed, ignoring finalize promise from {}", (Object)this.sessionID, (Object)participant);
        } else if (!success) {
            logger.warn("Finalization proposal of session {} rejected by {}. Aborting session", (Object)this.sessionID, (Object)participant);
            this.fail();
        } else {
            logger.trace("Successful finalize promise received from {} for repair session {}", (Object)participant, (Object)this.sessionID);
            this.setParticipantState(participant, ConsistentSession.State.FINALIZE_PROMISED);
            if (this.getState() == ConsistentSession.State.FINALIZE_PROMISED) {
                logger.info("Finalization proposal for repair session {} accepted by all participants.", (Object)this.sessionID);
                this.finalizeProposeFuture.trySuccess(null);
            }
        }
    }

    public synchronized void finalizeCommit() {
        Preconditions.checkArgument((boolean)this.allStates(ConsistentSession.State.FINALIZE_PROMISED));
        logger.info("Committing finalization of repair session {}", (Object)this.sessionID);
        Message<RepairMessage> message = Message.out(Verb.FINALIZE_COMMIT_MSG, new FinalizeCommit(this.sessionID));
        for (InetAddressAndPort participant : this.participants) {
            this.sendMessage(participant, message);
        }
        this.setAll(ConsistentSession.State.FINALIZED);
        logger.info("Incremental repair session {} completed", (Object)this.sessionID);
    }

    private void sendFailureMessageToParticipants() {
        Message<RepairMessage> message = Message.out(Verb.FAILED_SESSION_MSG, new FailSession(this.sessionID));
        for (InetAddressAndPort participant : this.participants) {
            if (this.participantStates.get(participant) == ConsistentSession.State.FAILED) continue;
            this.sendMessage(participant, message);
        }
    }

    public synchronized void fail() {
        Set cantFail = this.participantStates.entrySet().stream().filter(entry -> !((ConsistentSession.State)((Object)((Object)entry.getValue()))).canTransitionTo(ConsistentSession.State.FAILED)).collect(Collectors.toSet());
        if (!cantFail.isEmpty()) {
            logger.error("Can't transition endpoints {} to FAILED", cantFail, (Object)new RuntimeException());
            return;
        }
        logger.info("Incremental repair session {} failed", (Object)this.sessionID);
        this.sendFailureMessageToParticipants();
        this.setAll(ConsistentSession.State.FAILED);
        String exceptionMsg = String.format("Incremental repair session %s has failed", this.sessionID);
        this.finalizeProposeFuture.tryFailure(RepairException.warn(exceptionMsg));
        this.prepareFuture.tryFailure(RepairException.warn(exceptionMsg));
    }

    private static String formatDuration(long then, long now) {
        if (then == Long.MIN_VALUE || now == Long.MIN_VALUE) {
            return "n/a";
        }
        return DurationFormatUtils.formatDurationWords((long)(now - then), (boolean)true, (boolean)true);
    }

    public Future<CoordinatedRepairResult> execute(Supplier<Future<CoordinatedRepairResult>> sessionSubmitter) {
        logger.info("Beginning coordination of incremental repair session {}", (Object)this.sessionID);
        this.sessionStart = this.ctx.clock().currentTimeMillis();
        Future<Void> prepareResult = this.prepare();
        Future repairSessionResults = prepareResult.flatMap(ignore -> {
            this.repairStart = this.ctx.clock().currentTimeMillis();
            if (logger.isDebugEnabled()) {
                logger.debug("Incremental repair {} prepare phase completed in {}", (Object)this.sessionID, (Object)CoordinatorSession.formatDuration(this.sessionStart, this.repairStart));
            }
            this.setRepairing();
            return (Future)sessionSubmitter.get();
        });
        Future onlySuccessSessionResults = repairSessionResults.flatMap(result -> {
            this.finalizeStart = this.ctx.clock().currentTimeMillis();
            if (result.hasFailed()) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Incremental repair {} validation/stream phase completed in {}", (Object)this.sessionID, (Object)CoordinatorSession.formatDuration(this.repairStart, this.finalizeStart));
                }
                return ImmediateFuture.failure(SomeRepairFailedException.INSTANCE);
            }
            return ImmediateFuture.success(result);
        });
        Future proposeFuture = onlySuccessSessionResults.flatMap(results -> this.finalizePropose().map(ignore -> {
            if (logger.isDebugEnabled()) {
                logger.debug("Incremental repair {} finalization phase completed in {}", (Object)this.sessionID, (Object)CoordinatorSession.formatDuration(this.finalizeStart, this.ctx.clock().currentTimeMillis()));
            }
            this.finalizeCommit();
            if (logger.isDebugEnabled()) {
                logger.debug("Incremental repair {} phase completed in {}", (Object)this.sessionID, (Object)CoordinatorSession.formatDuration(this.sessionStart, this.ctx.clock().currentTimeMillis()));
            }
            return results;
        }));
        return proposeFuture.addCallback((ignore, failure) -> {
            if (failure != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Incremental repair {} phase failed in {}", (Object)this.sessionID, (Object)CoordinatorSession.formatDuration(this.sessionStart, this.ctx.clock().currentTimeMillis()));
                }
                this.fail();
            }
        }, (Executor)ImmediateExecutor.INSTANCE);
    }

    public static class Builder
    extends ConsistentSession.AbstractBuilder {
        private SharedContext ctx;

        public Builder(SharedContext ctx) {
            super(ctx);
        }

        public Builder withContext(SharedContext ctx) {
            this.ctx = ctx;
            return this;
        }

        public CoordinatorSession build() {
            this.validate();
            return new CoordinatorSession(this);
        }
    }
}

