/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.index.sai.disk.v1;

import com.google.common.base.Stopwatch;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.BooleanSupplier;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.index.sai.IndexContext;
import org.apache.cassandra.index.sai.analyzer.AbstractAnalyzer;
import org.apache.cassandra.index.sai.disk.PerColumnIndexWriter;
import org.apache.cassandra.index.sai.disk.format.IndexComponent;
import org.apache.cassandra.index.sai.disk.format.IndexDescriptor;
import org.apache.cassandra.index.sai.disk.v1.MetadataWriter;
import org.apache.cassandra.index.sai.disk.v1.segment.SegmentBuilder;
import org.apache.cassandra.index.sai.disk.v1.segment.SegmentMetadata;
import org.apache.cassandra.index.sai.utils.NamedMemoryLimiter;
import org.apache.cassandra.index.sai.utils.PrimaryKey;
import org.apache.cassandra.index.sai.utils.TypeUtil;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.NoSpamLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class SSTableIndexWriter
implements PerColumnIndexWriter {
    private static final Logger logger = LoggerFactory.getLogger(SSTableIndexWriter.class);
    private static final NoSpamLogger noSpamLogger = NoSpamLogger.getLogger(logger, 1L, TimeUnit.MINUTES);
    public static final int MAX_STRING_TERM_SIZE = CassandraRelevantProperties.SAI_MAX_STRING_TERM_SIZE.getInt() * 1024;
    public static final int MAX_FROZEN_TERM_SIZE = CassandraRelevantProperties.SAI_MAX_FROZEN_TERM_SIZE.getInt() * 1024;
    public static final String TERM_OVERSIZE_MESSAGE = "Can't add term of column {} to index for key: {}, term size {} max allowed size {}, use analyzed = true (if not yet set) for that column.";
    private final IndexDescriptor indexDescriptor;
    private final IndexContext indexContext;
    private final long nowInSec = FBUtilities.nowInSeconds();
    private final AbstractAnalyzer analyzer;
    private final NamedMemoryLimiter limiter;
    private final int maxTermSize;
    private final BooleanSupplier isIndexValid;
    private final List<SegmentMetadata> segments = new ArrayList<SegmentMetadata>();
    private boolean aborted = false;
    private SegmentBuilder currentBuilder;

    public SSTableIndexWriter(IndexDescriptor indexDescriptor, IndexContext indexContext, NamedMemoryLimiter limiter, BooleanSupplier isIndexValid) {
        this.indexDescriptor = indexDescriptor;
        this.indexContext = indexContext;
        this.analyzer = indexContext.getAnalyzerFactory().create();
        this.limiter = limiter;
        this.isIndexValid = isIndexValid;
        this.maxTermSize = indexContext.isFrozen() ? MAX_FROZEN_TERM_SIZE : MAX_STRING_TERM_SIZE;
    }

    @Override
    public void addRow(PrimaryKey key, Row row, long sstableRowId) throws IOException {
        if (this.maybeAbort()) {
            return;
        }
        if (this.indexContext.isNonFrozenCollection()) {
            Iterator<ByteBuffer> valueIterator = this.indexContext.getValuesOf(row, this.nowInSec);
            if (valueIterator != null) {
                while (valueIterator.hasNext()) {
                    ByteBuffer value = valueIterator.next();
                    this.addTerm(TypeUtil.asIndexBytes(value.duplicate(), this.indexContext.getValidator()), key, sstableRowId, this.indexContext.getValidator());
                }
            }
        } else {
            ByteBuffer value = this.indexContext.getValueOf(key.partitionKey(), row, this.nowInSec);
            if (value != null) {
                this.addTerm(TypeUtil.asIndexBytes(value.duplicate(), this.indexContext.getValidator()), key, sstableRowId, this.indexContext.getValidator());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void complete(Stopwatch stopwatch) throws IOException {
        if (this.maybeAbort()) {
            return;
        }
        long start = stopwatch.elapsed(TimeUnit.MILLISECONDS);
        boolean emptySegment = this.currentBuilder == null || this.currentBuilder.isEmpty();
        logger.debug(this.indexContext.logMessage("Completing index flush with {}buffered data..."), (Object)(emptySegment ? "no " : ""));
        try {
            if (!emptySegment) {
                this.flushSegment();
                long elapsed = stopwatch.elapsed(TimeUnit.MILLISECONDS);
                logger.debug(this.indexContext.logMessage("Completed flush of final segment for SSTable {}. Duration: {} ms. Total elapsed: {} ms"), new Object[]{this.indexDescriptor.sstableDescriptor, elapsed - start, elapsed});
            }
            if (this.currentBuilder != null) {
                long bytesAllocated = this.currentBuilder.totalBytesAllocated();
                long globalBytesUsed = this.currentBuilder.release(this.indexContext);
                logger.debug(this.indexContext.logMessage("Flushing final segment for SSTable {} released {}. Global segment memory usage now at {}."), new Object[]{this.indexDescriptor.sstableDescriptor, FBUtilities.prettyPrintMemory(bytesAllocated), FBUtilities.prettyPrintMemory(globalBytesUsed)});
            }
            this.writeSegmentsMetadata();
            this.indexDescriptor.createComponentOnDisk(IndexComponent.COLUMN_COMPLETION_MARKER, this.indexContext);
        }
        finally {
            if (this.indexContext.getIndexMetrics() != null) {
                this.indexContext.getIndexMetrics().segmentsPerCompaction.update(this.segments.size());
                this.segments.clear();
                this.indexContext.getIndexMetrics().compactionCount.inc();
            }
        }
    }

    @Override
    public void abort(Throwable cause) {
        this.aborted = true;
        logger.warn(this.indexContext.logMessage("Aborting SSTable index flush for {}..."), (Object)this.indexDescriptor.sstableDescriptor, (Object)cause);
        if (this.currentBuilder != null) {
            long allocated = this.currentBuilder.totalBytesAllocated();
            long globalBytesUsed = this.currentBuilder.release(this.indexContext);
            logger.debug(this.indexContext.logMessage("Aborting index writer for SSTable {} released {}. Global segment memory usage now at {}."), new Object[]{this.indexDescriptor.sstableDescriptor, FBUtilities.prettyPrintMemory(allocated), FBUtilities.prettyPrintMemory(globalBytesUsed)});
        }
        this.indexDescriptor.deleteColumnIndex(this.indexContext);
    }

    private boolean maybeAbort() {
        if (this.aborted) {
            return true;
        }
        if (this.isIndexValid.getAsBoolean()) {
            return false;
        }
        this.abort(new RuntimeException(String.format("index %s is dropped", this.indexContext.getIndexName())));
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addTerm(ByteBuffer term, PrimaryKey key, long sstableRowId, AbstractType<?> type) throws IOException {
        if (term.remaining() >= this.maxTermSize) {
            noSpamLogger.warn(this.indexContext.logMessage(TERM_OVERSIZE_MESSAGE), this.indexContext.getColumnName(), this.indexContext.keyValidator().getString(key.partitionKey().getKey()), FBUtilities.prettyPrintMemory(term.remaining()), FBUtilities.prettyPrintMemory(this.maxTermSize));
            return;
        }
        if (this.currentBuilder == null) {
            this.currentBuilder = this.newSegmentBuilder();
        } else if (this.shouldFlush(sstableRowId)) {
            this.flushSegment();
            this.currentBuilder = this.newSegmentBuilder();
        }
        if (term.remaining() == 0) {
            return;
        }
        if (!TypeUtil.isLiteral(type)) {
            this.limiter.increment(this.currentBuilder.add(term, key, sstableRowId));
        } else {
            this.analyzer.reset(term);
            try {
                while (this.analyzer.hasNext()) {
                    ByteBuffer tokenTerm = this.analyzer.next();
                    this.limiter.increment(this.currentBuilder.add(tokenTerm, key, sstableRowId));
                }
            }
            finally {
                this.analyzer.end();
            }
        }
    }

    private boolean shouldFlush(long sstableRowId) {
        boolean reachMemoryLimit;
        boolean bl = reachMemoryLimit = this.limiter.usageExceedsLimit() && this.currentBuilder.hasReachedMinimumFlushSize();
        if (reachMemoryLimit) {
            logger.debug(this.indexContext.logMessage("Global limit of {} and minimum flush size of {} exceeded. Current builder usage is {} for {} cells. Global Usage is {}. Flushing..."), new Object[]{FBUtilities.prettyPrintMemory(this.limiter.limitBytes()), FBUtilities.prettyPrintMemory(this.currentBuilder.getMinimumFlushBytes()), FBUtilities.prettyPrintMemory(this.currentBuilder.totalBytesAllocated()), this.currentBuilder.getRowCount(), FBUtilities.prettyPrintMemory(this.limiter.currentBytesUsed())});
        }
        return reachMemoryLimit || this.currentBuilder.exceedsSegmentLimit(sstableRowId);
    }

    private void flushSegment() throws IOException {
        long start = Clock.Global.nanoTime();
        try {
            long bytesAllocated = this.currentBuilder.totalBytesAllocated();
            SegmentMetadata segmentMetadata = this.currentBuilder.flush(this.indexDescriptor, this.indexContext);
            long flushMillis = Math.max(1L, TimeUnit.NANOSECONDS.toMillis(Clock.Global.nanoTime() - start));
            if (segmentMetadata != null) {
                this.segments.add(segmentMetadata);
                double rowCount = segmentMetadata.numRows;
                if (this.indexContext.getIndexMetrics() != null) {
                    this.indexContext.getIndexMetrics().compactionSegmentCellsPerSecond.update((long)(rowCount / (double)flushMillis * 1000.0));
                }
                double segmentBytes = segmentMetadata.componentMetadatas.indexSize();
                if (this.indexContext.getIndexMetrics() != null) {
                    this.indexContext.getIndexMetrics().compactionSegmentBytesPerSecond.update((long)(segmentBytes / (double)flushMillis * 1000.0));
                }
                logger.debug(this.indexContext.logMessage("Flushed segment with {} cells for a total of {} in {} ms."), new Object[]{(long)rowCount, FBUtilities.prettyPrintMemory((long)segmentBytes), flushMillis});
            }
            long globalBytesUsed = this.currentBuilder.release(this.indexContext);
            this.currentBuilder = null;
            logger.debug(this.indexContext.logMessage("Flushing index segment for SSTable {} released {}. Global segment memory usage now at {}."), new Object[]{this.indexDescriptor.sstableDescriptor, FBUtilities.prettyPrintMemory(bytesAllocated), FBUtilities.prettyPrintMemory(globalBytesUsed)});
        }
        catch (Throwable t) {
            logger.error(this.indexContext.logMessage("Failed to build index for SSTable {}."), (Object)this.indexDescriptor.sstableDescriptor, (Object)t);
            this.indexDescriptor.deleteColumnIndex(this.indexContext);
            this.indexContext.getIndexMetrics().segmentFlushErrors.inc();
            throw t;
        }
    }

    private void writeSegmentsMetadata() throws IOException {
        if (this.segments.isEmpty()) {
            return;
        }
        try (MetadataWriter writer = new MetadataWriter(this.indexDescriptor.openPerIndexOutput(IndexComponent.META, this.indexContext));){
            SegmentMetadata.write(writer, this.segments);
        }
        catch (IOException e) {
            this.abort(e);
            throw e;
        }
    }

    private SegmentBuilder newSegmentBuilder() {
        SegmentBuilder builder = TypeUtil.isLiteral(this.indexContext.getValidator()) ? new SegmentBuilder.RAMStringSegmentBuilder(this.indexContext.getValidator(), this.limiter) : new SegmentBuilder.BlockBalancedTreeSegmentBuilder(this.indexContext.getValidator(), this.limiter);
        long globalBytesUsed = this.limiter.increment(builder.totalBytesAllocated());
        logger.debug(this.indexContext.logMessage("Created new segment builder while flushing SSTable {}. Global segment memory usage now at {}."), (Object)this.indexDescriptor.sstableDescriptor, (Object)FBUtilities.prettyPrintMemory(globalBytesUsed));
        return builder;
    }
}

