/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.net;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.HashMultimap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.gms.IEndpointStateChangeSubscriber;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.ConnectionType;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.PingRequest;
import org.apache.cassandra.net.RequestCallback;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.concurrent.CountDownLatch;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StartupClusterConnectivityChecker {
    private static final Logger logger = LoggerFactory.getLogger(StartupClusterConnectivityChecker.class);
    private final boolean blockForRemoteDcs;
    private final long timeoutNanos;

    public static StartupClusterConnectivityChecker create(long timeoutSecs, boolean blockForRemoteDcs) {
        if (timeoutSecs > 100L) {
            logger.warn("setting the block-for-peers timeout (in seconds) to {} might be a bit excessive, but using it nonetheless", (Object)timeoutSecs);
        }
        long timeoutNanos = TimeUnit.SECONDS.toNanos(timeoutSecs);
        return new StartupClusterConnectivityChecker(timeoutNanos, blockForRemoteDcs);
    }

    @VisibleForTesting
    StartupClusterConnectivityChecker(long timeoutNanos, boolean blockForRemoteDcs) {
        this.blockForRemoteDcs = blockForRemoteDcs;
        this.timeoutNanos = timeoutNanos;
    }

    public boolean execute(Set<InetAddressAndPort> peers, Function<InetAddressAndPort, String> getDatacenterSource) {
        if (peers == null || this.timeoutNanos < 0L) {
            return true;
        }
        peers = new HashSet<InetAddressAndPort>(peers);
        InetAddressAndPort localAddress = FBUtilities.getBroadcastAddressAndPort();
        String localDc = getDatacenterSource.apply(localAddress);
        peers.remove(localAddress);
        if (peers.isEmpty()) {
            return true;
        }
        HashMap<InetAddressAndPort, String> peerToDatacenter = new HashMap<InetAddressAndPort, String>();
        HashMultimap datacenterToPeers = HashMultimap.create();
        for (InetAddressAndPort peer2 : peers) {
            String datacenter = getDatacenterSource.apply(peer2);
            peerToDatacenter.put(peer2, datacenter);
            datacenterToPeers.put((Object)datacenter, (Object)peer2);
        }
        if (!this.blockForRemoteDcs) {
            datacenterToPeers.keySet().retainAll(Collections.singleton(localDc));
            logger.info("Blocking coordination until only a single peer is DOWN in the local datacenter, timeout={}s", (Object)TimeUnit.NANOSECONDS.toSeconds(this.timeoutNanos));
        } else {
            logger.info("Blocking coordination until only a single peer is DOWN in each datacenter, timeout={}s", (Object)TimeUnit.NANOSECONDS.toSeconds(this.timeoutNanos));
        }
        AckMap acks = new AckMap(3, peers);
        HashMap<String, CountDownLatch> dcToRemainingPeers = new HashMap<String, CountDownLatch>(datacenterToPeers.size());
        for (String datacenter : datacenterToPeers.keys()) {
            dcToRemainingPeers.put(datacenter, CountDownLatch.newCountDownLatch(Math.max(datacenterToPeers.get((Object)datacenter).size() - 1, 0)));
        }
        long startNanos = Clock.Global.nanoTime();
        Set<InetAddressAndPort> alivePeers = Collections.newSetFromMap(new ConcurrentHashMap());
        AliveListener listener = new AliveListener(alivePeers, dcToRemainingPeers, acks, peerToDatacenter::get);
        Gossiper.instance.register(listener);
        this.sendPingMessages(peers, dcToRemainingPeers, acks, peerToDatacenter::get);
        for (InetAddressAndPort inetAddressAndPort : peers) {
            String datacenter;
            if (!Gossiper.instance.isAlive(inetAddressAndPort) || !alivePeers.add(inetAddressAndPort) || !acks.incrementAndCheck(inetAddressAndPort) || !dcToRemainingPeers.containsKey(datacenter = (String)peerToDatacenter.get(inetAddressAndPort))) continue;
            ((CountDownLatch)dcToRemainingPeers.get(datacenter)).decrement();
        }
        boolean succeeded = true;
        for (CountDownLatch countDownLatch : dcToRemainingPeers.values()) {
            long remainingNanos = Math.max(1L, this.timeoutNanos - (Clock.Global.nanoTime() - startNanos));
            succeeded &= countDownLatch.awaitUninterruptibly(remainingNanos, TimeUnit.NANOSECONDS);
        }
        Gossiper.instance.unregister(listener);
        if (succeeded) {
            logger.info("Ensured sufficient healthy connections with {} after {} milliseconds", dcToRemainingPeers.keySet(), (Object)TimeUnit.NANOSECONDS.toMillis(Clock.Global.nanoTime() - startNanos));
        } else {
            Map map = acks.getMissingPeers().stream().collect(Collectors.groupingBy(peer -> {
                String dc = (String)peerToDatacenter.get(peer);
                if (dc != null) {
                    return dc;
                }
                return StringUtils.defaultString((String)((String)getDatacenterSource.apply((InetAddressAndPort)peer)), (String)"unknown");
            }, Collectors.mapping(InetAddressAndPort::getHostAddressAndPort, Collectors.toList())));
            logger.warn("Timed out after {} milliseconds, was waiting for remaining peers to connect: {}", (Object)TimeUnit.NANOSECONDS.toMillis(Clock.Global.nanoTime() - startNanos), map);
        }
        return succeeded;
    }

    private void sendPingMessages(Set<InetAddressAndPort> peers, Map<String, CountDownLatch> dcToRemainingPeers, AckMap acks, Function<InetAddressAndPort, String> getDatacenter) {
        RequestCallback responseHandler = msg -> {
            String datacenter;
            if (acks.incrementAndCheck(msg.from()) && dcToRemainingPeers.containsKey(datacenter = (String)getDatacenter.apply(msg.from()))) {
                ((CountDownLatch)dcToRemainingPeers.get(datacenter)).decrement();
            }
        };
        Message<PingRequest> small = Message.out(Verb.PING_REQ, PingRequest.forSmall);
        Message<PingRequest> large = Message.out(Verb.PING_REQ, PingRequest.forLarge);
        for (InetAddressAndPort peer : peers) {
            MessagingService.instance().sendWithCallback(small, peer, responseHandler, ConnectionType.SMALL_MESSAGES);
            MessagingService.instance().sendWithCallback(large, peer, responseHandler, ConnectionType.LARGE_MESSAGES);
        }
    }

    private static final class AckMap {
        private final int threshold;
        private final Map<InetAddressAndPort, AtomicInteger> acks;

        AckMap(int threshold, Iterable<InetAddressAndPort> initialPeers) {
            this.threshold = threshold;
            this.acks = new ConcurrentHashMap<InetAddressAndPort, AtomicInteger>();
            for (InetAddressAndPort peer : initialPeers) {
                this.initOrGetCounter(peer);
            }
        }

        boolean incrementAndCheck(InetAddressAndPort address) {
            return this.initOrGetCounter(address).incrementAndGet() == this.threshold;
        }

        List<InetAddressAndPort> getMissingPeers() {
            ArrayList<InetAddressAndPort> missingPeers = new ArrayList<InetAddressAndPort>();
            for (Map.Entry<InetAddressAndPort, AtomicInteger> entry : this.acks.entrySet()) {
                if (entry.getValue().get() >= this.threshold) continue;
                missingPeers.add(entry.getKey());
            }
            return missingPeers;
        }

        private AtomicInteger initOrGetCounter(InetAddressAndPort address) {
            return this.acks.computeIfAbsent(address, addr -> new AtomicInteger(0));
        }
    }

    private static final class AliveListener
    implements IEndpointStateChangeSubscriber {
        private final Map<String, CountDownLatch> dcToRemainingPeers;
        private final Set<InetAddressAndPort> livePeers;
        private final Function<InetAddressAndPort, String> getDatacenter;
        private final AckMap acks;

        AliveListener(Set<InetAddressAndPort> livePeers, Map<String, CountDownLatch> dcToRemainingPeers, AckMap acks, Function<InetAddressAndPort, String> getDatacenter) {
            this.livePeers = livePeers;
            this.dcToRemainingPeers = dcToRemainingPeers;
            this.acks = acks;
            this.getDatacenter = getDatacenter;
        }

        @Override
        public void onAlive(InetAddressAndPort endpoint, EndpointState state) {
            String datacenter;
            if (this.livePeers.add(endpoint) && this.acks.incrementAndCheck(endpoint) && this.dcToRemainingPeers.containsKey(datacenter = this.getDatacenter.apply(endpoint))) {
                this.dcToRemainingPeers.get(datacenter).decrement();
            }
        }
    }
}

