/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import com.google.common.util.concurrent.Futures;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Future;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.exceptions.AlreadyExistsException;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.schema.KeyspaceMetadata;
import org.apache.cassandra.schema.Keyspaces;
import org.apache.cassandra.schema.MigrationCoordinator;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.schema.SchemaAnnouncementDiagnostics;
import org.apache.cassandra.schema.SchemaKeyspace;
import org.apache.cassandra.schema.SchemaMigrationDiagnostics;
import org.apache.cassandra.schema.SchemaTransformation;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.schema.Tables;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationManager {
    private static final Logger logger = LoggerFactory.getLogger(MigrationManager.class);
    public static final MigrationManager instance = new MigrationManager();

    private MigrationManager() {
    }

    private static boolean shouldPushSchemaTo(InetAddressAndPort endpoint) {
        return !endpoint.equals(FBUtilities.getBroadcastAddressAndPort()) && MessagingService.instance().versions.knows(endpoint) && MessagingService.instance().versions.getRaw(endpoint) == 12;
    }

    public static void announceNewKeyspace(KeyspaceMetadata ksm) throws ConfigurationException {
        MigrationManager.announceNewKeyspace(ksm, false);
    }

    public static void announceNewKeyspace(KeyspaceMetadata ksm, boolean announceLocally) throws ConfigurationException {
        MigrationManager.announceNewKeyspace(ksm, FBUtilities.timestampMicros(), announceLocally);
    }

    public static void announceNewKeyspace(KeyspaceMetadata ksm, long timestamp, boolean announceLocally) throws ConfigurationException {
        ksm.validate();
        if (Schema.instance.getKeyspaceMetadata(ksm.name) != null) {
            throw new AlreadyExistsException(ksm.name);
        }
        logger.info("Create new Keyspace: {}", (Object)ksm);
        MigrationManager.announce(SchemaKeyspace.makeCreateKeyspaceMutation(ksm, timestamp), announceLocally);
    }

    public static void announceNewTable(TableMetadata cfm) {
        MigrationManager.announceNewTable(cfm, true, FBUtilities.timestampMicros());
    }

    private static void announceNewTable(TableMetadata cfm, boolean throwOnDuplicate, long timestamp) {
        cfm.validate();
        KeyspaceMetadata ksm = Schema.instance.getKeyspaceMetadata(cfm.keyspace);
        if (ksm == null) {
            throw new ConfigurationException(String.format("Cannot add table '%s' to non existing keyspace '%s'.", cfm.name, cfm.keyspace));
        }
        if (throwOnDuplicate && ksm.getTableOrViewNullable(cfm.name) != null) {
            throw new AlreadyExistsException(cfm.keyspace, cfm.name);
        }
        logger.info("Create new table: {}", (Object)cfm);
        MigrationManager.announce(SchemaKeyspace.makeCreateTableMutation(ksm, cfm, timestamp), false);
    }

    static void announceKeyspaceUpdate(KeyspaceMetadata ksm) {
        ksm.validate();
        KeyspaceMetadata oldKsm = Schema.instance.getKeyspaceMetadata(ksm.name);
        if (oldKsm == null) {
            throw new ConfigurationException(String.format("Cannot update non existing keyspace '%s'.", ksm.name));
        }
        logger.info("Update Keyspace '{}' From {} To {}", new Object[]{ksm.name, oldKsm, ksm});
        MigrationManager.announce(SchemaKeyspace.makeCreateKeyspaceMutation(ksm.name, ksm.params, FBUtilities.timestampMicros()), false);
    }

    public static void announceTableUpdate(TableMetadata tm) {
        MigrationManager.announceTableUpdate(tm, false);
    }

    public static void announceTableUpdate(TableMetadata updated, boolean announceLocally) {
        updated.validate();
        TableMetadata current = Schema.instance.getTableMetadata(updated.keyspace, updated.name);
        if (current == null) {
            throw new ConfigurationException(String.format("Cannot update non existing table '%s' in keyspace '%s'.", updated.name, updated.keyspace));
        }
        KeyspaceMetadata ksm = Schema.instance.getKeyspaceMetadata(current.keyspace);
        updated.validateCompatibility(current);
        long timestamp = FBUtilities.timestampMicros();
        logger.info("Update table '{}/{}' From {} To {}", new Object[]{current.keyspace, current.name, current, updated});
        Mutation.SimpleBuilder builder = SchemaKeyspace.makeUpdateTableMutation(ksm, current, updated, timestamp);
        MigrationManager.announce(builder, announceLocally);
    }

    static void announceKeyspaceDrop(String ksName) {
        KeyspaceMetadata oldKsm = Schema.instance.getKeyspaceMetadata(ksName);
        if (oldKsm == null) {
            throw new ConfigurationException(String.format("Cannot drop non existing keyspace '%s'.", ksName));
        }
        logger.info("Drop Keyspace '{}'", (Object)oldKsm.name);
        MigrationManager.announce(SchemaKeyspace.makeDropKeyspaceMutation(oldKsm, FBUtilities.timestampMicros()), false);
    }

    public static void announceTableDrop(String ksName, String cfName, boolean announceLocally) {
        TableMetadata tm = Schema.instance.getTableMetadata(ksName, cfName);
        if (tm == null) {
            throw new ConfigurationException(String.format("Cannot drop non existing table '%s' in keyspace '%s'.", cfName, ksName));
        }
        KeyspaceMetadata ksm = Schema.instance.getKeyspaceMetadata(ksName);
        logger.info("Drop table '{}/{}'", (Object)tm.keyspace, (Object)tm.name);
        MigrationManager.announce(SchemaKeyspace.makeDropTableMutation(ksm, tm, FBUtilities.timestampMicros()), announceLocally);
    }

    private static void announce(Mutation.SimpleBuilder schema, boolean announceLocally) {
        List<Mutation> mutations = Collections.singletonList(schema.build());
        if (announceLocally) {
            Schema.instance.merge(mutations);
        } else {
            MigrationManager.announce(mutations);
        }
    }

    public static void announce(Mutation change) {
        MigrationManager.announce(Collections.singleton(change));
    }

    public static void announce(Collection<Mutation> schema) {
        Future<?> f = MigrationManager.announceWithoutPush(schema);
        HashSet<InetAddressAndPort> schemaDestinationEndpoints = new HashSet<InetAddressAndPort>();
        HashSet<InetAddressAndPort> schemaEndpointsIgnored = new HashSet<InetAddressAndPort>();
        Message<Collection<Mutation>> message = Message.out(Verb.SCHEMA_PUSH_REQ, schema);
        for (InetAddressAndPort endpoint : Gossiper.instance.getLiveMembers()) {
            if (MigrationManager.shouldPushSchemaTo(endpoint)) {
                MessagingService.instance().send(message, endpoint);
                schemaDestinationEndpoints.add(endpoint);
                continue;
            }
            schemaEndpointsIgnored.add(endpoint);
        }
        SchemaAnnouncementDiagnostics.schemaMutationsAnnounced(schemaDestinationEndpoints, schemaEndpointsIgnored);
        FBUtilities.waitOnFuture(f);
    }

    public static Future<?> announceWithoutPush(Collection<Mutation> schema) {
        return Stage.MIGRATION.submit(() -> Schema.instance.mergeAndAnnounceVersion(schema));
    }

    public static Keyspaces.KeyspacesDiff announce(SchemaTransformation transformation, boolean locally) {
        long now = FBUtilities.timestampMicros();
        Future<Schema.TransformationResult> future = Stage.MIGRATION.submit(() -> Schema.instance.transform(transformation, locally, now));
        Schema.TransformationResult result = (Schema.TransformationResult)Futures.getUnchecked(future);
        if (!result.success) {
            throw result.exception;
        }
        if (locally || result.diff.isEmpty()) {
            return result.diff;
        }
        HashSet<InetAddressAndPort> schemaDestinationEndpoints = new HashSet<InetAddressAndPort>();
        HashSet<InetAddressAndPort> schemaEndpointsIgnored = new HashSet<InetAddressAndPort>();
        Message<Collection<Mutation>> message = Message.out(Verb.SCHEMA_PUSH_REQ, result.mutations);
        for (InetAddressAndPort endpoint : Gossiper.instance.getLiveMembers()) {
            if (MigrationManager.shouldPushSchemaTo(endpoint)) {
                MessagingService.instance().send(message, endpoint);
                schemaDestinationEndpoints.add(endpoint);
                continue;
            }
            schemaEndpointsIgnored.add(endpoint);
        }
        SchemaAnnouncementDiagnostics.schemaTransformationAnnounced(schemaDestinationEndpoints, schemaEndpointsIgnored, transformation);
        return result.diff;
    }

    public static void resetLocalSchema() {
        logger.info("Starting local schema reset...");
        logger.debug("Truncating schema tables...");
        SchemaMigrationDiagnostics.resetLocalSchema();
        Schema.instance.truncateSchemaKeyspace();
        logger.debug("Clearing local schema keyspace definitions...");
        Schema.instance.clear();
        Set<InetAddressAndPort> liveEndpoints = Gossiper.instance.getLiveMembers();
        liveEndpoints.remove(FBUtilities.getBroadcastAddressAndPort());
        for (InetAddressAndPort node : liveEndpoints) {
            EndpointState state;
            Future<Void> pull = MigrationCoordinator.instance.reportEndpointVersion(node, state = Gossiper.instance.getEndpointStateForEndpoint(node));
            if (pull == null) continue;
            FBUtilities.waitOnFuture(pull);
        }
        logger.info("Local schema reset is complete.");
    }

    public static Optional<Mutation> evolveSystemKeyspace(KeyspaceMetadata keyspace, long generation) {
        Mutation.SimpleBuilder builder = null;
        KeyspaceMetadata definedKeyspace = Schema.instance.getKeyspaceMetadata(keyspace.name);
        Tables definedTables = null == definedKeyspace ? Tables.none() : definedKeyspace.tables;
        for (TableMetadata table : keyspace.tables) {
            if (table.equals(definedTables.getNullable(table.name))) continue;
            if (null == builder) {
                builder = SchemaKeyspace.makeCreateKeyspaceMutation(keyspace.name, keyspace.params, 0L);
                builder.timestamp(generation);
            }
            SchemaKeyspace.addTableToSchemaMutation(table, true, builder);
        }
        return builder == null ? Optional.empty() : Optional.of(builder.build());
    }

    public static class MigrationsSerializer
    implements IVersionedSerializer<Collection<Mutation>> {
        public static MigrationsSerializer instance = new MigrationsSerializer();

        @Override
        public void serialize(Collection<Mutation> schema, DataOutputPlus out, int version) throws IOException {
            out.writeInt(schema.size());
            for (Mutation mutation : schema) {
                Mutation.serializer.serialize(mutation, out, version);
            }
        }

        @Override
        public Collection<Mutation> deserialize(DataInputPlus in, int version) throws IOException {
            int count = in.readInt();
            ArrayList<Mutation> schema = new ArrayList<Mutation>(count);
            for (int i = 0; i < count; ++i) {
                schema.add(Mutation.serializer.deserialize(in, version));
            }
            return schema;
        }

        @Override
        public long serializedSize(Collection<Mutation> schema, int version) {
            int size = TypeSizes.sizeof(schema.size());
            for (Mutation mutation : schema) {
                size += mutation.serializedSize(version);
            }
            return size;
        }
    }
}

