/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.carbondata.presto;

import java.util.Objects;
import java.util.Optional;

import static java.util.Objects.requireNonNull;

import com.facebook.presto.spi.predicate.Domain;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSetter;

import static com.google.common.base.MoreObjects.toStringHelper;

/**
 * Encapsulating presto Tuple-domain
 */
public class CarbondataColumnConstraint {
  private final String name;
  private final boolean invertedIndexed;
  private Optional<Domain> domain;

  @JsonCreator public CarbondataColumnConstraint(@JsonProperty("name") String name,
      @JsonProperty("domain") Optional<Domain> domain,
      @JsonProperty("invertedIndexed") boolean invertedIndexed) {
    this.name = requireNonNull(name, "name is null");
    this.invertedIndexed = requireNonNull(invertedIndexed, "invertedIndexed is null");
    this.domain = requireNonNull(domain, "domain is null");
  }

  @JsonProperty
  public boolean isInvertedIndexed() {
    return invertedIndexed;
  }

  @JsonProperty
  public String getName() {
    return name;
  }

  @JsonProperty
  public Optional<Domain> getDomain() {
    return domain;
  }

  @JsonSetter
  public void setDomain(Optional<Domain> domain) {
    this.domain = domain;
  }

  @Override
  public int hashCode() {
    return Objects.hash(name, domain, invertedIndexed);
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }

    if ((obj == null) || (getClass() != obj.getClass())) {
      return false;
    }

    CarbondataColumnConstraint other = (CarbondataColumnConstraint) obj;
    return Objects.equals(this.name, other.name) && Objects.equals(this.domain, other.domain)
        && Objects.equals(this.invertedIndexed, other.invertedIndexed);
  }

  @Override
  public String toString() {
    return toStringHelper(this).add("name", this.name).add("invertedindexed", this.invertedIndexed)
        .add("domain", this.domain).toString();
  }
}
