/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.quarkus.component.azure.key.vault.deployment;

import java.util.Set;
import java.util.stream.Collectors;

import com.azure.json.JsonSerializable;
import com.azure.security.keyvault.secrets.implementation.SecretClientImpl;
import com.microsoft.azure.proton.transport.proxy.impl.DigestProxyChallengeProcessorImpl;
import io.quarkus.deployment.annotations.BuildProducer;
import io.quarkus.deployment.annotations.BuildStep;
import io.quarkus.deployment.builditem.CombinedIndexBuildItem;
import io.quarkus.deployment.builditem.ExtensionSslNativeSupportBuildItem;
import io.quarkus.deployment.builditem.FeatureBuildItem;
import io.quarkus.deployment.builditem.IndexDependencyBuildItem;
import io.quarkus.deployment.builditem.nativeimage.NativeImageProxyDefinitionBuildItem;
import io.quarkus.deployment.builditem.nativeimage.ReflectiveClassBuildItem;
import io.quarkus.deployment.builditem.nativeimage.RuntimeInitializedClassBuildItem;
import org.jboss.jandex.ClassInfo;

class AzureKeyVaultProcessor {
    private static final String FEATURE = "camel-azure-key-vault";

    @BuildStep
    FeatureBuildItem feature() {
        return new FeatureBuildItem(FEATURE);
    }

    @BuildStep
    ExtensionSslNativeSupportBuildItem enableNativeSslSupport() {
        return new ExtensionSslNativeSupportBuildItem(FEATURE);
    }

    @BuildStep
    IndexDependencyBuildItem indexDependencies() {
        return new IndexDependencyBuildItem("com.azure", "azure-security-keyvault-secrets");
    }

    @BuildStep
    void registerForReflection(CombinedIndexBuildItem combinedIndex, BuildProducer<ReflectiveClassBuildItem> reflectiveClass) {
        Set<String> keyVaultModelClasses = combinedIndex.getIndex()
                .getAllKnownImplementors(JsonSerializable.class)
                .stream()
                .map(ClassInfo::toString)
                .filter(className -> className.startsWith("com.azure.security.keyvault"))
                .collect(Collectors.toSet());

        reflectiveClass.produce(ReflectiveClassBuildItem.builder(keyVaultModelClasses.toArray(new String[0]))
                .methods(true)
                .build());
    }

    @BuildStep
    void runtimeInitializedClasses(BuildProducer<RuntimeInitializedClassBuildItem> runtimeInitializedClass) {
        runtimeInitializedClass
                .produce(new RuntimeInitializedClassBuildItem("com.microsoft.azure.proton.transport.ws.impl.Utils"));
        runtimeInitializedClass
                .produce(new RuntimeInitializedClassBuildItem(DigestProxyChallengeProcessorImpl.class.getName()));
    }

    @BuildStep
    NativeImageProxyDefinitionBuildItem nativeImageProxyDefinitions() {
        return new NativeImageProxyDefinitionBuildItem(SecretClientImpl.SecretClientService.class.getName());
    }
}
