/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import React, {useRef, useState} from 'react';
import {
    FormGroup,
    TextInput,
    Popover,
    Switch, InputGroup, Button, TextArea, Tooltip, capitalize
} from '@patternfly/react-core';
import '../../karavan.css';
import "@patternfly/patternfly/patternfly.css";
import HelpIcon from "@patternfly/react-icons/dist/js/icons/help-icon";
import ExpandIcon from "@patternfly/react-icons/dist/js/icons/expand-icon";
import CompressIcon from "@patternfly/react-icons/dist/js/icons/compress-icon";
import {Property} from "karavan-core/lib/model/KameletModels";
import {InfrastructureSelector} from "./InfrastructureSelector";
import {InfrastructureAPI} from "../../utils/InfrastructureAPI";
import KubernetesIcon from "@patternfly/react-icons/dist/js/icons/openshift-icon";
import ShowIcon from "@patternfly/react-icons/dist/js/icons/eye-icon";
import HideIcon from "@patternfly/react-icons/dist/js/icons/eye-slash-icon";
import DockerIcon from "@patternfly/react-icons/dist/js/icons/docker-icon";
import {usePropertiesHook} from "../usePropertiesHook";

interface Props {
    property: Property,
    value: any,
    required: boolean,
}

export function KameletPropertyField(props: Props) {

    const {onParametersChange} = usePropertiesHook();

    const [selectIsOpen, setSelectIsOpen] = useState<boolean>(false);
    const [showEditor, setShowEditor] = useState<boolean>(false);
    const [showPassword, setShowPassword] = useState<boolean>(false);
    const [infrastructureSelector, setInfrastructureSelector] = useState<boolean>(false);
    const [infrastructureSelectorProperty, setInfrastructureSelectorProperty] = useState<string | undefined>(undefined);

    const ref = useRef<any>(null);

    function parametersChanged (parameter: string, value: string | number | boolean | any, pathParameter?: boolean)  {
        onParametersChange(parameter, value, pathParameter);
        setSelectIsOpen(false);
    }

    function selectInfrastructure (value: string)  {
        // check if there is a selection
        const textVal = ref.current;
        const cursorStart = textVal.selectionStart;
        const cursorEnd = textVal.selectionEnd;
        if (cursorStart !== cursorEnd){
            const prevValue =  props.value;
            const selectedText = prevValue.substring(cursorStart, cursorEnd)
            value = prevValue.replace(selectedText, value);
        }
        const propertyId = infrastructureSelectorProperty;
        if (propertyId){
            if (value.startsWith("config") || value.startsWith("secret")) value = "{{" + value + "}}";
            parametersChanged(propertyId, value);
            setInfrastructureSelector(false);
            setInfrastructureSelectorProperty(undefined);
        }
    }

    function openInfrastructureSelector (propertyName: string)  {
        setInfrastructureSelector(true);
        setInfrastructureSelectorProperty(propertyName);
    }

    function getInfrastructureSelectorModal() {
        return (
            <InfrastructureSelector
                dark={false}
                isOpen={infrastructureSelector}
                onClose={() => setInfrastructureSelector(false)}
                onSelect={selectInfrastructure}/>)
    }

    function getStringInput() {
        const {property, value} = props;
        const prefix = "parameters";
        const id = prefix + "-" + property.id;
        const inInfrastructure = InfrastructureAPI.infrastructure !== 'local';
        const noInfraSelectorButton = ["uri", "id", "description", "group"].includes(property.id);
        const icon = InfrastructureAPI.infrastructure === 'kubernetes' ? <KubernetesIcon/> : <DockerIcon/>
        const showInfraSelectorButton = inInfrastructure && !showEditor && !noInfraSelectorButton
        return <InputGroup>
            {showInfraSelectorButton  &&
                <Tooltip position="bottom-end" content={"Select from " + capitalize(InfrastructureAPI.infrastructure)}>
                    <Button variant="control" onClick={e => openInfrastructureSelector(property.id)}>
                        {icon}
                    </Button>
                </Tooltip>}
            {(!showEditor || property.format === "password") &&
                <TextInput
                    ref={ref}
                    className="text-field" isRequired
                    type={property.format && !showPassword ? "password" : "text"}
                    id={id} name={id}
                    value={value}
                    onChange={(e, value) => parametersChanged(property.id, value)}/>}
            {showEditor && property.format !== "password" &&
                <TextArea autoResize={true}
                          className="text-field" isRequired
                          type="text"
                          id={id} name={id}
                          value={value}
                          onChange={(e, value) => parametersChanged(property.id, value)}/>}
            {property.format !== "password" &&
                <Tooltip position="bottom-end" content={showEditor ? "Change to TextField" : "Change to Text Area"}>
                    <Button variant="control" onClick={e => setShowEditor(!showEditor)}>
                        {showEditor ? <CompressIcon/> : <ExpandIcon/>}
                    </Button>
                </Tooltip>
            }
            {property.format === "password" &&
                <Tooltip position="bottom-end" content={showPassword ? "Hide" : "Show"}>
                    <Button variant="control" onClick={e => setShowPassword(!showPassword)}>
                        {showPassword ? <ShowIcon/> : <HideIcon/>}
                    </Button>
                </Tooltip>
            }
        </InputGroup>
    }

    const property =  props.property;
    const value =  props.value;
    const prefix = "parameters";
    const id = prefix + "-" + property.id;
    return (
        <div>
            <FormGroup
                key={id}
                label={property.title}
                fieldId={id}
                isRequired={ props.required}
                labelIcon={
                    <Popover
                        position={"left"}
                        headerContent={property.title}
                        bodyContent={property.description}
                        footerContent={
                            <div>
                                {property.default !== undefined &&
                                    <div>Default: {property.default.toString()}</div>}
                                {property.example !== undefined && <div>Example: {property.example}</div>}
                            </div>
                        }>
                        <button type="button" aria-label="More info" onClick={e => e.preventDefault()}
                                className="pf-v5-c-form__group-label-help">
                            <HelpIcon />
                        </button>
                    </Popover>
                }>
                {property.type === 'string' && getStringInput()
                }
                {['integer', 'int', 'number'].includes(property.type) &&
                    <TextInput className="text-field" isRequired type='number' id={id} name={id} value={value}
                               onChange={(e, value) => parametersChanged(property.id, Number(value))}
                    />
                }
                {property.type === 'boolean' && <Switch
                    id={id} name={id}
                    value={value?.toString()}
                    aria-label={id}
                    isChecked={Boolean(value) === true}
                    onChange={e => parametersChanged(property.id, !Boolean(value))}/>
                }
            </FormGroup>
            {getInfrastructureSelectorModal()}
        </div>
    )
}