/*
Licensed to the Apache Software Foundation (ASF) under one or more
contributor license agreements.  See the NOTICE file distributed with
this work for additional information regarding copyright ownership.
The ASF licenses this file to You under the Apache License, Version 2.0
(the "License"); you may not use this file except in compliance with
the License.  You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package trait

import (
	"net/http"

	"github.com/pkg/errors"

	k8serrors "k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"k8s.io/apimachinery/pkg/types"

	ctrl "sigs.k8s.io/controller-runtime/pkg/client"

	v1 "github.com/apache/camel-k/pkg/apis/camel/v1"
	"github.com/apache/camel-k/pkg/util/patch"
)

// The deployer trait can be used to explicitly select the kind of high level resource that
// will deploy the integration.
//
// +camel-k:trait=deployer
type deployerTrait struct {
	BaseTrait `property:",squash"`
	// Allows to explicitly select the desired deployment kind between `deployment`, `cron-job` or `knative-service` when creating the resources for running the integration.
	Kind string `property:"kind" json:"kind,omitempty"`
}

var _ ControllerStrategySelector = &deployerTrait{}

var hasServerSideApply = true

func newDeployerTrait() Trait {
	return &deployerTrait{
		BaseTrait: NewBaseTrait("deployer", 900),
	}
}

func (t *deployerTrait) Configure(e *Environment) (bool, error) {
	return e.IntegrationInPhase(
		v1.IntegrationPhaseNone,
		v1.IntegrationPhaseWaitingForBindings,
		v1.IntegrationPhaseWaitingForPlatform,
		v1.IntegrationPhaseInitialization,
		v1.IntegrationPhaseBuildingKit,
		v1.IntegrationPhaseResolvingKit,
		v1.IntegrationPhaseDeploying,
		v1.IntegrationPhaseRunning,
	), nil
}

func (t *deployerTrait) Apply(e *Environment) error {
	switch e.Integration.Status.Phase {

	case v1.IntegrationPhaseBuildingKit, v1.IntegrationPhaseResolvingKit:
		if e.IntegrationKitInPhase(v1.IntegrationKitPhaseReady) {
			e.PostProcessors = append(e.PostProcessors, func(environment *Environment) error {
				// trigger integration deploy
				e.Integration.Status.Phase = v1.IntegrationPhaseDeploying
				return nil
			})
		}

	case v1.IntegrationPhaseNone, v1.IntegrationPhaseInitialization,
		v1.IntegrationPhaseWaitingForPlatform, v1.IntegrationPhaseWaitingForBindings,
		v1.IntegrationPhaseDeploying, v1.IntegrationPhaseRunning:
		// Register a post action that patches the resources generated by the traits
		e.PostActions = append(e.PostActions, func(env *Environment) error {
			for _, resource := range env.Resources.Items() {
				// We assume that server-side apply is enabled by default.
				// It is currently convoluted to check pro-actively whether server-side apply
				// is enabled. This is possible to fetch the OpenAPI endpoint, which returns
				// the entire server API document, then lookup the resource PATCH endpoint, and
				// check its list of accepted MIME types.
				// As a simpler solution, we fallback to client-side apply at the first
				// 415 error, and assume server-side apply is not available globally.
				if hasServerSideApply {
					if err := t.serverSideApply(env, resource); err == nil {
						continue
					} else if isIncompatibleServerError(err) {
						t.L.Info("Fallback to client-side apply to patch resources")
						hasServerSideApply = false
					}
				}
				if err := t.clientSideApply(env, resource); err != nil {
					return err
				}
			}
			return nil
		})
	}

	return nil
}

func (t *deployerTrait) serverSideApply(env *Environment, resource ctrl.Object) error {
	target, err := patch.PositiveApplyPatch(resource)
	if err != nil {
		return err
	}
	err = env.Client.Patch(env.C, target, ctrl.Apply, ctrl.ForceOwnership, ctrl.FieldOwner("camel-k-operator"))
	if err != nil {
		return errors.Wrapf(err, "error during apply resource: %v", resource)
	}
	return nil
}

func (t *deployerTrait) clientSideApply(env *Environment, resource ctrl.Object) error {
	err := env.Client.Create(env.C, resource)
	if err == nil {
		return nil
	} else if !k8serrors.IsAlreadyExists(err) {
		return errors.Wrapf(err, "error during create resource: %v", resource)
	}
	object := &unstructured.Unstructured{}
	object.SetNamespace(resource.GetNamespace())
	object.SetName(resource.GetName())
	object.SetGroupVersionKind(resource.GetObjectKind().GroupVersionKind())
	err = env.Client.Get(env.C, ctrl.ObjectKeyFromObject(object), object)
	if err != nil {
		return err
	}
	p, err := patch.PositiveMergePatch(object, resource)
	if err != nil {
		return err
	} else if len(p) == 0 {
		// Avoid triggering a patch request for nothing
		return nil
	}
	err = env.Client.Patch(env.C, resource, ctrl.RawPatch(types.MergePatchType, p))
	if err != nil {
		return errors.Wrapf(err, "error during patch resource: %v", resource)
	}
	return nil
}

func isIncompatibleServerError(err error) bool {
	// 415: Unsupported media type means we're talking to a server which doesn't
	// support server-side apply.
	if _, ok := err.(*k8serrors.StatusError); !ok {
		// Non-StatusError means the error isn't because the server is incompatible.
		return false
	}
	return err.(*k8serrors.StatusError).Status().Code == http.StatusUnsupportedMediaType
}

func (t *deployerTrait) SelectControllerStrategy(e *Environment) (*ControllerStrategy, error) {
	if t.Enabled != nil && !*t.Enabled {
		return nil, nil
	}
	if t.Kind != "" {
		strategy := ControllerStrategy(t.Kind)
		return &strategy, nil
	}
	return nil, nil
}

func (t *deployerTrait) ControllerStrategySelectorOrder() int {
	return 0
}

// IsPlatformTrait overrides base class method
func (t *deployerTrait) IsPlatformTrait() bool {
	return true
}

// RequiresIntegrationPlatform overrides base class method
func (t *deployerTrait) RequiresIntegrationPlatform() bool {
	return false
}
