/*
Licensed to the Apache Software Foundation (ASF) under one or more
contributor license agreements.  See the NOTICE file distributed with
this work for additional information regarding copyright ownership.
The ASF licenses this file to You under the Apache License, Version 2.0
(the "License"); you may not use this file except in compliance with
the License.  You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package cmd

import (
	"errors"
	"fmt"

	"github.com/apache/camel-k/pkg/cmd/local"
	"github.com/spf13/cobra"
)

func newCmdLocalInspect(localCmdOptions *LocalCmdOptions) (*cobra.Command, *localInspectCmdOptions) {
	options := localInspectCmdOptions{
		LocalCmdOptions: localCmdOptions,
	}

	cmd := cobra.Command{
		Use:   "inspect [files to inspect]",
		Short: "Generate dependencies list given integration files.",
		Long: `Output dependencies for a list of integration files. By default this command returns the
top level dependencies only. When --all-dependencies is enabled, the transitive dependencies
will be generated by calling Maven and then printed in the selected output format.`,
		Deprecated: "it may be removed in future versions",
		PreRunE:    decode(&options),
		RunE: func(cmd *cobra.Command, args []string) error {
			if err := options.validate(args); err != nil {
				return err
			}
			if err := options.init(); err != nil {
				return err
			}
			defer func() {
				if err := options.deinit(); err != nil {
					fmt.Fprintln(cmd.ErrOrStderr(), err.Error())
				}
			}()
			return options.run(cmd, args)
		},
		Annotations: map[string]string{
			offlineCommandLabel: "true",
		},
	}

	cmd.Flags().Bool("all-dependencies", false, "Enable computation of transitive dependencies.")
	cmd.Flags().StringP("output", "o", "", "Output format. One of: json|yaml")

	return &cmd, &options
}

type localInspectCmdOptions struct {
	*LocalCmdOptions
	AllDependencies bool   `mapstructure:"all-dependencies"`
	OutputFormat    string `mapstructure:"output"`
}

func (o *localInspectCmdOptions) validate(args []string) error {
	// If no source files have been provided there is nothing to inspect.
	if len(args) == 0 {
		return errors.New("no integration files have been provided")
	}

	if err := local.ValidateFiles(args); err != nil {
		return err
	}

	if err := local.ValidateDependencies(o.Dependencies); err != nil {
		return err
	}

	return nil
}

func (o *localInspectCmdOptions) init() error {
	return local.CreateMavenWorkingDirectory()
}

func (o *localInspectCmdOptions) run(cmd *cobra.Command, args []string) error {
	dependencies, err := local.GetDependencies(o.Context, cmd,
		args, o.Dependencies, o.MavenRepositories, o.AllDependencies)
	if err != nil {
		return err
	}

	if err = local.OutputDependencies(dependencies, o.OutputFormat, cmd); err != nil {
		return err
	}

	return nil
}

func (o *localInspectCmdOptions) deinit() error {
	return local.DeleteMavenWorkingDirectory()
}
