/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.file.remote.integration;

import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.mock.MockEndpoint;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;

/**
 * Unit test to verify remotefile sortby option.
 */
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
public class FromFtpRemoteFileSortByIgnoreCaseExpressionIT extends FtpServerTestSupport {

    private String getFtpUrl() {
        return "ftp://admin@localhost:{{ftp.server.port}}/sortbyignore?password=admin&delay=5000";
    }

    @Override
    @BeforeEach
    public void doPostSetup() throws Exception {
        prepareFtpServer();
    }

    @Test
    public void testSortFiles() throws Exception {
        context.addRoutes(new RouteBuilder() {
            @Override
            public void configure() {
                from(getFtpUrl() + "&sortBy=file:name").to("mock:result");
            }
        });
        context.start();

        MockEndpoint mock = getMockEndpoint("mock:result");
        mock.expectedBodiesReceived("Hello London", "Hello Copenhagen", "Hello Paris");

        MockEndpoint.assertIsSatisfied(context);
    }

    @Test
    public void testSortFilesNoCase() throws Exception {
        context.addRoutes(new RouteBuilder() {
            @Override
            public void configure() {
                from(getFtpUrl() + "&sortBy=ignoreCase:file:name").to("mock:nocase");
            }
        });
        context.start();

        MockEndpoint nocase = getMockEndpoint("mock:nocase");
        nocase.expectedBodiesReceived("Hello Copenhagen", "Hello London", "Hello Paris");

        MockEndpoint.assertIsSatisfied(context);
    }

    @Test
    public void testSortFilesNoCaseReverse() throws Exception {
        context.addRoutes(new RouteBuilder() {
            @Override
            public void configure() {
                from(getFtpUrl() + "&sortBy=reverse:ignoreCase:file:name").to("mock:nocasereverse");
            }
        });
        context.start();

        MockEndpoint nocasereverse = getMockEndpoint("mock:nocasereverse");
        nocasereverse.expectedBodiesReceived("Hello Paris", "Hello London", "Hello Copenhagen");

        MockEndpoint.assertIsSatisfied(context);
    }

    private void prepareFtpServer() {
        // prepares the FTP Server by creating files on the server that we want
        // to unit
        // test that we can pool
        sendFile(getFtpUrl(), "Hello Paris", "report-3.dat");
        sendFile(getFtpUrl(), "Hello London", "REPORT-2.txt");
        sendFile(getFtpUrl(), "Hello Copenhagen", "Report-1.xml");
    }
}
