/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.jira;

import org.apache.camel.RuntimeCamelException;
import org.apache.camel.spi.Metadata;
import org.apache.camel.spi.UriParam;
import org.apache.camel.spi.UriParams;

@UriParams
public class JiraConfiguration implements Cloneable {

    @UriParam(label = "security", secret = true)
    private String verificationCode;

    @UriParam(label = "security", secret = true)
    private String consumerKey;

    @UriParam(label = "security", secret = true)
    private String privateKey;

    @UriParam(label = "security", secret = true)
    private String accessToken;

    @UriParam
    @Metadata(required = true)
    private String jiraUrl;

    @UriParam(label = "security", secret = true)
    private String username;

    @UriParam(label = "security", secret = true)
    private String password;

    @UriParam(defaultValue = "6000")
    private Integer delay = 6000;

    public String getVerificationCode() {
        return verificationCode;
    }

    /**
     * (OAuth only) The verification code from Jira generated in the first step of the authorization proccess.
     */
    public void setVerificationCode(String verificationCode) {
        this.verificationCode = verificationCode;
    }

    public String getConsumerKey() {
        return consumerKey;
    }

    /**
     * (OAuth only) The consumer key from Jira settings.
     */
    public void setConsumerKey(String consumerKey) {
        this.consumerKey = consumerKey;
    }

    public String getPrivateKey() {
        return privateKey;
    }

    /**
     * (OAuth only) The private key generated by the client to encrypt the conversation to the server.
     */
    public void setPrivateKey(String privateKey) {
        this.privateKey = privateKey;
    }

    public String getAccessToken() {
        return accessToken;
    }

    /**
     * (OAuth or Personal Access Token authentication) The access token generated by the Jira server.
     */
    public void setAccessToken(String accessToken) {
        this.accessToken = accessToken;
    }

    public String getJiraUrl() {
        return jiraUrl;
    }

    /**
     * The Jira server url, example: http://my_jira.com:8081
     */
    public void setJiraUrl(String jiraUrl) {
        this.jiraUrl = jiraUrl;
    }

    public Integer getDelay() {
        return delay;
    }

    /**
     * Time in milliseconds to elapse for the next poll.
     *
     * @param delay Integer time in milliseconds
     */
    public void setDelay(Integer delay) {
        this.delay = delay;
    }

    public String getUsername() {
        return username;
    }

    /**
     * (Basic authentication only) The username to authenticate to the Jira server. Use only if OAuth is not enabled on
     * the Jira server. Do not set the username and OAuth token parameter, if they are both set, the username basic
     * authentication takes precedence.
     */
    public void setUsername(String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    /**
     * (Basic authentication only) The password or the API Token to authenticate to the Jira server. Use only if
     * username basic authentication is used.
     */
    public void setPassword(String password) {
        this.password = password;
    }

    public JiraConfiguration copy() {
        try {
            return (JiraConfiguration) super.clone();
        } catch (CloneNotSupportedException e) {
            throw new RuntimeCamelException(e);
        }
    }
}
