/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.jms;

import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.mock.MockEndpoint;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;

public class JmsInOnlyWithReplyToHeaderTest extends AbstractJMSTest {

    @Test
    public void testJmsInOnlyWithReplyToHeader() throws Exception {
        MockEndpoint mock = getMockEndpoint("mock:result");
        mock.expectedBodiesReceived("Hello World");
        mock.expectedHeaderReceived("JMSReplyTo", "ActiveMQQueue[JmsInOnlyWithReplyToHeaderTest.Reply]");

        template.send("activemq:queue:JmsInOnlyWithReplyToHeaderTest.Request?preserveMessageQos=true", exchange -> {
            exchange.getIn().setBody("World");
            exchange.getIn().setHeader("JMSReplyTo", "JmsInOnlyWithReplyToHeaderTest.Reply");
        });

        MockEndpoint.assertIsSatisfied(context);

        // reply is in JmsInOnlyWithReplyToHeaderTest.Reply queue so lets consume it
        String reply = consumer.receiveBody("activemq:queue:JmsInOnlyWithReplyToHeaderTest.Reply", 5000, String.class);
        assertEquals("Hello World", reply);
    }

    @Override
    protected String getComponentName() {
        return "activemq";
    }

    @Override
    protected RouteBuilder createRouteBuilder() {
        return new RouteBuilder() {
            @Override
            public void configure() {
                from("activemq:queue:JmsInOnlyWithReplyToHeaderTest.Request")
                        .transform(body().prepend("Hello "))
                        .to("mock:result");
            }
        };
    }
}
