/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.language.joor;

import java.util.Collection;

import org.apache.camel.RoutesBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.impl.DefaultCamelContext;
import org.apache.camel.model.ProcessDefinition;
import org.apache.camel.model.SetBodyDefinition;
import org.apache.camel.model.ToDefinition;
import org.apache.camel.spi.Resource;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

import static org.assertj.core.api.Assertions.assertThat;

public class JoorRoutesBuilderLoaderTest {

    @ParameterizedTest
    @ValueSource(strings = {
            "/routes/MyRoutes.java",
            "/routes/MyRoutesWithPackage.java",
            "/routes/MyRoutesWithPackageAndComment.java",
            "/routes/MyRoutesWithPackageAndLineComment.java"
    })
    void testLoadRoutes(String location) throws Exception {
        try (DefaultCamelContext context = new DefaultCamelContext()) {
            Resource resource = Resource.fromClasspath(JoorRoutesBuilderLoaderTest.class, location);
            Collection<RoutesBuilder> builders = context.getRoutesLoader().findRoutesBuilders(resource);

            assertThat(builders).hasSize(1);

            RouteBuilder builder = (RouteBuilder) builders.iterator().next();
            builder.setContext(context);
            builder.configure();

            assertThat(builder.getRouteCollection().getRoutes())
                    .hasSize(1)
                    .first()
                    .satisfies(rd -> {
                        assertThat(rd.getInput().getEndpointUri()).matches("timer:.*tick");
                        assertThat(rd.getOutputs().get(0)).isInstanceOf(ToDefinition.class);
                    });
        }
    }

    @Test
    void testLoadRoutesWithNestedClass() throws Exception {
        final String location = "/routes/MyRoutesWithNestedClass.java";

        try (DefaultCamelContext context = new DefaultCamelContext()) {
            Resource resource = Resource.fromClasspath(JoorRoutesBuilderLoaderTest.class, location);
            Collection<RoutesBuilder> builders = context.getRoutesLoader().findRoutesBuilders(resource);

            assertThat(builders).hasSize(1);

            RouteBuilder builder = (RouteBuilder) builders.iterator().next();
            builder.setContext(context);
            builder.configure();

            assertThat(builder.getRouteCollection().getRoutes())
                    .hasSize(1)
                    .first()
                    .satisfies(rd -> {
                        assertThat(rd.getInput().getEndpointUri()).matches("timer:.*tick");
                        assertThat(rd.getOutputs().get(0)).isInstanceOf(SetBodyDefinition.class);
                        assertThat(rd.getOutputs().get(1)).isInstanceOf(ProcessDefinition.class);
                        assertThat(rd.getOutputs().get(2)).isInstanceOf(ToDefinition.class);
                    });
        }
    }

    @Test
    void testLoadRoutesWithRestConfiguration() throws Exception {
        final String location = "/routes/MyRoutesWithRestConfiguration.java";

        try (DefaultCamelContext context = new DefaultCamelContext()) {
            Resource resource = Resource.fromClasspath(JoorRoutesBuilderLoaderTest.class, location);
            Collection<RoutesBuilder> builders = context.getRoutesLoader().findRoutesBuilders(resource);

            assertThat(builders).hasSize(1);

            RouteBuilder builder = (RouteBuilder) builders.iterator().next();
            builder.setContext(context);
            builder.configure();

            assertThat(builder.getRestConfiguration())
                    .hasFieldOrPropertyWithValue("component", "restlet");
        }
    }

    @Test
    void testLoadRoutesWithModel() throws Exception {
        final String location = "/routes/MyRoutesWithModel.java";

        try (DefaultCamelContext context = new DefaultCamelContext()) {
            Resource resource = Resource.fromClasspath(JoorRoutesBuilderLoaderTest.class, location);
            Collection<RoutesBuilder> builders = context.getRoutesLoader().findRoutesBuilders(resource);

            assertThat(builders).hasSize(1);

            RouteBuilder builder = (RouteBuilder) builders.iterator().next();
            builder.setContext(context);
            builder.configure();

            assertThat(builder.getRestCollection().getRests()).anySatisfy(rd -> {
                assertThat(rd.getVerbs())
                        .first()
                        .hasFieldOrPropertyWithValue("outType", MyUser.class.getName());
            });
        }
    }
}
