/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.brooklyn.rest.api;

import io.swagger.annotations.Api;
import org.apache.brooklyn.rest.domain.EntitySummary;
import org.apache.brooklyn.rest.domain.LocationSummary;
import org.apache.brooklyn.rest.domain.TaskSummary;

import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import java.util.List;
import java.util.Map;

@Path("/applications/{application}/entities")
@Api("Entities")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public interface EntityApi {

    @GET
    @ApiOperation(value = "Fetch the list of entities for a given application",
            response = org.apache.brooklyn.rest.domain.EntitySummary.class,
            responseContainer = "List")
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Application not found")
    })
    public List<EntitySummary> list(
            @ApiParam(value = "Application ID or name", required = true)
            @PathParam("application") final String application);

    @GET
    @Path("/{entity}")
    @ApiOperation(value = "Fetch details about a specific application entity",
            response = org.apache.brooklyn.rest.domain.EntitySummary.class)
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Application or entity missing")
    })
    public EntitySummary get(
            @ApiParam(value = "Application ID or name", required = true)
            @PathParam("application") String application,
            @ApiParam(value = "Entity ID or name", required = true)
            @PathParam("entity") String entity);

    // TODO rename as "/children" ?
    @GET
    @ApiOperation(value = "Fetch details about a specific application entity's children",
            response = org.apache.brooklyn.rest.domain.EntitySummary.class)
    @Path("/{entity}/children")
    public List<EntitySummary> getChildren(
            @PathParam("application") final String application,
            @PathParam("entity") final String entity);

    @POST
    @ApiOperation(value = "Add a child or children to this entity given a YAML spec",
            response = org.apache.brooklyn.rest.domain.TaskSummary.class)
    @Consumes({"application/x-yaml",
            // see http://stackoverflow.com/questions/332129/yaml-mime-type
            "text/yaml", "text/x-yaml", "application/yaml", MediaType.APPLICATION_JSON})
    @Path("/{entity}/children")
    public Response addChildren(
            @PathParam("application") final String application,
            @PathParam("entity") final String entity,

            @ApiParam(
                    name = "start",
                    value = "Whether to automatically start this child; if omitted, true for Startable entities")
            @QueryParam("start") final Boolean start,

            @ApiParam(name = "timeout", value = "Delay before server should respond with incomplete activity task, rather than completed task: " +
                    "'never' means block until complete; " +
                    "'0' means return task immediately; " +
                    "and e.g. '20ms' (the default) will wait 20ms for completed task information to be available", 
                    required = false, defaultValue = "20ms")
            @QueryParam("timeout") final String timeout,

            @ApiParam(
                    name = "childrenSpec",
                    value = "Entity spec in CAMP YAML format (including 'services' root element)",
                    required = true)
            String yaml);

    @GET
    @Path("/{entity}/activities")
    @ApiOperation(value = "Fetch list of tasks for this entity")
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Could not find application or entity")
    })
    public List<TaskSummary> listTasks(
            @ApiParam(value = "Application ID or name", required = true) @PathParam("application") String applicationId,
            @ApiParam(value = "Entity ID or name", required = true) @PathParam("entity") String entityId,
            @ApiParam(value = "Max number of tasks, or -1 for all (default 200)", required = false) 
            @QueryParam("limit") @DefaultValue("200") int limit,
            @ApiParam(value = "Whether to include subtasks recursively across different entities (default false)", required = false)
            @QueryParam("recurse") @DefaultValue("false") Boolean recurse);

    /** @deprecated since 0.12.0 use {@link #listTasks(String, String, int, Boolean)} */
    @GET
    @Path("/{entity}/activities/deprecated")
    @ApiOperation(value = "Fetch list of tasks for this entity")
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Could not find application or entity")
    })
    @Deprecated
    public List<TaskSummary> listTasks(
            @ApiParam(value = "Application ID or name", required = true) @PathParam("application") String applicationId,
            @ApiParam(value = "Entity ID or name", required = true) @PathParam("entity") String entityId);
        
    @GET
    @Path("/{entity}/activities/{task}")
    @ApiOperation(value = "Fetch task details", response = org.apache.brooklyn.rest.domain.TaskSummary.class)
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Could not find application, entity or task")
    })
    @Produces("text/json")
    public TaskSummary getTask(
            @ApiParam(value = "Application ID or name", required = true) @PathParam("application") final String application,
            @ApiParam(value = "Entity ID or name", required = true) @PathParam("entity") final String entityToken,
            @ApiParam(value = "Task ID", required = true) @PathParam("task") String taskId);

    @GET
    @ApiOperation(value = "Returns an icon for the entity, if defined")
    @Path("/{entity}/icon")
    public Response getIcon(
            @PathParam("application") final String application,
            @PathParam("entity") final String entity);

    @GET
    @Path("/{entity}/tags")
    @ApiOperation(value = "Fetch list of tags on this entity")
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Could not find application or entity")
    })
    public List<Object> listTags(
            @ApiParam(value = "Application ID or name", required = true) @PathParam("application") String applicationId,
            @ApiParam(value = "Entity ID or name", required = true) @PathParam("entity") String entityId);

    @POST
    @ApiOperation(
            value = "Rename an entity"
    )
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Undefined application or entity")
    })
    @Path("/{entity}/name")
    public Response rename(
            @ApiParam(value = "Application ID or name", required = true) @PathParam("application") final String applicationId, 
            @ApiParam(value = "Entity ID or name", required = true) @PathParam("entity") final String entityId, 
            @ApiParam(value = "New name for this entity", required = true) @QueryParam("name") final String name);

    @POST
    @ApiOperation(
            value = "Expunge an entity",
            response = org.apache.brooklyn.rest.domain.TaskSummary.class
    )
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Undefined application or entity")
    })
    @Path("/{entity}/expunge")
    public Response expunge(
            @ApiParam(value = "Application ID or name", required = true) @PathParam("application") final String applicationId, 
            @ApiParam(value = "Entity ID or name", required = true) @PathParam("entity") final String entityId, 
            @ApiParam(value = "Whether to gracefully release all resources", required = true) @QueryParam("release") final boolean release);

    @GET
    @Path("/{entity}/descendants")
    @ApiOperation(value = "Fetch entity info for all (or filtered) descendants",
            response = org.apache.brooklyn.rest.domain.EntitySummary.class)
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Application or entity missing")
    })
    public List<EntitySummary> getDescendants(
            @ApiParam(value = "Application ID or name", required = true)
            @PathParam("application") String application,
            @ApiParam(value = "Entity ID or name", required = true)
            @PathParam("entity") String entity,
            @ApiParam(value="Regular expression for an entity type which must be matched", required=false)
            @DefaultValue(".*")
            @QueryParam("typeRegex") String typeRegex);

    @GET
    @Path("/{entity}/descendants/sensor/{sensor}")
    @ApiOperation(value = "Fetch values of a given sensor for all (or filtered) descendants")
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Application or entity missing")
    })
    public Map<String,Object> getDescendantsSensor(
            @ApiParam(value = "Application ID or name", required = true)
            @PathParam("application") String application,
            @ApiParam(value = "Entity ID or name", required = true)
            @PathParam("entity") String entity,
            @ApiParam(value = "Sensor name", required = true)
            @PathParam("sensor") String sensor,
            @ApiParam(value="Regular expression applied to filter descendant entities based on their type", required=false)
            @DefaultValue(".*")
            @QueryParam("typeRegex") String typeRegex);

    @GET
    @Path("/{entity}/locations")
    @ApiOperation(value = "List the locations set on the entity")
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Application or entity missing")
    })
    public List<LocationSummary> getLocations(
            @ApiParam(value = "Application ID or name", required = true)
            @PathParam("application") String application,
            @ApiParam(value = "Entity ID or name", required = true)
            @PathParam("entity") String entity);

    // see http://stackoverflow.com/questions/332129/yaml-mime-type for "@Produces"
    @GET
    @Path("/{entity}/spec")
    @ApiOperation(value = "Get the YAML spec used to create the entity, if available")
    @Produces({"text/x-yaml", "application/x-yaml", "text/yaml", "text/plain", "application/yaml", MediaType.TEXT_PLAIN})
    @ApiResponses(value = {
            @ApiResponse(code = 404, message = "Application or entity missing")
    })
    public String getSpec(
            @ApiParam(value = "Application ID or name", required = true)
            @PathParam("application") String application,
            @ApiParam(value = "Entity ID or name", required = true)
            @PathParam("entity") String entity);
}
