/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.util.core.file;

import com.google.common.annotations.Beta;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.io.Files;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.zip.ZipOutputStream;
import org.apache.brooklyn.util.core.file.ArchiveUtils;
import org.apache.brooklyn.util.exceptions.Exceptions;
import org.apache.brooklyn.util.os.Os;
import org.apache.brooklyn.util.text.Strings;

@Beta
public class ArchiveBuilder {
    private final ArchiveUtils.ArchiveType type;
    private File archive;
    private Manifest manifest;
    private Multimap<String, File> entries = LinkedHashMultimap.create();

    public static ArchiveBuilder archive(String archive) {
        return new ArchiveBuilder(archive);
    }

    public static ArchiveBuilder zip() {
        return new ArchiveBuilder(ArchiveUtils.ArchiveType.ZIP);
    }

    public static ArchiveBuilder jar() {
        return new ArchiveBuilder(ArchiveUtils.ArchiveType.JAR);
    }

    private ArchiveBuilder() {
        this(ArchiveUtils.ArchiveType.ZIP);
    }

    private ArchiveBuilder(String filename) {
        this(ArchiveUtils.ArchiveType.of(filename));
        this.named(filename);
    }

    private ArchiveBuilder(ArchiveUtils.ArchiveType type) {
        Preconditions.checkNotNull((Object)((Object)type));
        Preconditions.checkArgument((boolean)ArchiveUtils.ArchiveType.ZIP_ARCHIVES.contains((Object)type));
        this.type = type;
        this.manifest = new Manifest();
    }

    public ArchiveBuilder named(String name) {
        Preconditions.checkNotNull((Object)name);
        String ext = Files.getFileExtension((String)name);
        if (ext.isEmpty()) {
            name = name + "." + this.type.toString();
        } else if (this.type != ArchiveUtils.ArchiveType.of(name)) {
            throw new IllegalArgumentException(String.format("Extension for '%s' did not match archive type of %s", new Object[]{ext, this.type}));
        }
        this.archive = new File(Os.tidyPath((String)name));
        return this;
    }

    public ArchiveBuilder named(File file) {
        Preconditions.checkNotNull((Object)file);
        return this.named(file.getPath());
    }

    public ArchiveBuilder manifest(Object key, Object value) {
        Preconditions.checkNotNull((Object)key, (Object)"key");
        Preconditions.checkNotNull((Object)value, (Object)"value");
        this.manifest.getMainAttributes().put(key, value);
        return this;
    }

    @Deprecated
    public ArchiveBuilder add(String filePath) {
        Preconditions.checkNotNull((Object)filePath, (Object)"filePath");
        return this.add(new File(Os.tidyPath((String)filePath)));
    }

    @Deprecated
    public ArchiveBuilder add(File file) {
        Preconditions.checkNotNull((Object)file, (Object)"file");
        String filePath = Os.tidyPath((String)file.getPath());
        if (file.isAbsolute() || filePath.startsWith("../")) {
            return this.entry(Os.mergePaths((String[])new String[]{".", file.getName()}), file);
        }
        return this.entry(Os.mergePaths((String[])new String[]{".", filePath}), file);
    }

    public ArchiveBuilder addFromLocalBaseDir(File baseDir, String fileSubPath) {
        Preconditions.checkNotNull((Object)baseDir, (Object)"baseDir");
        Preconditions.checkNotNull((Object)fileSubPath, (Object)"filePath");
        return this.entry(fileSubPath, Os.mergePaths((String[])new String[]{baseDir.getPath(), fileSubPath}));
    }

    @Deprecated
    public ArchiveBuilder addFromLocalBaseDir(String baseDir, String fileSubPath) {
        return this.addFromLocalBaseDir(new File(baseDir), fileSubPath);
    }

    @Deprecated
    public ArchiveBuilder add(String baseDir, String fileSubPath) {
        return this.addFromLocalBaseDir(baseDir, fileSubPath);
    }

    public ArchiveBuilder addAt(File file, String archiveParentDir) {
        Preconditions.checkNotNull((Object)archiveParentDir, (Object)"archiveParentDir");
        Preconditions.checkNotNull((Object)file, (Object)"file");
        return this.entry(Os.mergePaths((String[])new String[]{archiveParentDir, file.getName()}), file);
    }

    public ArchiveBuilder addAtRoot(File file) {
        return this.addAt(file, "");
    }

    @Deprecated
    public ArchiveBuilder addDir(String dirName) {
        Preconditions.checkNotNull((Object)dirName, (Object)"dirName");
        return this.addDir(new File(Os.tidyPath((String)dirName)));
    }

    public ArchiveBuilder addDirContentsAt(File dir, String archiveParentDir) {
        Preconditions.checkNotNull((Object)dir, (Object)"dir");
        if (!dir.isDirectory()) {
            throw new IllegalArgumentException(dir + " is not a directory; cannot add contents to archive");
        }
        return this.entry(archiveParentDir, dir);
    }

    public ArchiveBuilder addDirContentsAtRoot(File dir) {
        return this.addDirContentsAt(dir, "");
    }

    @Deprecated
    public ArchiveBuilder addDir(File dir) {
        return this.addDirContentsAt(dir, ".");
    }

    @Deprecated
    public ArchiveBuilder add(Iterable<String> files) {
        Preconditions.checkNotNull(files, (Object)"files");
        for (String filePath : files) {
            this.add(filePath);
        }
        return this;
    }

    @Deprecated
    public ArchiveBuilder add(String baseDir, Iterable<String> files) {
        Preconditions.checkNotNull((Object)baseDir, (Object)"baseDir");
        Preconditions.checkNotNull(files, (Object)"files");
        for (String filePath : files) {
            this.add(baseDir, filePath);
        }
        return this;
    }

    public ArchiveBuilder entry(String entryPath, String filePath) {
        Preconditions.checkNotNull((Object)entryPath, (Object)"entryPath");
        Preconditions.checkNotNull((Object)filePath, (Object)"filePath");
        return this.entry(entryPath, new File(filePath));
    }

    public ArchiveBuilder entry(String entryPath, File file) {
        Preconditions.checkNotNull((Object)entryPath, (Object)"entryPath");
        Preconditions.checkNotNull((Object)file, (Object)"file");
        this.entries.put((Object)entryPath, (Object)file);
        return this;
    }

    public ArchiveBuilder entries(Map<String, File> entries) {
        Preconditions.checkNotNull(entries, (Object)"entries");
        for (Map.Entry<String, File> entry : entries.entrySet()) {
            this.entries.put((Object)entry.getKey(), (Object)entry.getValue());
        }
        return this;
    }

    public void stream(OutputStream output) {
        try {
            ZipOutputStream target;
            if (this.type == ArchiveUtils.ArchiveType.ZIP) {
                target = new ZipOutputStream(output);
            } else {
                this.manifest(Attributes.Name.MANIFEST_VERSION, "1.0");
                target = new JarOutputStream(output, this.manifest);
            }
            for (String entry : this.entries.keySet()) {
                this.addToArchive(entry, this.entries.get((Object)entry), target);
            }
            target.close();
        }
        catch (IOException ioe) {
            throw Exceptions.propagate((Throwable)ioe);
        }
    }

    public File create(String archiveFile) {
        return this.named(archiveFile).create();
    }

    public File create() {
        if (this.archive == null) {
            File temp = Os.newTempFile((String)"brooklyn-archive", (String)this.type.toString());
            temp.deleteOnExit();
            this.named(temp);
        }
        try {
            FileOutputStream output = new FileOutputStream(this.archive);
            this.stream(output);
            ((OutputStream)output).close();
        }
        catch (IOException ioe) {
            throw Exceptions.propagate((Throwable)ioe);
        }
        return this.archive;
    }

    private void addToArchive(String path, Iterable<File> sources, ZipOutputStream target) throws IOException {
        int size = Iterables.size(sources);
        if (size == 0) {
            return;
        }
        boolean isDirectory = size > 1 ? true : ((File)Iterables.getOnlyElement(sources)).isDirectory();
        String name = path.replace("\\", "/");
        if (isDirectory) {
            if ((name = Strings.removeAllFromEnd((String)name, (String[])new String[]{"/"})).length() > 0) {
                name = name + "/";
                JarEntry entry = new JarEntry(name);
                long lastModified = -1L;
                for (File source : sources) {
                    if (source.lastModified() <= lastModified) continue;
                    lastModified = source.lastModified();
                }
                entry.setTime(lastModified);
                target.putNextEntry(entry);
                target.closeEntry();
            }
            for (File source : sources) {
                if (!source.isDirectory()) {
                    throw new IllegalStateException("Cannot add multiple items at a path in archive unless they are directories: " + sources + " at " + path + " is not valid.");
                }
                Iterable children = Files.fileTreeTraverser().children((Object)source);
                for (File child : children) {
                    this.addToArchive(Os.mergePaths((String[])new String[]{name, child.getName()}), Collections.singleton(child), target);
                }
            }
            return;
        }
        File source = (File)Iterables.getOnlyElement(sources);
        JarEntry entry = new JarEntry(name);
        entry.setTime(source.lastModified());
        target.putNextEntry(entry);
        Files.asByteSource((File)source).copyTo((OutputStream)target);
        target.closeEntry();
    }
}

