/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.replication;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.NavigableSet;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import org.apache.bookkeeper.bookie.BookieThread;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.BookKeeperAdmin;
import org.apache.bookkeeper.client.LedgerChecker;
import org.apache.bookkeeper.client.LedgerFragment;
import org.apache.bookkeeper.client.LedgerHandle;
import org.apache.bookkeeper.client.api.LedgerMetadata;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerUnderreplicationManager;
import org.apache.bookkeeper.net.BookieId;
import org.apache.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.bookkeeper.replication.Auditor;
import org.apache.bookkeeper.replication.ReplicationException;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.stats.annotations.StatsDoc;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@StatsDoc(name="replication_worker", help="replication worker related stats")
public class ReplicationWorker
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicationWorker.class);
    private static final int REPLICATED_FAILED_LEDGERS_MAXSIZE = 2000;
    public static final int NUM_OF_EXPONENTIAL_BACKOFF_RETRIALS = 5;
    private final LedgerUnderreplicationManager underreplicationManager;
    private final ServerConfiguration conf;
    private volatile boolean workerRunning = false;
    private final BookKeeperAdmin admin;
    private final LedgerChecker ledgerChecker;
    private final BookKeeper bkc;
    private final boolean ownBkc;
    private final Thread workerThread;
    private final long rwRereplicateBackoffMs;
    private final long openLedgerRereplicationGracePeriod;
    private final Timer pendingReplicationTimer;
    private final long lockReleaseOfFailedLedgerGracePeriod;
    private final long baseBackoffForLockReleaseOfFailedLedger;
    private final BiConsumer<Long, Long> onReadEntryFailureCallback;
    private final StatsLogger statsLogger;
    @StatsDoc(name="exceptions", help="replication related exceptions")
    private final StatsLogger exceptionLogger;
    @StatsDoc(name="rereplicate", help="operation stats of re-replicating ledgers")
    private final OpStatsLogger rereplicateOpStats;
    @StatsDoc(name="NUM_FULL_OR_PARTIAL_LEDGERS_REPLICATED", help="the number of ledgers re-replicated")
    private final Counter numLedgersReplicated;
    @StatsDoc(name="NUM_DEFER_LEDGER_LOCK_RELEASE_OF_FAILED_LEDGER", help="the number of defer-ledger-lock-releases of failed ledgers")
    private final Counter numDeferLedgerLockReleaseOfFailedLedger;
    @StatsDoc(name="NUM_ENTRIES_UNABLE_TO_READ_FOR_REPLICATION", help="the number of entries ReplicationWorker unable to read")
    private final Counter numEntriesUnableToReadForReplication;
    private final Map<String, Counter> exceptionCounters;
    final LoadingCache<Long, AtomicInteger> replicationFailedLedgers;
    final LoadingCache<Long, ConcurrentSkipListSet<Long>> unableToReadEntriesForReplication;

    public ReplicationWorker(ServerConfiguration conf) throws ReplicationException.CompatibilityException, KeeperException, InterruptedException, IOException {
        this(conf, (StatsLogger)NullStatsLogger.INSTANCE);
    }

    public ReplicationWorker(ServerConfiguration conf, StatsLogger statsLogger) throws ReplicationException.CompatibilityException, KeeperException, InterruptedException, IOException {
        this(conf, Auditor.createBookKeeperClient(conf), true, statsLogger);
    }

    ReplicationWorker(ServerConfiguration conf, BookKeeper bkc, boolean ownBkc, StatsLogger statsLogger) throws ReplicationException.CompatibilityException, KeeperException, InterruptedException, IOException {
        this.conf = conf;
        this.bkc = bkc;
        this.ownBkc = ownBkc;
        this.underreplicationManager = bkc.getLedgerManagerFactory().newLedgerUnderreplicationManager();
        this.admin = new BookKeeperAdmin(bkc, statsLogger, new ClientConfiguration(conf));
        this.ledgerChecker = new LedgerChecker(bkc);
        this.workerThread = new BookieThread(this, "ReplicationWorker");
        this.openLedgerRereplicationGracePeriod = conf.getOpenLedgerRereplicationGracePeriod();
        this.lockReleaseOfFailedLedgerGracePeriod = conf.getLockReleaseOfFailedLedgerGracePeriod();
        this.baseBackoffForLockReleaseOfFailedLedger = this.lockReleaseOfFailedLedgerGracePeriod / (long)Math.pow(2.0, 5.0);
        this.rwRereplicateBackoffMs = conf.getRwRereplicateBackoffMs();
        this.pendingReplicationTimer = new Timer("PendingReplicationTimer");
        this.replicationFailedLedgers = CacheBuilder.newBuilder().maximumSize(2000L).build((CacheLoader)new CacheLoader<Long, AtomicInteger>(){

            public AtomicInteger load(Long key) throws Exception {
                return new AtomicInteger();
            }
        });
        this.unableToReadEntriesForReplication = CacheBuilder.newBuilder().maximumSize(2000L).build((CacheLoader)new CacheLoader<Long, ConcurrentSkipListSet<Long>>(){

            public ConcurrentSkipListSet<Long> load(Long key) throws Exception {
                return new ConcurrentSkipListSet<Long>();
            }
        });
        this.statsLogger = statsLogger;
        this.exceptionLogger = statsLogger.scope("exceptions");
        this.rereplicateOpStats = this.statsLogger.getOpStatsLogger("rereplicate");
        this.numLedgersReplicated = this.statsLogger.getCounter("NUM_FULL_OR_PARTIAL_LEDGERS_REPLICATED");
        this.numDeferLedgerLockReleaseOfFailedLedger = this.statsLogger.getCounter("NUM_DEFER_LEDGER_LOCK_RELEASE_OF_FAILED_LEDGER");
        this.numEntriesUnableToReadForReplication = this.statsLogger.getCounter("NUM_ENTRIES_UNABLE_TO_READ_FOR_REPLICATION");
        this.exceptionCounters = new HashMap<String, Counter>();
        this.onReadEntryFailureCallback = (ledgerid, entryid) -> {
            this.numEntriesUnableToReadForReplication.inc();
            ((ConcurrentSkipListSet)this.unableToReadEntriesForReplication.getUnchecked(ledgerid)).add(entryid);
        };
    }

    public void start() {
        this.workerThread.start();
    }

    @Override
    public void run() {
        this.workerRunning = true;
        while (this.workerRunning) {
            try {
                if (this.rereplicate()) continue;
                LOG.warn("failed while replicating fragments");
                ReplicationWorker.waitBackOffTime(this.rwRereplicateBackoffMs);
            }
            catch (InterruptedException e) {
                LOG.error("InterruptedException while replicating fragments", (Throwable)e);
                this.shutdown();
                Thread.currentThread().interrupt();
                return;
            }
            catch (BKException e) {
                LOG.error("BKException while replicating fragments", (Throwable)e);
                ReplicationWorker.waitBackOffTime(this.rwRereplicateBackoffMs);
            }
            catch (ReplicationException.UnavailableException e) {
                LOG.error("UnavailableException while replicating fragments", (Throwable)e);
                ReplicationWorker.waitBackOffTime(this.rwRereplicateBackoffMs);
                if (!Thread.currentThread().isInterrupted()) continue;
                LOG.error("Interrupted  while replicating fragments");
                this.shutdown();
                return;
            }
        }
        LOG.info("ReplicationWorker exited loop!");
    }

    private static void waitBackOffTime(long backoffMs) {
        try {
            Thread.sleep(backoffMs);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean rereplicate() throws InterruptedException, BKException, ReplicationException.UnavailableException {
        long ledgerIdToReplicate = this.underreplicationManager.getLedgerToRereplicate();
        Stopwatch stopwatch = Stopwatch.createStarted();
        boolean success = false;
        try {
            success = this.rereplicate(ledgerIdToReplicate);
        }
        finally {
            long latencyMillis = stopwatch.stop().elapsed(TimeUnit.MILLISECONDS);
            if (success) {
                this.rereplicateOpStats.registerSuccessfulEvent(latencyMillis, TimeUnit.MILLISECONDS);
            } else {
                this.rereplicateOpStats.registerFailedEvent(latencyMillis, TimeUnit.MILLISECONDS);
            }
        }
        return success;
    }

    private void logBKExceptionAndReleaseLedger(BKException e, long ledgerIdToReplicate) throws ReplicationException.UnavailableException {
        LOG.info("{} while rereplicating ledger {}. Enough Bookies might not have available So, no harm to continue", (Object)e.getClass().getSimpleName(), (Object)ledgerIdToReplicate);
        this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
        this.getExceptionCounter(e.getClass().getSimpleName()).inc();
    }

    private boolean tryReadingFaultyEntries(LedgerHandle lh, LedgerFragment ledgerFragment) {
        long ledgerId = lh.getId();
        ConcurrentSkipListSet entriesUnableToReadForThisLedger = (ConcurrentSkipListSet)this.unableToReadEntriesForReplication.getIfPresent((Object)ledgerId);
        if (entriesUnableToReadForThisLedger == null) {
            return true;
        }
        long firstEntryIdOfFragment = ledgerFragment.getFirstEntryId();
        long lastEntryIdOfFragment = ledgerFragment.getLastKnownEntryId();
        NavigableSet<Long> entriesOfThisFragmentUnableToRead = entriesUnableToReadForThisLedger.subSet(firstEntryIdOfFragment, true, lastEntryIdOfFragment, true);
        if (entriesOfThisFragmentUnableToRead.isEmpty()) {
            return true;
        }
        CountDownLatch multiReadComplete = new CountDownLatch(1);
        AtomicInteger numOfResponsesToWaitFor = new AtomicInteger(entriesOfThisFragmentUnableToRead.size());
        AtomicInteger returnRCValue = new AtomicInteger(0);
        for (long entryIdToRead : entriesOfThisFragmentUnableToRead) {
            if (multiReadComplete.getCount() == 0L) break;
            lh.asyncReadEntries(entryIdToRead, entryIdToRead, (rc, ledHan, seq, ctx) -> {
                long thisEntryId = (Long)ctx;
                if (rc == 0) {
                    entriesUnableToReadForThisLedger.remove(thisEntryId);
                    if (numOfResponsesToWaitFor.decrementAndGet() == 0) {
                        multiReadComplete.countDown();
                    }
                } else {
                    LOG.error("Received error: {} while trying to read entry: {} of ledger: {} in ReplicationWorker", new Object[]{rc, entryIdToRead, ledgerId});
                    returnRCValue.compareAndSet(0, rc);
                    multiReadComplete.countDown();
                }
            }, entryIdToRead);
        }
        try {
            multiReadComplete.await();
        }
        catch (InterruptedException e) {
            LOG.error("Got interrupted exception while trying to read entries", (Throwable)e);
            Thread.currentThread().interrupt();
            return false;
        }
        return returnRCValue.get() == 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     */
    @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_WOULD_HAVE_BEEN_A_NPE"})
    private boolean rereplicate(long ledgerIdToReplicate) throws InterruptedException, BKException, ReplicationException.UnavailableException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Going to replicate the fragments of the ledger: {}", (Object)ledgerIdToReplicate);
        }
        boolean deferLedgerLockRelease = false;
        try {
            boolean bl;
            Throwable throwable;
            LedgerHandle lh;
            block57: {
                block58: {
                    block54: {
                        boolean bl2;
                        block55: {
                            block56: {
                                Set<LedgerFragment> fragments;
                                block51: {
                                    boolean bl3;
                                    block52: {
                                        block53: {
                                            lh = this.admin.openLedgerNoRecovery(ledgerIdToReplicate);
                                            throwable = null;
                                            fragments = this.getUnderreplicatedFragments(lh, this.conf.getAuditorLedgerVerificationPercentage());
                                            if (LOG.isDebugEnabled()) {
                                                LOG.debug("Founds fragments {} for replication from ledger: {}", fragments, (Object)ledgerIdToReplicate);
                                            }
                                            boolean foundOpenFragments = false;
                                            long numFragsReplicated = 0L;
                                            for (LedgerFragment ledgerFragment : fragments) {
                                                if (!ledgerFragment.isClosed()) {
                                                    foundOpenFragments = true;
                                                    continue;
                                                }
                                                if (!this.tryReadingFaultyEntries(lh, ledgerFragment)) {
                                                    LOG.error("Failed to read faulty entries, so giving up replicating ledgerFragment {}", (Object)ledgerFragment);
                                                    continue;
                                                }
                                                try {
                                                    this.admin.replicateLedgerFragment(lh, ledgerFragment, this.onReadEntryFailureCallback);
                                                    ++numFragsReplicated;
                                                }
                                                catch (BKException.BKBookieHandleNotAvailableException e) {
                                                    LOG.warn("BKBookieHandleNotAvailableException while replicating the fragment", (Throwable)e);
                                                }
                                                catch (BKException.BKLedgerRecoveryException e) {
                                                    LOG.warn("BKLedgerRecoveryException while replicating the fragment", (Throwable)e);
                                                }
                                                catch (BKException.BKNotEnoughBookiesException e) {
                                                    LOG.warn("BKNotEnoughBookiesException while replicating the fragment", (Throwable)e);
                                                }
                                            }
                                            if (numFragsReplicated > 0L) {
                                                this.numLedgersReplicated.inc();
                                            }
                                            if (!foundOpenFragments && !this.isLastSegmentOpenAndMissingBookies(lh)) break block51;
                                            deferLedgerLockRelease = true;
                                            this.deferLedgerLockRelease(ledgerIdToReplicate);
                                            bl3 = false;
                                            if (lh == null) break block52;
                                            if (throwable == null) break block53;
                                            try {
                                                lh.close();
                                            }
                                            catch (Throwable ledgerFragment) {
                                                throwable.addSuppressed(ledgerFragment);
                                            }
                                            break block52;
                                        }
                                        lh.close();
                                    }
                                    return bl3;
                                }
                                fragments = this.getUnderreplicatedFragments(lh, this.conf.getAuditorLedgerVerificationPercentage());
                                if (fragments.size() != 0) break block54;
                                LOG.info("Ledger replicated successfully. ledger id is: " + ledgerIdToReplicate);
                                this.underreplicationManager.markLedgerReplicated(ledgerIdToReplicate);
                                bl2 = true;
                                if (lh == null) break block55;
                                if (throwable == null) break block56;
                                try {
                                    lh.close();
                                }
                                catch (Throwable e) {
                                    throwable.addSuppressed(e);
                                }
                                break block55;
                            }
                            lh.close();
                        }
                        return bl2;
                    }
                    deferLedgerLockRelease = true;
                    this.deferLedgerLockReleaseOfFailedLedger(ledgerIdToReplicate);
                    this.numDeferLedgerLockReleaseOfFailedLedger.inc();
                    bl = false;
                    if (lh == null) break block57;
                    if (throwable == null) break block58;
                    try {
                        lh.close();
                    }
                    catch (Throwable e) {
                        throwable.addSuppressed(e);
                    }
                    break block57;
                }
                lh.close();
            }
            return bl;
            catch (Throwable fragments) {
                try {
                    try {
                        throwable = fragments;
                        throw fragments;
                    }
                    catch (Throwable throwable2) {
                        if (lh != null) {
                            if (throwable != null) {
                                try {
                                    lh.close();
                                }
                                catch (Throwable throwable3) {
                                    throwable.addSuppressed(throwable3);
                                }
                            } else {
                                lh.close();
                            }
                        }
                        throw throwable2;
                    }
                }
                catch (BKException.BKNoSuchLedgerExistsOnMetadataServerException e) {
                    LOG.info("BKNoSuchLedgerExistsOnMetadataServerException while opening ledger {} for replication. Other clients might have deleted the ledger. So, no harm to continue", (Object)ledgerIdToReplicate);
                    this.underreplicationManager.markLedgerReplicated(ledgerIdToReplicate);
                    this.getExceptionCounter("BKNoSuchLedgerExistsOnMetadataServerException").inc();
                    boolean bl4 = false;
                    return bl4;
                }
                catch (BKException.BKNotEnoughBookiesException e) {
                    this.logBKExceptionAndReleaseLedger(e, ledgerIdToReplicate);
                    throw e;
                }
                catch (BKException e) {
                    this.logBKExceptionAndReleaseLedger(e, ledgerIdToReplicate);
                    boolean bl5 = false;
                    return bl5;
                }
            }
        }
        finally {
            if (!deferLedgerLockRelease) {
                try {
                    this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
                }
                catch (ReplicationException.UnavailableException e) {
                    LOG.error("UnavailableException while releasing the underreplicated lock for ledger {}:", (Object)ledgerIdToReplicate, (Object)e);
                    this.shutdown();
                }
            }
        }
    }

    private boolean isLastSegmentOpenAndMissingBookies(LedgerHandle lh) throws BKException {
        LedgerMetadata md = this.admin.getLedgerMetadata(lh);
        if (md.isClosed()) {
            return false;
        }
        NavigableMap<Long, ? extends List<BookieId>> ensembles = this.admin.getLedgerMetadata(lh).getAllEnsembles();
        List finalEnsemble = (List)ensembles.get(ensembles.lastKey());
        Collection<BookieId> available = this.admin.getAvailableBookies();
        for (BookieId b : finalEnsemble) {
            if (available.contains(b)) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Bookie {} is missing from the list of Available Bookies. ledger {}:ensemble {}.", new Object[]{b, lh.getId(), finalEnsemble});
            }
            return true;
        }
        return false;
    }

    private Set<LedgerFragment> getUnderreplicatedFragments(LedgerHandle lh, Long ledgerVerificationPercentage) throws InterruptedException {
        CheckerCallback checkerCb = new CheckerCallback();
        this.ledgerChecker.checkLedger(lh, checkerCb, ledgerVerificationPercentage);
        Set<LedgerFragment> fragments = checkerCb.waitAndGetResult();
        return fragments;
    }

    void scheduleTaskWithDelay(TimerTask timerTask, long delayPeriod) {
        this.pendingReplicationTimer.schedule(timerTask, delayPeriod);
    }

    private void deferLedgerLockRelease(final long ledgerId) {
        long gracePeriod = this.openLedgerRereplicationGracePeriod;
        TimerTask timerTask = new TimerTask(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             * Enabled aggressive block sorting
             * Enabled unnecessary exception pruning
             * Enabled aggressive exception aggregation
             */
            @Override
            public void run() {
                boolean isRecoveryOpen = false;
                LedgerHandle lh = null;
                try {
                    LedgerFragment fragment;
                    lh = ReplicationWorker.this.admin.openLedgerNoRecovery(ledgerId);
                    if (ReplicationWorker.this.isLastSegmentOpenAndMissingBookies(lh)) {
                        lh.close();
                        LOG.warn("Missing bookie(s) from last segment. Opening Ledger{} for Recovery.", (Object)ledgerId);
                        lh = ReplicationWorker.this.admin.openLedger(ledgerId);
                        isRecoveryOpen = true;
                    }
                    if (isRecoveryOpen) return;
                    Set fragments = ReplicationWorker.this.getUnderreplicatedFragments(lh, ReplicationWorker.this.conf.getAuditorLedgerVerificationPercentage());
                    Iterator iterator = fragments.iterator();
                    do {
                        if (!iterator.hasNext()) return;
                    } while ((fragment = (LedgerFragment)iterator.next()).isClosed());
                    lh.close();
                    LOG.warn("Open Fragment{}. Opening Ledger{} for Recovery.", fragment.getEnsemble(), (Object)ledgerId);
                    lh = ReplicationWorker.this.admin.openLedger(ledgerId);
                    isRecoveryOpen = true;
                    return;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    LOG.info("InterruptedException while fencing the ledger {} for rereplication of postponed ledgers", (Object)ledgerId, (Object)e);
                    return;
                }
                catch (BKException.BKNoSuchLedgerExistsOnMetadataServerException bknsle) {
                    if (!LOG.isDebugEnabled()) return;
                    LOG.debug("Ledger {} was deleted, safe to continue", (Object)ledgerId, (Object)bknsle);
                    return;
                }
                catch (BKException e) {
                    LOG.error("BKException while fencing the ledger {} for rereplication of postponed ledgers", (Object)ledgerId, (Object)e);
                    return;
                }
                finally {
                    try {
                        if (lh != null) {
                            lh.close();
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        LOG.info("InterruptedException while closing ledger {}", (Object)ledgerId, (Object)e);
                    }
                    catch (BKException e) {
                        LOG.warn("BKException while closing ledger {} ", (Object)ledgerId, (Object)e);
                    }
                    finally {
                        try {
                            ReplicationWorker.this.underreplicationManager.releaseUnderreplicatedLedger(ledgerId);
                        }
                        catch (ReplicationException.UnavailableException e) {
                            LOG.error("UnavailableException while replicating fragments of ledger {}", (Object)ledgerId, (Object)e);
                            ReplicationWorker.this.shutdown();
                        }
                    }
                }
            }
        };
        this.scheduleTaskWithDelay(timerTask, gracePeriod);
    }

    private void deferLedgerLockReleaseOfFailedLedger(final long ledgerId) {
        int numOfTimesFailedSoFar = ((AtomicInteger)this.replicationFailedLedgers.getUnchecked((Object)ledgerId)).getAndIncrement();
        long delayOfLedgerLockReleaseInMSecs = numOfTimesFailedSoFar >= 5 ? this.lockReleaseOfFailedLedgerGracePeriod : this.baseBackoffForLockReleaseOfFailedLedger * (long)((int)Math.pow(2.0, numOfTimesFailedSoFar));
        LOG.error("ReplicationWorker failed to replicate Ledger : {} for {} number of times, so deferring the ledger lock release by {} msecs", new Object[]{ledgerId, numOfTimesFailedSoFar, delayOfLedgerLockReleaseInMSecs});
        TimerTask timerTask = new TimerTask(){

            @Override
            public void run() {
                try {
                    ReplicationWorker.this.underreplicationManager.releaseUnderreplicatedLedger(ledgerId);
                }
                catch (ReplicationException.UnavailableException e) {
                    LOG.error("UnavailableException while replicating fragments of ledger {}", (Object)ledgerId, (Object)e);
                    ReplicationWorker.this.shutdown();
                }
            }
        };
        this.scheduleTaskWithDelay(timerTask, delayOfLedgerLockReleaseInMSecs);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        LOG.info("Shutting down replication worker");
        ReplicationWorker replicationWorker = this;
        synchronized (replicationWorker) {
            if (!this.workerRunning) {
                return;
            }
            this.workerRunning = false;
        }
        LOG.info("Shutting down ReplicationWorker");
        this.pendingReplicationTimer.cancel();
        try {
            this.workerThread.interrupt();
            this.workerThread.join();
        }
        catch (InterruptedException e) {
            LOG.error("Interrupted during shutting down replication worker : ", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        if (this.ownBkc) {
            try {
                this.bkc.close();
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while closing the Bookie client", (Throwable)e);
                Thread.currentThread().interrupt();
            }
            catch (BKException e) {
                LOG.warn("Exception while closing the Bookie client", (Throwable)e);
            }
        }
        try {
            this.underreplicationManager.close();
        }
        catch (ReplicationException.UnavailableException e) {
            LOG.warn("Exception while closing the ZkLedgerUnderrepliationManager", (Throwable)e);
        }
    }

    @VisibleForTesting
    public boolean isRunning() {
        return this.workerRunning && this.workerThread.isAlive();
    }

    private Counter getExceptionCounter(String name) {
        Counter counter = this.exceptionCounters.get(name);
        if (counter == null) {
            counter = this.exceptionLogger.getCounter(name);
            this.exceptionCounters.put(name, counter);
        }
        return counter;
    }

    private static class CheckerCallback
    implements BookkeeperInternalCallbacks.GenericCallback<Set<LedgerFragment>> {
        private Set<LedgerFragment> result = null;
        private CountDownLatch latch = new CountDownLatch(1);

        private CheckerCallback() {
        }

        @Override
        public void operationComplete(int rc, Set<LedgerFragment> result) {
            this.result = result;
            this.latch.countDown();
        }

        Set<LedgerFragment> waitAndGetResult() throws InterruptedException {
            this.latch.await();
            return this.result;
        }
    }
}

