/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.tools.perf.journal;

import com.beust.jcommander.Parameter;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.RateLimiter;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.Unpooled;
import io.netty.util.ReferenceCountUtil;
import java.io.File;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.LongAdder;
import org.HdrHistogram.Histogram;
import org.HdrHistogram.Recorder;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.CheckpointSource;
import org.apache.bookkeeper.bookie.Journal;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.common.allocator.ByteBufAllocatorBuilder;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.Stats;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.stats.prometheus.PrometheusMetricsProvider;
import org.apache.bookkeeper.tools.framework.CliFlags;
import org.apache.bookkeeper.tools.perf.utils.PaddingDecimalFormat;
import org.apache.bookkeeper.util.DiskChecker;
import org.apache.commons.configuration.CompositeConfiguration;
import org.apache.commons.configuration.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JournalWriter
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(JournalWriter.class);
    private final LongAdder recordsWritten = new LongAdder();
    private final LongAdder bytesWritten = new LongAdder();
    private final ServerConfiguration conf;
    private final Flags flags;
    private final Recorder recorder = new Recorder(TimeUnit.SECONDS.toMillis(120000L), 5);
    private final Recorder cumulativeRecorder = new Recorder(TimeUnit.SECONDS.toMillis(120000L), 5);
    private final AtomicBoolean isDone = new AtomicBoolean(false);
    private static final DecimalFormat throughputFormat = new PaddingDecimalFormat("0.0", 8);
    private static final DecimalFormat dec = new PaddingDecimalFormat("0.000", 7);

    JournalWriter(CompositeConfiguration conf, Flags flags) {
        this.conf = new ServerConfiguration();
        this.conf.addConfiguration((Configuration)conf);
        this.flags = flags;
    }

    @Override
    public void run() {
        try {
            this.execute();
        }
        catch (Exception e) {
            log.error("Encountered exception at running dlog perf writer", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void execute() throws Exception {
        ObjectMapper m = new ObjectMapper();
        ObjectWriter w = m.writerWithDefaultPrettyPrinter();
        log.info("Starting journal perf writer with config : {}", (Object)w.writeValueAsString((Object)this.flags));
        Preconditions.checkArgument((this.flags.journalDirs.size() > 0 ? 1 : 0) != 0, (Object)"No journal dirs is provided");
        JournalWriter.updateServerConf(this.conf, this.flags);
        log.info("Benchmark the journal perf with server config : {}", (Object)this.conf.asJson());
        Stats.loadStatsProvider((Configuration)this.conf);
        Stats.get().start((Configuration)this.conf);
        StatsLogger statsLogger = Stats.get().getStatsLogger("").scope("bookie");
        ByteBufAllocator allocator = JournalWriter.getAllocator(this.conf);
        DiskChecker checker = new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold());
        LedgerDirsManager manager = new LedgerDirsManager(this.conf, this.conf.getLedgerDirs(), checker, (StatsLogger)NullStatsLogger.INSTANCE);
        Journal[] journals = new Journal[this.flags.journalDirs.size()];
        for (int i = 0; i < journals.length; ++i) {
            Journal journal;
            journals[i] = journal = new Journal(i, new File(this.flags.journalDirs.get(i)), this.conf, manager, statsLogger.scope("journal"), allocator);
            journal.start();
        }
        try {
            this.execute(journals);
        }
        finally {
            for (Journal journal : journals) {
                journal.shutdown();
            }
            Stats.get().stop();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void execute(Journal[] journals) throws Exception {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            this.isDone.set(true);
            JournalWriter.printAggregatedStats(this.cumulativeRecorder);
        }));
        ScheduledExecutorService flushExecutor = Executors.newSingleThreadScheduledExecutor();
        flushExecutor.scheduleAtFixedRate(() -> {
            for (Journal journal : journals) {
                CheckpointSource.Checkpoint cp = journal.newCheckpoint();
                try {
                    journal.checkpointComplete(cp, true);
                }
                catch (IOException e) {
                    log.error("Failed to complete checkpoint {}", (Object)cp, (Object)e);
                }
            }
        }, 30L, 30L, TimeUnit.SECONDS);
        ExecutorService executor = Executors.newFixedThreadPool(this.flags.numTestThreads);
        try {
            int i = 0;
            while (i < this.flags.numTestThreads) {
                int idx = i++;
                long numRecordsForThisThread = this.flags.numEntries / (long)this.flags.numTestThreads;
                long numBytesForThisThread = this.flags.numBytes / (long)this.flags.numTestThreads;
                double writeRateForThisThread = (double)this.flags.writeRate / (double)this.flags.numTestThreads;
                long maxOutstandingBytesForThisThread = this.flags.maxOutstandingMB * 1024L * 1024L / (long)this.flags.numTestThreads;
                int numLedgersForThisThread = this.flags.numLedgers / this.flags.numTestThreads;
                executor.submit(() -> {
                    try {
                        this.write(idx, journals, numLedgersForThisThread, writeRateForThisThread, (int)maxOutstandingBytesForThisThread, numRecordsForThisThread, numBytesForThisThread);
                    }
                    catch (Throwable t) {
                        log.error("Encountered error at writing records", t);
                    }
                });
            }
            log.info("Started {} write threads", (Object)this.flags.numTestThreads);
            this.reportStats();
        }
        finally {
            flushExecutor.shutdown();
            executor.shutdown();
            if (!executor.awaitTermination(5L, TimeUnit.SECONDS)) {
                executor.shutdownNow();
            }
        }
    }

    void write(int threadIdx, Journal[] journals, int numLedgersForThisThread, double writeRate, int maxOutstandingBytesForThisThread, long numRecordsForThisThread, long numBytesForThisThread) throws Exception {
        log.info("Write thread {} started with : rate = {}, num records = {}, num bytes = {}, max outstanding bytes = {}", new Object[]{threadIdx, writeRate, numRecordsForThisThread, numBytesForThisThread, maxOutstandingBytesForThisThread});
        RateLimiter limiter = writeRate > 0.0 ? RateLimiter.create((double)writeRate) : null;
        Semaphore semaphore = maxOutstandingBytesForThisThread > 0 ? new Semaphore(maxOutstandingBytesForThisThread) : null;
        if (limiter != null) {
            limiter.acquire((int)writeRate);
        }
        long totalWritten = 0L;
        long totalBytesWritten = 0L;
        int numJournals = journals.length;
        byte[] payload = new byte[this.flags.recordSize];
        ThreadLocalRandom.current().nextBytes(payload);
        ByteBuf payloadBuf = Unpooled.wrappedBuffer((byte[])payload);
        long[] entryIds = new long[numLedgersForThisThread];
        Arrays.fill(entryIds, 0L);
        block0: while (true) {
            int i = 0;
            while (true) {
                long eid;
                if (i >= numJournals) continue block0;
                int ledgerIdx = ThreadLocalRandom.current().nextInt(numLedgersForThisThread);
                long lid = threadIdx * numLedgersForThisThread + ledgerIdx;
                int n = ledgerIdx;
                entryIds[n] = entryIds[n] + 1L;
                ByteBuf buf = payloadBuf.retainedDuplicate();
                int len = buf.readableBytes();
                if (numRecordsForThisThread > 0L && totalWritten >= numRecordsForThisThread) {
                    this.markPerfDone();
                }
                if (numBytesForThisThread > 0L && totalBytesWritten >= numBytesForThisThread) {
                    this.markPerfDone();
                }
                if (null != semaphore) {
                    semaphore.acquire(len);
                }
                ++totalWritten;
                totalBytesWritten += (long)len;
                if (null != limiter) {
                    limiter.acquire(len);
                }
                long sendTime = System.nanoTime();
                journals[i].logAddEntry(lid, eid, buf, false, (rc, ledgerId, entryId, addr, ctx) -> {
                    ReferenceCountUtil.safeRelease((Object)buf);
                    if (0 == rc) {
                        if (null != semaphore) {
                            semaphore.release(len);
                        }
                        this.recordsWritten.increment();
                        this.bytesWritten.add(len);
                        long latencyMicros = TimeUnit.NANOSECONDS.toMicros(System.nanoTime() - sendTime);
                        this.recorder.recordValue(latencyMicros);
                        this.cumulativeRecorder.recordValue(latencyMicros);
                    } else {
                        log.warn("Error at writing records : {}", (Throwable)BookieException.create((int)rc));
                        Runtime.getRuntime().exit(-1);
                    }
                }, null);
                ++i;
            }
            break;
        }
    }

    @SuppressFBWarnings(value={"DM_EXIT"})
    void markPerfDone() throws Exception {
        log.info("------------------- DONE -----------------------");
        JournalWriter.printAggregatedStats(this.cumulativeRecorder);
        this.isDone.set(true);
        Thread.sleep(5000L);
        System.exit(0);
    }

    void reportStats() {
        long oldTime = System.nanoTime();
        Histogram reportHistogram = null;
        while (true) {
            try {
                Thread.sleep(10000L);
            }
            catch (InterruptedException e) {
                break;
            }
            if (this.isDone.get()) break;
            long now = System.nanoTime();
            double elapsed = (double)(now - oldTime) / 1.0E9;
            double rate = (double)this.recordsWritten.sumThenReset() / elapsed;
            double throughput = (double)this.bytesWritten.sumThenReset() / elapsed / 1024.0 / 1024.0;
            reportHistogram = this.recorder.getIntervalHistogram(reportHistogram);
            log.info("Throughput written : {}  records/s --- {} MB/s --- Latency: mean: {} ms - med: {} - 95pct: {} - 99pct: {} - 99.9pct: {} - 99.99pct: {} - Max: {}", new Object[]{throughputFormat.format(rate), throughputFormat.format(throughput), dec.format(reportHistogram.getMean() / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(50.0) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(95.0) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(99.0) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(99.9) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(99.99) / 1000.0), dec.format((double)reportHistogram.getMaxValue() / 1000.0)});
            reportHistogram.reset();
            oldTime = now;
        }
    }

    private static void updateServerConf(ServerConfiguration conf, Flags flags) {
        File[] currentDirs;
        conf.setJournalWriteBufferSizeKB(flags.writeBufferSizeKB);
        conf.setJournalMaxGroupWaitMSec((long)flags.groupCommitIntervalMs);
        conf.setJournalBufferedWritesThreshold((long)flags.groupCommitMaxBytes);
        conf.setNumJournalCallbackThreads(flags.numJournalCallbackThreads);
        conf.setJournalQueueSize(flags.journalQueueSize);
        conf.setJournalSyncData(flags.journalSyncEnabled);
        conf.setLedgerDirNames(flags.journalDirs.toArray(new String[0]));
        conf.setStatsProviderClass(PrometheusMetricsProvider.class);
        for (File dir : currentDirs = Bookie.getCurrentDirectories((File[])conf.getLedgerDirs())) {
            if (!dir.mkdirs()) continue;
            log.info("Successfully created dir {}", (Object)dir);
        }
    }

    private static ByteBufAllocator getAllocator(ServerConfiguration conf) {
        return ByteBufAllocatorBuilder.create().poolingPolicy(conf.getAllocatorPoolingPolicy()).poolingConcurrency(conf.getAllocatorPoolingConcurrency()).outOfMemoryPolicy(conf.getAllocatorOutOfMemoryPolicy()).outOfMemoryListener(ex -> log.error("Unable to allocate memory, exiting bookie", (Throwable)ex)).leakDetectionPolicy(conf.getAllocatorLeakDetectionPolicy()).build();
    }

    private static void printAggregatedStats(Recorder recorder) {
        Histogram reportHistogram = recorder.getIntervalHistogram();
        log.info("Aggregated latency stats --- Latency: mean: {} ms - med: {} - 95pct: {} - 99pct: {} - 99.9pct: {} - 99.99pct: {} - 99.999pct: {} - Max: {}", new Object[]{dec.format(reportHistogram.getMean() / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(50.0) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(95.0) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(99.0) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(99.9) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(99.99) / 1000.0), dec.format((double)reportHistogram.getValueAtPercentile(99.999) / 1000.0), dec.format((double)reportHistogram.getMaxValue() / 1000.0)});
    }

    public static class Flags
    extends CliFlags {
        @Parameter(names={"-t", "--num-test-threads"}, description="Num of test threads to append entries to journal")
        public int numTestThreads = 1;
        @Parameter(names={"-nl", "--num-ledgers"}, description="Num of ledgers to append entries to journal")
        public int numLedgers = 24;
        @Parameter(names={"-r", "--rate"}, description="Write rate bytes/s across journals")
        public int writeRate = 0;
        @Parameter(names={"-s", "--entry-size"}, description="Entry size")
        public int recordSize = 1024;
        @Parameter(names={"-j", "--journal-dirs"}, description="The list of journal directories, separated by comma", required=true)
        public List<String> journalDirs;
        @Parameter(names={"-mob", "--max-outstanding-megabytes"}, description="Number of threads writing")
        public long maxOutstandingMB = 200L;
        @Parameter(names={"-n", "--num-entries"}, description="Number of entries to write in total. If 0, it will keep writing")
        public long numEntries = 0L;
        @Parameter(names={"-b", "--num-bytes"}, description="Number of bytes to write in total. If 0, it will keep writing")
        public long numBytes = 0L;
        @Parameter(names={"-wb", "--write-buffer-size-kb"}, description="Journal write buffer size")
        public int writeBufferSizeKB = 1024;
        @Parameter(names={"--sync"}, description="Journal sync enabled")
        public boolean journalSyncEnabled = false;
        @Parameter(names={"-gci", "--group-commit-interval-ms"}, description="Journal group commit interval in milliseconds")
        public int groupCommitIntervalMs = 1;
        @Parameter(names={"-gcb", "--group-commit-max-bytes"}, description="Journal group commit max buffered bytes")
        public int groupCommitMaxBytes = 524288;
        @Parameter(names={"-q", "--journal-queue-size"}, description="Journal queue size")
        public int journalQueueSize = 10000;
        @Parameter(names={"-jt", "--num-journal-callback-threads"}, description="Number of journal callback threads")
        public int numJournalCallbackThreads = 8;
    }
}

