/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.http.impl.pool;

import io.netty.channel.Channel;
import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.http.ConnectionPoolTooBusyException;
import io.vertx.core.http.impl.pool.ConnectResult;
import io.vertx.core.http.impl.pool.ConnectionListener;
import io.vertx.core.http.impl.pool.ConnectionProvider;
import io.vertx.core.http.impl.pool.Waiter;
import io.vertx.core.impl.ContextImpl;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Queue;
import java.util.function.BiConsumer;

public class Pool<C> {
    private static final Logger log = LoggerFactory.getLogger(Pool.class);
    private final ConnectionProvider<C> connector;
    private final BiConsumer<Channel, C> connectionAdded;
    private final BiConsumer<Channel, C> connectionRemoved;
    private final int queueMaxSize;
    private final Queue<Waiter<C>> waitersQueue = new ArrayDeque<Waiter<C>>();
    private int waitersCount;
    private final Deque<Holder<C>> available;
    private final boolean fifo;
    private final long initialWeight;
    private final long maxWeight;
    private long weight;
    private boolean closed;
    private final Handler<Void> poolClosed;

    public Pool(ConnectionProvider<C> connector, int queueMaxSize, long initialWeight, long maxWeight, Handler<Void> poolClosed, BiConsumer<Channel, C> connectionAdded, BiConsumer<Channel, C> connectionRemoved, boolean fifo) {
        this.maxWeight = maxWeight;
        this.initialWeight = initialWeight;
        this.connector = connector;
        this.queueMaxSize = queueMaxSize;
        this.poolClosed = poolClosed;
        this.available = new ArrayDeque<Holder<C>>();
        this.connectionAdded = connectionAdded;
        this.connectionRemoved = connectionRemoved;
        this.fifo = fifo;
    }

    public synchronized int waitersInQueue() {
        return this.waitersQueue.size();
    }

    public synchronized int waitersCount() {
        return this.waitersCount;
    }

    public synchronized long weight() {
        return this.weight;
    }

    public synchronized long capacity() {
        return this.available.stream().mapToLong(c -> c.capacity).sum();
    }

    public synchronized boolean getConnection(ContextImpl context, Handler<AsyncResult<C>> handler) {
        if (this.closed) {
            return false;
        }
        Waiter<C> waiter = new Waiter<C>(context, handler);
        int size = this.waitersQueue.size();
        if (size == 0 && this.acquireConnection(waiter)) {
            ++this.waitersCount;
        } else if (this.queueMaxSize < 0 || size < this.queueMaxSize) {
            ++this.waitersCount;
            this.waitersQueue.add(waiter);
        } else {
            waiter.context.nettyEventLoop().execute(() -> waiter.handler.handle(Future.failedFuture(new ConnectionPoolTooBusyException("Connection pool reached max wait queue size of " + this.queueMaxSize))));
        }
        return true;
    }

    private boolean acquireConnection(Waiter<C> waiter) {
        if (this.available.size() > 0) {
            long capacity;
            Holder<C> conn = this.available.peek();
            if ((capacity = conn.capacity--) == 1L) {
                conn.expirationTimestamp = -1L;
                this.available.poll();
            }
            ContextImpl ctx = conn.context;
            --this.waitersCount;
            if (capacity == conn.concurrency) {
                this.connector.activate(conn.connection);
            }
            ctx.nettyEventLoop().execute(() -> waiter.handler.handle(Future.succeededFuture(conn.connection)));
            return true;
        }
        if (this.weight < this.maxWeight) {
            this.weight += this.initialWeight;
            waiter.context.nettyEventLoop().execute(() -> this.createConnection(waiter));
            return true;
        }
        return false;
    }

    public synchronized int closeIdle(long timestamp) {
        int removed = 0;
        if (this.waitersQueue.isEmpty() && this.available.size() > 0) {
            ArrayList<Holder<C>> copy = new ArrayList<Holder<C>>(this.available);
            for (Holder holder : copy) {
                if (holder.capacity != holder.concurrency || holder.expirationTimestamp > timestamp) continue;
                ++removed;
                this.closeConnection(holder);
                this.connector.close(holder.connection);
            }
        }
        return removed;
    }

    private void checkPending() {
        Waiter<C> waiter;
        while (this.waitersQueue.size() > 0 && this.acquireConnection(waiter = this.waitersQueue.peek())) {
            this.waitersQueue.poll();
        }
    }

    private ConnectionListener<C> createListener(final Holder<C> holder) {
        return new ConnectionListener<C>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void onConcurrencyChange(long concurrency) {
                Pool pool = Pool.this;
                synchronized (pool) {
                    if (holder.removed) {
                        return;
                    }
                    if (holder.concurrency < concurrency) {
                        long diff = concurrency - holder.concurrency;
                        if (holder.capacity == 0L) {
                            Pool.this.available.add(holder);
                        }
                        holder.capacity += diff;
                        holder.concurrency = concurrency;
                        Pool.this.checkPending();
                    } else if (holder.concurrency > concurrency) {
                        throw new UnsupportedOperationException("Not yet implemented");
                    }
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void onRecycle(long expirationTimestamp) {
                if (expirationTimestamp < 0L) {
                    throw new IllegalArgumentException("Invalid TTL");
                }
                Pool pool = Pool.this;
                synchronized (pool) {
                    if (holder.removed) {
                        return;
                    }
                    Pool.this.recycle(holder, 1, expirationTimestamp);
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void onDiscard() {
                Pool pool = Pool.this;
                synchronized (pool) {
                    if (holder.removed) {
                        return;
                    }
                    Pool.this.closed(holder);
                }
            }
        };
    }

    private void createConnection(Waiter<C> waiter) {
        Holder holder = new Holder();
        ConnectionListener listener = this.createListener(holder);
        this.connector.connect(listener, waiter.context, ar -> {
            if (ar.succeeded()) {
                ConnectResult result = (ConnectResult)ar.result();
                Pool pool = this;
                synchronized (pool) {
                    this.initConnection(holder, result.context(), result.concurrency(), result.connection(), result.channel(), result.weight());
                }
                pool = this;
                synchronized (pool) {
                    if (holder.capacity == 0L) {
                        this.waitersQueue.add(waiter);
                        this.checkPending();
                        return;
                    }
                    --this.waitersCount;
                    --holder.capacity;
                    if (holder.capacity > 0L) {
                        this.available.add(holder);
                    }
                    this.connector.activate(holder.connection);
                }
                waiter.handler.handle(Future.succeededFuture(holder.connection));
                pool = this;
                synchronized (pool) {
                    this.checkPending();
                }
            }
            waiter.handler.handle(Future.failedFuture(ar.cause()));
            Pool pool = this;
            synchronized (pool) {
                --this.waitersCount;
                this.weight -= this.initialWeight;
                holder.removed = true;
                this.checkPending();
                this.checkClose();
            }
        });
    }

    private synchronized void recycle(Holder<C> holder, int capacity, long timestamp) {
        this.recycleConnection(holder, capacity, timestamp);
        this.checkPending();
        this.checkClose();
    }

    private synchronized void closed(Holder<C> holder) {
        this.closeConnection(holder);
        this.checkPending();
        this.checkClose();
    }

    private void closeConnection(Holder<C> holder) {
        holder.removed = true;
        this.connectionRemoved.accept(holder.channel, holder.connection);
        if (holder.capacity > 0L) {
            this.available.remove(holder);
            holder.capacity = 0L;
        }
        this.weight -= holder.weight;
    }

    private void recycleConnection(Holder<C> conn, int c, long timestamp) {
        long newCapacity = conn.capacity + (long)c;
        if (newCapacity > conn.concurrency) {
            log.debug("Attempt to recycle a connection more than permitted");
            return;
        }
        if (timestamp == 0L && newCapacity == conn.concurrency && this.waitersQueue.isEmpty()) {
            this.available.remove(conn);
            conn.expirationTimestamp = -1L;
            conn.capacity = 0L;
            this.connector.close(conn.connection);
        } else {
            if (conn.capacity == 0L) {
                if (this.fifo) {
                    this.available.addLast(conn);
                } else {
                    this.available.addFirst(conn);
                }
            }
            conn.expirationTimestamp = timestamp;
            conn.capacity = newCapacity;
            if (newCapacity == conn.concurrency) {
                this.connector.deactivate(conn.connection);
            }
        }
    }

    private void initConnection(Holder<C> holder, ContextImpl context, long concurrency, C conn, Channel channel, long weight) {
        this.weight += this.initialWeight - weight;
        holder.context = context;
        holder.concurrency = concurrency;
        holder.connection = conn;
        holder.channel = channel;
        holder.weight = weight;
        holder.capacity = concurrency;
        holder.expirationTimestamp = -1L;
        this.connectionAdded.accept(holder.channel, holder.connection);
    }

    private void checkClose() {
        if (this.weight == 0L && this.waitersCount == 0) {
            this.closed = true;
            this.poolClosed.handle(null);
        }
    }

    public static class Holder<C> {
        boolean removed;
        C connection;
        long concurrency;
        long capacity;
        Channel channel;
        ContextImpl context;
        long weight;
        long expirationTimestamp;

        public String toString() {
            return "Holder[removed=" + this.removed + ",capacity=" + this.capacity + ",concurrency=" + this.concurrency + ",expirationTimestamp=" + this.expirationTimestamp + "]";
        }
    }
}

