/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.client;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookieWatcher;
import org.apache.bookkeeper.client.EnsemblePlacementPolicy;
import org.apache.bookkeeper.common.concurrent.FutureUtils;
import org.apache.bookkeeper.common.util.MathUtils;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.discover.RegistrationClient;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.stats.annotations.StatsDoc;
import org.apache.bookkeeper.versioning.Versioned;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@StatsDoc(name="bookie_watcher", help="Bookie watcher related stats")
class BookieWatcherImpl
implements BookieWatcher {
    private static final Logger log = LoggerFactory.getLogger(BookieWatcherImpl.class);
    private static final Function<Throwable, BKException> EXCEPTION_FUNC = cause -> {
        if (cause instanceof BKException) {
            log.error("Failed to get bookie list : ", cause);
            return (BKException)cause;
        }
        if (cause instanceof InterruptedException) {
            log.error("Interrupted reading bookie list : ", cause);
            return new BKException.BKInterruptedException();
        }
        BKException.MetaStoreException mse = new BKException.MetaStoreException((Throwable)cause);
        return mse;
    };
    private final ClientConfiguration conf;
    private final RegistrationClient registrationClient;
    private final EnsemblePlacementPolicy placementPolicy;
    @StatsDoc(name="NEW_ENSEMBLE_TIME", help="operation stats of new ensembles", parent="LEDGER_CREATE")
    private final OpStatsLogger newEnsembleTimer;
    @StatsDoc(name="REPLACE_BOOKIE_TIME", help="operation stats of replacing bookie in an ensemble")
    private final OpStatsLogger replaceBookieTimer;
    @StatsDoc(name="ENSEMBLE_NOT_ADHERING_TO_PLACEMENT_POLICY_COUNTER", help="total number of newEnsemble/replaceBookie operations failed to adhere EnsemblePlacementPolicy")
    private final Counter ensembleNotAdheringToPlacementPolicy;
    final Cache<BookieSocketAddress, Boolean> quarantinedBookies;
    private volatile Set<BookieSocketAddress> writableBookies = Collections.emptySet();
    private volatile Set<BookieSocketAddress> readOnlyBookies = Collections.emptySet();
    private CompletableFuture<?> initialWritableBookiesFuture = null;
    private CompletableFuture<?> initialReadonlyBookiesFuture = null;

    public BookieWatcherImpl(ClientConfiguration conf, EnsemblePlacementPolicy placementPolicy, RegistrationClient registrationClient, StatsLogger statsLogger) {
        this.conf = conf;
        this.placementPolicy = placementPolicy;
        this.registrationClient = registrationClient;
        this.quarantinedBookies = CacheBuilder.newBuilder().expireAfterWrite((long)conf.getBookieQuarantineTimeSeconds(), TimeUnit.SECONDS).removalListener((RemovalListener)new RemovalListener<BookieSocketAddress, Boolean>(){

            public void onRemoval(RemovalNotification<BookieSocketAddress, Boolean> bookie) {
                log.info("Bookie {} is no longer quarantined", bookie.getKey());
            }
        }).build();
        this.newEnsembleTimer = statsLogger.getOpStatsLogger("NEW_ENSEMBLE_TIME");
        this.replaceBookieTimer = statsLogger.getOpStatsLogger("REPLACE_BOOKIE_TIME");
        this.ensembleNotAdheringToPlacementPolicy = statsLogger.getCounter("ENSEMBLE_NOT_ADHERING_TO_PLACEMENT_POLICY_COUNTER");
    }

    @Override
    public Set<BookieSocketAddress> getBookies() throws BKException {
        try {
            return (Set)((Versioned)FutureUtils.result(this.registrationClient.getWritableBookies(), EXCEPTION_FUNC)).getValue();
        }
        catch (BKException.BKInterruptedException ie) {
            Thread.currentThread().interrupt();
            throw ie;
        }
    }

    @Override
    public Set<BookieSocketAddress> getAllBookies() throws BKException {
        try {
            return (Set)((Versioned)FutureUtils.result(this.registrationClient.getAllBookies(), EXCEPTION_FUNC)).getValue();
        }
        catch (BKException.BKInterruptedException ie) {
            Thread.currentThread().interrupt();
            throw ie;
        }
    }

    @Override
    public Set<BookieSocketAddress> getReadOnlyBookies() throws BKException {
        try {
            return (Set)((Versioned)FutureUtils.result(this.registrationClient.getReadOnlyBookies(), EXCEPTION_FUNC)).getValue();
        }
        catch (BKException.BKInterruptedException ie) {
            Thread.currentThread().interrupt();
            throw ie;
        }
    }

    private synchronized void processWritableBookiesChanged(Set<BookieSocketAddress> newBookieAddrs) {
        this.writableBookies = newBookieAddrs;
        this.placementPolicy.onClusterChanged(newBookieAddrs, this.readOnlyBookies);
    }

    private synchronized void processReadOnlyBookiesChanged(Set<BookieSocketAddress> readOnlyBookies) {
        this.readOnlyBookies = readOnlyBookies;
        this.placementPolicy.onClusterChanged(this.writableBookies, readOnlyBookies);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void initialBlockingBookieRead() throws BKException {
        CompletableFuture<Object> readonly;
        CompletableFuture<Object> writable;
        BookieWatcherImpl bookieWatcherImpl = this;
        synchronized (bookieWatcherImpl) {
            if (this.initialReadonlyBookiesFuture == null) {
                assert (this.initialWritableBookiesFuture == null);
                writable = this.registrationClient.watchWritableBookies(bookies -> this.processWritableBookiesChanged((Set)bookies.getValue()));
                readonly = this.registrationClient.watchReadOnlyBookies(bookies -> this.processReadOnlyBookiesChanged((Set)bookies.getValue()));
                this.initialWritableBookiesFuture = writable;
                this.initialReadonlyBookiesFuture = readonly;
            } else {
                writable = this.initialWritableBookiesFuture;
                readonly = this.initialReadonlyBookiesFuture;
            }
        }
        try {
            FutureUtils.result(writable, EXCEPTION_FUNC);
        }
        catch (BKException.BKInterruptedException ie) {
            Thread.currentThread().interrupt();
            throw ie;
        }
        try {
            FutureUtils.result(readonly, EXCEPTION_FUNC);
        }
        catch (BKException.BKInterruptedException ie) {
            Thread.currentThread().interrupt();
            throw ie;
        }
        catch (Exception e) {
            log.error("Failed getReadOnlyBookies: ", (Throwable)e);
        }
    }

    @Override
    public List<BookieSocketAddress> newEnsemble(int ensembleSize, int writeQuorumSize, int ackQuorumSize, Map<String, byte[]> customMetadata) throws BKException.BKNotEnoughBookiesException {
        List<BookieSocketAddress> socketAddresses;
        long startTime = MathUtils.nowInNano();
        try {
            Set quarantinedBookiesSet = this.quarantinedBookies.asMap().keySet();
            EnsemblePlacementPolicy.PlacementResult<List<BookieSocketAddress>> newEnsembleResponse = this.placementPolicy.newEnsemble(ensembleSize, writeQuorumSize, ackQuorumSize, customMetadata, new HashSet<BookieSocketAddress>(quarantinedBookiesSet));
            socketAddresses = newEnsembleResponse.getResult();
            EnsemblePlacementPolicy.PlacementPolicyAdherence isEnsembleAdheringToPlacementPolicy = newEnsembleResponse.isAdheringToPolicy();
            if (isEnsembleAdheringToPlacementPolicy == EnsemblePlacementPolicy.PlacementPolicyAdherence.FAIL) {
                this.ensembleNotAdheringToPlacementPolicy.inc();
                if (ensembleSize > 1) {
                    log.warn("New ensemble: {} is not adhering to Placement Policy. quarantinedBookies: {}", socketAddresses, quarantinedBookiesSet);
                }
            }
            this.newEnsembleTimer.registerSuccessfulEvent(MathUtils.nowInNano() - startTime, TimeUnit.NANOSECONDS);
        }
        catch (BKException.BKNotEnoughBookiesException e) {
            if (log.isDebugEnabled()) {
                log.debug("Not enough healthy bookies available, using quarantined bookies");
            }
            EnsemblePlacementPolicy.PlacementResult<List<BookieSocketAddress>> newEnsembleResponse = this.placementPolicy.newEnsemble(ensembleSize, writeQuorumSize, ackQuorumSize, customMetadata, new HashSet<BookieSocketAddress>());
            socketAddresses = newEnsembleResponse.getResult();
            EnsemblePlacementPolicy.PlacementPolicyAdherence isEnsembleAdheringToPlacementPolicy = newEnsembleResponse.isAdheringToPolicy();
            if (isEnsembleAdheringToPlacementPolicy == EnsemblePlacementPolicy.PlacementPolicyAdherence.FAIL) {
                this.ensembleNotAdheringToPlacementPolicy.inc();
                log.warn("New ensemble: {} is not adhering to Placement Policy", socketAddresses);
            }
            this.newEnsembleTimer.registerFailedEvent(MathUtils.nowInNano() - startTime, TimeUnit.NANOSECONDS);
        }
        return socketAddresses;
    }

    @Override
    public BookieSocketAddress replaceBookie(int ensembleSize, int writeQuorumSize, int ackQuorumSize, Map<String, byte[]> customMetadata, List<BookieSocketAddress> existingBookies, int bookieIdx, Set<BookieSocketAddress> excludeBookies) throws BKException.BKNotEnoughBookiesException {
        BookieSocketAddress socketAddress;
        long startTime = MathUtils.nowInNano();
        BookieSocketAddress addr = existingBookies.get(bookieIdx);
        EnsemblePlacementPolicy.PlacementPolicyAdherence isEnsembleAdheringToPlacementPolicy = EnsemblePlacementPolicy.PlacementPolicyAdherence.FAIL;
        try {
            HashSet<BookieSocketAddress> excludedBookiesAndQuarantinedBookies = new HashSet<BookieSocketAddress>(excludeBookies);
            Set quarantinedBookiesSet = this.quarantinedBookies.asMap().keySet();
            excludedBookiesAndQuarantinedBookies.addAll(quarantinedBookiesSet);
            EnsemblePlacementPolicy.PlacementResult<BookieSocketAddress> replaceBookieResponse = this.placementPolicy.replaceBookie(ensembleSize, writeQuorumSize, ackQuorumSize, customMetadata, existingBookies, addr, excludedBookiesAndQuarantinedBookies);
            socketAddress = replaceBookieResponse.getResult();
            isEnsembleAdheringToPlacementPolicy = replaceBookieResponse.isAdheringToPolicy();
            if (isEnsembleAdheringToPlacementPolicy == EnsemblePlacementPolicy.PlacementPolicyAdherence.FAIL) {
                this.ensembleNotAdheringToPlacementPolicy.inc();
                log.warn("replaceBookie for bookie: {} in ensemble: {} is not adhering to placement policy and chose {}. excludedBookies {} and quarantinedBookies {}", new Object[]{addr, existingBookies, socketAddress, excludeBookies, quarantinedBookiesSet});
            }
            this.replaceBookieTimer.registerSuccessfulEvent(MathUtils.nowInNano() - startTime, TimeUnit.NANOSECONDS);
        }
        catch (BKException.BKNotEnoughBookiesException e) {
            if (log.isDebugEnabled()) {
                log.debug("Not enough healthy bookies available, using quarantined bookies");
            }
            EnsemblePlacementPolicy.PlacementResult<BookieSocketAddress> replaceBookieResponse = this.placementPolicy.replaceBookie(ensembleSize, writeQuorumSize, ackQuorumSize, customMetadata, existingBookies, addr, excludeBookies);
            socketAddress = replaceBookieResponse.getResult();
            isEnsembleAdheringToPlacementPolicy = replaceBookieResponse.isAdheringToPolicy();
            if (isEnsembleAdheringToPlacementPolicy == EnsemblePlacementPolicy.PlacementPolicyAdherence.FAIL) {
                this.ensembleNotAdheringToPlacementPolicy.inc();
                log.warn("replaceBookie for bookie: {} in ensemble: {} is not adhering to placement policy and chose {}. excludedBookies {}", new Object[]{addr, existingBookies, socketAddress, excludeBookies});
            }
            this.replaceBookieTimer.registerFailedEvent(MathUtils.nowInNano() - startTime, TimeUnit.NANOSECONDS);
        }
        return socketAddress;
    }

    @Override
    public void quarantineBookie(BookieSocketAddress bookie) {
        if (this.quarantinedBookies.getIfPresent((Object)bookie) == null) {
            this.quarantinedBookies.put((Object)bookie, (Object)Boolean.TRUE);
            log.warn("Bookie {} has been quarantined because of read/write errors.", (Object)bookie);
        }
    }
}

