#!/bin/bash

# Licensed to the Apache Software Foundation (ASF) under one or more
# contributor license agreements.  See the NOTICE file distributed with
# this work for additional information regarding copyright ownership.
# The ASF licenses this file to You under the Apache License, Version 2.0
# (the "License"); you may not use this file except in compliance with
# the License.  You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

set -e

usage() {
  echo "
usage: $0 <options>
  Required not-so-options:
     --distro-dir=DIR            path to distro specific files (debian/RPM)
     --build-dir=DIR             path to dist dir
     --prefix=PREFIX             path to install into

  Optional options:
     --doc-dir=DIR               path to install docs into [/usr/share/doc/elasticsearch]
     --lib-dir=DIR               path to install bits [/usr/lib/elasticsearch]
     --installed-lib-dir=DIR     path where lib-dir will end up on target system
     --bin-dir=DIR               path to install bins [/usr/bin]
     --examples-dir=DIR          path to install examples [doc-dir/examples]
     ... [ see source for more similar options ]
  "
  exit 1
}

OPTS=$(getopt \
  -n $0 \
  -o '' \
  -l 'prefix:' \
  -l 'distro-dir:' \
  -l 'doc-dir:' \
  -l 'lib-dir:' \
  -l 'installed-lib-dir:' \
  -l 'bin-dir:' \
  -l 'initd-dir:' \
  -l 'build-dir:' -- "$@")

if [ $? != 0 ] ; then
    usage
fi

eval set -- "$OPTS"
while true ; do
    case "$1" in
        --prefix)
        PREFIX=$2 ; shift 2
        ;;
        --distro-dir)
        DISTRO_DIR=$2 ; shift 2
        ;;
        --build-dir)
        BUILD_DIR=$2 ; shift 2
        ;;
        --doc-dir)
        DOC_DIR=$2 ; shift 2
        ;;
        --lib-dir)
        LIB_DIR=$2 ; shift 2
        ;;
        --installed-lib-dir)
        INSTALLED_LIB_DIR=$2 ; shift 2
        ;;
        --bin-dir)
        BIN_DIR=$2 ; shift 2
        ;;
        --initd-dir)
        INITD_DIR=$2 ; shift 2
        ;;
        --)
        shift; break
        ;;
        *)
        echo "Unknown option: $1"
        usage
        exit 1
        ;;
    esac
done

for var in PREFIX BUILD_DIR DISTRO_DIR ; do
  if [ -z "$(eval "echo \$$var")" ]; then
    echo Missing param: $var
    usage
  fi
done

MAN_DIR=${MAN_DIR:-/usr/share/man/man1}
DOC_DIR=${DOC_DIR:-/usr/share/doc/elasticsearch}
LIB_DIR=${LIB_DIR:-/usr/lib/elasticsearch}
INITD_DIR=${INITD_DIR:-/etc/init.d}
INSTALLED_LIB_DIR=${INSTALLED_LIB_DIR:-/usr/lib/elasticsearch}
BIN_DIR=${BIN_DIR:-/usr/bin}
CONF_DIR=${CONF_DIR:-/etc/elasticsearch/conf}
DEFAULT_DIR=${ETC_DIR:-/etc/default}

VAR_DIR=$PREFIX/var

install -d -m 0755 $PREFIX/$LIB_DIR
cp -ra ${BUILD_DIR}/lib $PREFIX/$LIB_DIR/lib

install -d -m 0755 $PREFIX/$LIB_DIR/modules
cp -ra ${BUILD_DIR}/modules/* $PREFIX/$LIB_DIR/modules

install -d -m 0755 $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/bin/elasticsearch $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/bin/elasticsearch.in.sh $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/bin/elasticsearch-keystore $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/bin/elasticsearch-plugin $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/bin/elasticsearch-systemd-pre-exec $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/bin/elasticsearch-translog $PREFIX/$LIB_DIR/bin
chmod 755 $PREFIX/$LIB_DIR/bin/*

install -d -m 0755 $PREFIX/$LIB_DIR/plugins
install -d -m 0755 $PREFIX/$LIB_DIR/licenses
cp -a  ${BUILD_DIR}/LICENSE.txt $PREFIX/$LIB_DIR/licenses/

# copy service script
install -d -m 0755 $PREFIX/${INITD_DIR}
cp ${DISTRO_DIR}/elasticsearch.init $PREFIX/${INITD_DIR}/elasticsearch
chmod 755 $PREFIX/${INITD_DIR}/elasticsearch

# Copy in the configuration files
install -d -m 0755 $PREFIX/$DEFAULT_DIR
cp ${DISTRO_DIR}/elasticsearch.default $PREFIX/$DEFAULT_DIR/elasticsearch

install -d -m 0755 $PREFIX/${CONF_DIR}.dist
cp ${BUILD_DIR}/config/elasticsearch.yml $PREFIX/${CONF_DIR}.dist
cp ${BUILD_DIR}/config/jvm.options $PREFIX/${CONF_DIR}.dist
cp ${BUILD_DIR}/config/log4j2.properties $PREFIX/${CONF_DIR}.dist
install -d -m 0755 $PREFIX/${CONF_DIR}.dist/scripts

# precreating /var layout
install -d -m 0755 $VAR_DIR/log/elasticsearch
install -d -m 0755 $VAR_DIR/run/elasticsearch
install -d -m 0755 $VAR_DIR/lib/elasticsearch
