// Licensed to the Apache Software Foundation (ASF) under one or more
// contributor license agreements.  See the NOTICE file distributed with
// this work for additional information regarding copyright ownership.
// The ASF licenses this file to You under the Apache License, Version 2.0
// (the "License"); you may not use this file except in compliance with
// the License.  You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package executors

import (
	"beam.apache.org/playground/backend/internal/preparators"
	"beam.apache.org/playground/backend/internal/validators"
)

type handler func(executor *Executor)

//ExecutorBuilder struct
type ExecutorBuilder struct {
	actions []handler
}

//CompileBuilder facet of ExecutorBuilder
type CompileBuilder struct {
	ExecutorBuilder
}

//RunBuilder facet of ExecutorBuilder
type RunBuilder struct {
	ExecutorBuilder
}

//ValidatorBuilder facet of ExecutorBuilder
type ValidatorBuilder struct {
	ExecutorBuilder
}

//PreparatorBuilder facet of ExecutorBuilder
type PreparatorBuilder struct {
	ExecutorBuilder
}

//UnitTestExecutorBuilder facet of ExecutorBuilder
type UnitTestExecutorBuilder struct {
	ExecutorBuilder
}

//NewExecutorBuilder constructor for Executor
func NewExecutorBuilder() *ExecutorBuilder {
	return &ExecutorBuilder{}
}

//WithExecutableFileName adds file name to executor
func (b *ExecutorBuilder) WithExecutableFileName(name string) *ExecutorBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.runArgs.fileName = name
		e.testArgs.fileName = name
	})
	return b
}

//WithWorkingDir adds dir path to executor
func (b *ExecutorBuilder) WithWorkingDir(dir string) *ExecutorBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.compileArgs.workingDir = dir
		e.runArgs.workingDir = dir
		e.testArgs.workingDir = dir
	})
	return b
}

// WithCompiler - Lives chains to type *ExecutorBuilder and returns a *CompileBuilder
func (b *ExecutorBuilder) WithCompiler() *CompileBuilder {
	return &CompileBuilder{*b}
}

// WithRunner - Lives chains to type *ExecutorBuilder and returns a *CompileBuilder
func (b *ExecutorBuilder) WithRunner() *RunBuilder {
	return &RunBuilder{*b}
}

// WithValidator - Lives chains to type *ExecutorBuilder and returns a *CompileBuilder
func (b *ExecutorBuilder) WithValidator() *ValidatorBuilder {
	return &ValidatorBuilder{*b}
}

// WithPreparator - Lives chains to type *ExecutorBuilder and returns a *PreparatorBuilder
func (b *ExecutorBuilder) WithPreparator() *PreparatorBuilder {
	return &PreparatorBuilder{*b}
}

// WithTestRunner - Lives chains to type *ExecutorBuilder and returns a *UnitTestExecutorBuilder
func (b *ExecutorBuilder) WithTestRunner() *UnitTestExecutorBuilder {
	return &UnitTestExecutorBuilder{*b}
}

//WithCommand adds compile command to executor
func (b *CompileBuilder) WithCommand(compileCmd string) *CompileBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.compileArgs.commandName = compileCmd
	})
	return b
}

//WithArgs adds compile args to executor
func (b *CompileBuilder) WithArgs(compileArgs []string) *CompileBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.compileArgs.commandArgs = compileArgs
	})
	return b
}

//WithFileName adds file name to executor
func (b *CompileBuilder) WithFileName(fileName string) *CompileBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.compileArgs.fileName = fileName
	})
	return b
}

//WithCommand adds run command to executor
func (b *RunBuilder) WithCommand(runCmd string) *RunBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.runArgs.commandName = runCmd
	})
	return b
}

//WithArgs adds run args to executor
func (b *RunBuilder) WithArgs(runArgs []string) *RunBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.runArgs.commandArgs = runArgs
	})
	return b
}

//WithGraphOutput adds the need of graph output to executor
func (b *RunBuilder) WithGraphOutput() *RunBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		//todo
	})
	return b
}

//WithCommand adds test command to executor
func (b *UnitTestExecutorBuilder) WithCommand(testCmd string) *UnitTestExecutorBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.testArgs.commandName = testCmd
	})
	return b
}

//WithArgs adds test args to executor
func (b *UnitTestExecutorBuilder) WithArgs(testArgs []string) *UnitTestExecutorBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.testArgs.commandArgs = testArgs
	})
	return b
}

// WithWorkingDir adds dir path to executor
func (b *UnitTestExecutorBuilder) WithWorkingDir(dir string) *UnitTestExecutorBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.testArgs.workingDir = dir
	})
	return b
}

//WithGraphOutput adds the need of graph output to executor
func (b *UnitTestExecutorBuilder) WithGraphOutput() *UnitTestExecutorBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		//todo
	})
	return b
}

//WithSdkValidators sets validators to executor
func (b *ValidatorBuilder) WithSdkValidators(validators *[]validators.Validator) *ValidatorBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.validators = *validators
	})
	return b
}

//WithSdkPreparators sets preparators to executor
func (b *PreparatorBuilder) WithSdkPreparators(preparators *[]preparators.Preparator) *PreparatorBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.preparators = *preparators
	})
	return b
}

//Build builds the executor object
func (b *ExecutorBuilder) Build() Executor {
	executor := Executor{}
	for _, a := range b.actions {
		a(&executor)
	}
	return executor
}

//WithPipelineOptions adds pipeline options to executor
func (b *RunBuilder) WithPipelineOptions(pipelineOptions []string) *RunBuilder {
	b.actions = append(b.actions, func(e *Executor) {
		e.runArgs.pipelineOptions = pipelineOptions
	})
	return b
}
