/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.asterix.om.base;

import com.fasterxml.jackson.databind.ObjectMapper;

import com.fasterxml.jackson.databind.node.ObjectNode;

import org.apache.asterix.common.exceptions.AsterixException;
import org.apache.asterix.om.types.BuiltinType;
import org.apache.asterix.om.types.IAType;
import org.apache.asterix.om.visitors.IOMVisitor;

public class AInt64 implements IAObject {

    protected long value;

    public AInt64(Long value) {
        this.value = value;
    }

    public long getLongValue() {
        return value;
    }

    @Override
    public IAType getType() {
        return BuiltinType.AINT64;
    }

    @Override
    public void accept(IOMVisitor visitor) throws AsterixException {
        visitor.visitAInt64(this);
    }

    @Override
    public boolean deepEqual(IAObject obj) {
        if (!(obj instanceof AInt64)) {
            return false;
        }
        return value == ((AInt64) obj).getLongValue();
    }

    @Override
    public int hash() {
        return (int) (value ^ (value >>> 32));
    }

    @Override
    public String toString() {
        return "AInt64: {" + value + "}";
    }

    @Override
    public ObjectNode toJSON()  {
        ObjectMapper om = new ObjectMapper();
        ObjectNode json = om.createObjectNode();

        json.put("AInt64", value);

        return json;
    }
}
