# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

# Common definitions for the CMake projects in this repository.
# Must define REPOSITORY_ROOT before including this.

enable_language(C CXX)

set(BUILD_SUPPORT_DIR "${REPOSITORY_ROOT}/ci/build_support")

include(AdbcVersion)
include(CheckLinkerFlag)
include(DefineOptions)
include(GNUInstallDirs) # Populates CMAKE_INSTALL_INCLUDEDIR
include(san-config)

# XXX: remove this, rely on user config
if(ADBC_DEPENDENCY_SOURCE STREQUAL "CONDA")
  message(STATUS "Adding \$CONDA_PREFIX to CMAKE_PREFIX_PATH")
  list(APPEND CMAKE_PREFIX_PATH "$ENV{CONDA_PREFIX}")
endif()

# pkg-config (.pc file) support.
if(IS_ABSOLUTE "${CMAKE_INSTALL_INCLUDEDIR}")
  set(ADBC_PKG_CONFIG_INCLUDEDIR "${CMAKE_INSTALL_INCLUDEDIR}")
else()
  set(ADBC_PKG_CONFIG_INCLUDEDIR "\${prefix}/${CMAKE_INSTALL_INCLUDEDIR}")
endif()
if(IS_ABSOLUTE "${CMAKE_INSTALL_LIBDIR}")
  set(ADBC_PKG_CONFIG_LIBDIR "${CMAKE_INSTALL_LIBDIR}")
else()
  set(ADBC_PKG_CONFIG_LIBDIR "\${prefix}/${CMAKE_INSTALL_LIBDIR}")
endif()

# ------------------------------------------------------------
# Common build utilities

# Link flags
set(ADBC_LINK_FLAGS)

set(ADBC_VERSION_SCRIPT_LINK_FLAG "-Wl,--version-script=${REPOSITORY_ROOT}/c/symbols.map")

check_linker_flag(CXX ${ADBC_VERSION_SCRIPT_LINK_FLAG} CXX_LINKER_SUPPORTS_VERSION_SCRIPT)
if(CXX_LINKER_SUPPORTS_VERSION_SCRIPT)
  list(APPEND ADBC_LINK_FLAGS ${ADBC_VERSION_SCRIPT_LINK_FLAG})
endif()

# Set common build options
if("${ADBC_BUILD_WARNING_LEVEL}" STREQUAL "")
  string(TOLOWER "${CMAKE_BUILD_TYPE}" _lower_build_type)
  if("${_lower_build_type}" STREQUAL "release")
    set(ADBC_BUILD_WARNING_LEVEL "PRODUCTION")
  else()
    set(ADBC_BUILD_WARNING_LEVEL "CHECKIN")
  endif()
endif()

if(MSVC)
  set(ADBC_C_CXX_FLAGS_CHECKIN /Wall /WX)
  set(ADBC_C_CXX_FLAGS_PRODUCTION /Wall)
elseif(CMAKE_CXX_COMPILER_ID STREQUAL "AppleClang"
       OR CMAKE_CXX_COMPILER_ID STREQUAL "Clang"
       OR CMAKE_CXX_COMPILER_ID STREQUAL "GNU")
  set(ADBC_C_CXX_FLAGS_CHECKIN
      -Wall
      -Wextra
      -Wpedantic
      -Werror
      -Wno-unused-parameter)
  set(ADBC_C_CXX_FLAGS_PRODUCTION -Wall)
else()
  message(WARNING "Unknown compiler: ${CMAKE_CXX_COMPILER_ID}")
endif()

macro(adbc_configure_target TARGET)
  target_compile_options(${TARGET}
                         PRIVATE ${ADBC_C_CXX_FLAGS_${ADBC_BUILD_WARNING_LEVEL}})
endmacro()

# Common testing setup
add_custom_target(all-tests)
if(ADBC_BUILD_TESTS)
  find_package(GTest)
  if(NOT GTest_FOUND)
    message(STATUS "Building googletest from source")
    # Required for GoogleTest
    set(CMAKE_CXX_STANDARD 17)
    include(FetchContent)
    fetchcontent_declare(googletest
                         URL https://github.com/google/googletest/archive/03597a01ee50ed33e9dfd640b249b4be3799d395.zip
    )
    # Windows: https://stackoverflow.com/questions/12540970/
    set(gtest_force_shared_crt
        ON
        CACHE BOOL "" FORCE)
    fetchcontent_makeavailable(googletest)
  endif()
  set(ADBC_TEST_LINK_LIBS GTest::gtest_main GTest::gtest GTest::gmock)
endif()
