#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#
from typing import TYPE_CHECKING, Any, Dict, List, Optional, Sequence

from airflow.models import BaseOperator
from airflow.providers.opsgenie.hooks.opsgenie import OpsgenieAlertHook

if TYPE_CHECKING:
    from airflow.utils.context import Context


class OpsgenieCreateAlertOperator(BaseOperator):
    """
    This operator allows you to post alerts to Opsgenie.
    Accepts a connection that has an Opsgenie API key as the connection's password.
    This operator sets the domain to conn_id.host, and if not set will default
    to ``https://api.opsgenie.com``.

    Each Opsgenie API key can be pre-configured to a team integration.
    You can override these defaults in this operator.

    .. seealso::
        For more information on how to use this operator, take a look at the guide:
        :ref:`howto/operator:OpsgenieCreateAlertOperator`

    :param opsgenie_conn_id: The name of the Opsgenie connection to use
    :type opsgenie_conn_id: str
    :param message: The Message of the Opsgenie alert (templated)
    :type message: str
    :param alias: Client-defined identifier of the alert (templated)
    :type alias: str
    :param description: Description field of the alert (templated)
    :type description: str
    :param responders: Teams, users, escalations and schedules that
        the alert will be routed to send notifications.
    :type responders: list[dict]
    :param visible_to: Teams and users that the alert will become visible
        to without sending any notification.
    :type visible_to: list[dict]
    :param actions: Custom actions that will be available for the alert.
    :type actions: list[str]
    :param tags: Tags of the alert.
    :type tags: list[str]
    :param details: Map of key-value pairs to use as custom properties of the alert.
    :type details: dict
    :param entity: Entity field of the alert that is
        generally used to specify which domain alert is related to. (templated)
    :type entity: str
    :param source: Source field of the alert. Default value is
        IP address of the incoming request.
    :type source: str
    :param priority: Priority level of the alert. Default value is P3. (templated)
    :type priority: str
    :param user: Display name of the request owner.
    :type user: str
    :param note: Additional note that will be added while creating the alert. (templated)
    :type note: str
    """

    template_fields: Sequence[str] = ('message', 'alias', 'description', 'entity', 'priority', 'note')

    def __init__(
        self,
        *,
        message: str,
        opsgenie_conn_id: str = 'opsgenie_default',
        alias: Optional[str] = None,
        description: Optional[str] = None,
        responders: Optional[List[dict]] = None,
        visible_to: Optional[List[dict]] = None,
        actions: Optional[List[str]] = None,
        tags: Optional[List[str]] = None,
        details: Optional[dict] = None,
        entity: Optional[str] = None,
        source: Optional[str] = None,
        priority: Optional[str] = None,
        user: Optional[str] = None,
        note: Optional[str] = None,
        **kwargs,
    ) -> None:
        super().__init__(**kwargs)

        self.message = message
        self.opsgenie_conn_id = opsgenie_conn_id
        self.alias = alias
        self.description = description
        self.responders = responders
        self.visible_to = visible_to
        self.actions = actions
        self.tags = tags
        self.details = details
        self.entity = entity
        self.source = source
        self.priority = priority
        self.user = user
        self.note = note
        self.hook: Optional[OpsgenieAlertHook] = None

    def _build_opsgenie_payload(self) -> Dict[str, Any]:
        """
        Construct the Opsgenie JSON payload. All relevant parameters are combined here
        to a valid Opsgenie JSON payload.

        :return: Opsgenie payload (dict) to send
        """
        payload = {}

        for key in [
            "message",
            "alias",
            "description",
            "responders",
            "visible_to",
            "actions",
            "tags",
            "details",
            "entity",
            "source",
            "priority",
            "user",
            "note",
        ]:
            val = getattr(self, key)
            if val:
                payload[key] = val
        return payload

    def execute(self, context: 'Context') -> None:
        """Call the OpsgenieAlertHook to post message"""
        self.hook = OpsgenieAlertHook(self.opsgenie_conn_id)
        self.hook.create_alert(self._build_opsgenie_payload())


class OpsgenieCloseAlertOperator(BaseOperator):
    """
    This operator allows you to close alerts to Opsgenie.
    Accepts a connection that has an Opsgenie API key as the connection's password.
    This operator sets the domain to conn_id.host, and if not set will default
    to ``https://api.opsgenie.com``.

    Each Opsgenie API key can be pre-configured to a team integration.
    You can override these defaults in this operator.

    .. seealso::
        For more information on how to use this operator, take a look at the guide:
        :ref:`howto/operator:OpsgenieCloseAlertOperator`

    :param opsgenie_conn_id: The name of the Opsgenie connection to use
    :type opsgenie_conn_id: str
    :param identifier: Identifier of alert which could be alert id, tiny id or alert alias
    :type identifier: str
    :param identifier_type: Type of the identifier that is provided as an in-line parameter.
        Possible values are 'id', 'alias' or 'tiny'
    :type identifier_type: str
    :param user: display name of the request owner
    :type user: str
    :param note: additional note that will be added while creating the alert
    :type note: str
    :param source: source field of the alert. Default value is IP address of the incoming request
    :type source: str
    :param close_alert_kwargs: additional params to pass
    :type close_alert_kwargs: dict
    """

    def __init__(
        self,
        *,
        identifier: str,
        opsgenie_conn_id: str = 'opsgenie_default',
        identifier_type: Optional[str] = None,
        user: Optional[str] = None,
        note: Optional[str] = None,
        source: Optional[str] = None,
        close_alert_kwargs: Optional[dict] = None,
        **kwargs,
    ) -> None:
        super().__init__(**kwargs)

        self.opsgenie_conn_id = opsgenie_conn_id
        self.identifier = identifier
        self.identifier_type = identifier_type
        self.user = user
        self.note = note
        self.source = source
        self.close_alert_kwargs = close_alert_kwargs
        self.hook: Optional[OpsgenieAlertHook] = None

    def _build_opsgenie_close_alert_payload(self) -> Dict[str, Any]:
        """
        Construct the Opsgenie JSON payload. All relevant parameters are combined here
        to a valid Opsgenie JSON payload.

        :return: Opsgenie close alert payload (dict) to send
        """
        payload = {}

        for key in [
            "user",
            "note",
            "source",
        ]:
            val = getattr(self, key)
            if val:
                payload[key] = val
        return payload

    def execute(self, context: 'Context') -> None:
        """Call the OpsgenieAlertHook to close alert"""
        self.hook = OpsgenieAlertHook(self.opsgenie_conn_id)
        self.hook.close_alert(
            identifier=self.identifier,
            identifier_type=self.identifier_type,
            payload=self._build_opsgenie_close_alert_payload(),
            kwargs=self.close_alert_kwargs,
        )
