/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.tserver.logger;

import com.google.common.base.Preconditions;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Arrays;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.KeyBuilder;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.tserver.logger.LogEvents;
import org.apache.hadoop.io.DataInputBuffer;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.io.WritableComparable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogFileKey
implements WritableComparable<LogFileKey> {
    private static final Logger log = LoggerFactory.getLogger(LogFileKey.class);
    public LogEvents event;
    public String filename = null;
    public KeyExtent tablet = null;
    public long seq = -1L;
    public int tabletId = -1;
    public static final int VERSION = 2;
    public String tserverSession;

    public void readFields(DataInput in) throws IOException {
        byte value = in.readByte();
        if (value >= LogEvents.values().length) {
            throw new IOException("Invalid LogEvent type, got ordinal " + value + ", but only know about " + LogEvents.values().length + " possible types.");
        }
        this.event = LogEvents.values()[value];
        switch (this.event) {
            case OPEN: {
                this.tabletId = in.readInt();
                this.tserverSession = in.readUTF();
                if (this.tabletId == 2) break;
                throw new RuntimeException(String.format("Bad version number for log file: expected %d, but saw %d", 2, this.tabletId));
            }
            case COMPACTION_FINISH: {
                this.seq = in.readLong();
                this.tabletId = in.readInt();
                break;
            }
            case COMPACTION_START: {
                this.seq = in.readLong();
                this.tabletId = in.readInt();
                this.filename = in.readUTF();
                break;
            }
            case DEFINE_TABLET: {
                this.seq = in.readLong();
                this.tabletId = in.readInt();
                this.tablet = KeyExtent.readFrom((DataInput)in);
                break;
            }
            case MANY_MUTATIONS: {
                this.seq = in.readLong();
                this.tabletId = in.readInt();
                break;
            }
            case MUTATION: {
                this.seq = in.readLong();
                this.tabletId = in.readInt();
                break;
            }
            default: {
                throw new RuntimeException("Unknown log event type: " + this.event);
            }
        }
    }

    public void write(DataOutput out) throws IOException {
        out.writeByte(this.event.ordinal());
        switch (this.event) {
            case OPEN: {
                this.seq = -1L;
                this.tabletId = -1;
                out.writeInt(2);
                out.writeUTF(this.tserverSession);
                break;
            }
            case COMPACTION_FINISH: {
                out.writeLong(this.seq);
                out.writeInt(this.tabletId);
                break;
            }
            case COMPACTION_START: {
                out.writeLong(this.seq);
                out.writeInt(this.tabletId);
                out.writeUTF(this.filename);
                break;
            }
            case DEFINE_TABLET: {
                out.writeLong(this.seq);
                out.writeInt(this.tabletId);
                this.tablet.writeTo(out);
                break;
            }
            case MANY_MUTATIONS: {
                out.writeLong(this.seq);
                out.writeInt(this.tabletId);
                break;
            }
            case MUTATION: {
                out.writeLong(this.seq);
                out.writeInt(this.tabletId);
                break;
            }
            default: {
                throw new IllegalArgumentException("Bad value for LogFileEntry type");
            }
        }
    }

    static int eventType(LogEvents event) {
        if (event == LogEvents.MUTATION || event == LogEvents.MANY_MUTATIONS) {
            return 3;
        }
        if (event == LogEvents.DEFINE_TABLET) {
            return 1;
        }
        if (event == LogEvents.OPEN) {
            return 0;
        }
        return 2;
    }

    private static int sign(long l) {
        if (l < 0L) {
            return -1;
        }
        if (l > 0L) {
            return 1;
        }
        return 0;
    }

    public int compareTo(LogFileKey o) {
        if (LogFileKey.eventType(this.event) != LogFileKey.eventType(o.event)) {
            return LogFileKey.eventType(this.event) - LogFileKey.eventType(o.event);
        }
        if (this.event == LogEvents.OPEN) {
            return 0;
        }
        if (this.tabletId != o.tabletId) {
            return this.tabletId - o.tabletId;
        }
        return LogFileKey.sign(this.seq - o.seq);
    }

    public boolean equals(Object obj) {
        if (obj instanceof LogFileKey) {
            return this.compareTo((LogFileKey)obj) == 0;
        }
        return false;
    }

    public int hashCode() {
        return (int)this.seq;
    }

    public String toString() {
        switch (this.event) {
            case OPEN: {
                return String.format("OPEN %s", this.tserverSession);
            }
            case COMPACTION_FINISH: {
                return String.format("COMPACTION_FINISH %d %d", this.tabletId, this.seq);
            }
            case COMPACTION_START: {
                return String.format("COMPACTION_START %d %d %s", this.tabletId, this.seq, this.filename);
            }
            case MUTATION: {
                return String.format("MUTATION %d %d", this.tabletId, this.seq);
            }
            case MANY_MUTATIONS: {
                return String.format("MANY_MUTATIONS %d %d", this.tabletId, this.seq);
            }
            case DEFINE_TABLET: {
                return String.format("DEFINE_TABLET %d %d %s", this.tabletId, this.seq, this.tablet);
            }
        }
        throw new RuntimeException("Unknown type of entry: " + this.event);
    }

    public Key toKey() throws IOException {
        String family = this.event.name();
        KeyBuilder.RowStep kb = Key.builder();
        switch (this.event) {
            case OPEN: {
                byte[] formattedRow = this.formatRow(0, 0L);
                return kb.row(formattedRow).family((CharSequence)family).qualifier((CharSequence)this.tserverSession).build();
            }
            case COMPACTION_START: {
                byte[] formattedRow = this.formatRow(this.tabletId, this.seq);
                return kb.row(formattedRow).family((CharSequence)family).qualifier((CharSequence)this.filename).build();
            }
            case COMPACTION_FINISH: 
            case MANY_MUTATIONS: 
            case MUTATION: {
                return kb.row(this.formatRow(this.tabletId, this.seq)).family((CharSequence)family).build();
            }
            case DEFINE_TABLET: {
                byte[] formattedRow = this.formatRow(this.tabletId, this.seq);
                DataOutputBuffer buffer = new DataOutputBuffer();
                this.tablet.writeTo((DataOutput)buffer);
                byte[] q = Arrays.copyOf(buffer.getData(), buffer.getLength());
                buffer.close();
                return kb.row(formattedRow).family((CharSequence)family).qualifier(q).build();
            }
        }
        throw new AssertionError((Object)("Invalid event type in LogFileKey: " + this.event));
    }

    private byte getEventByte() {
        int eventTypeInteger = LogFileKey.eventType(this.event);
        return (byte)(eventTypeInteger & 0xFF);
    }

    private Text formatRow() {
        return new Text(this.formatRow(this.tabletId, this.seq));
    }

    private byte[] formatRow(int tabletId, long seq) {
        byte eventNum = this.getEventByte();
        Preconditions.checkArgument((eventNum >= 0 && seq >= 0L ? 1 : 0) != 0);
        byte[] row = new byte[13];
        int encodedTabletId = tabletId ^ Integer.MIN_VALUE;
        int mask = 255;
        row[0] = eventNum;
        row[1] = (byte)(encodedTabletId >>> 24 & mask);
        row[2] = (byte)(encodedTabletId >>> 16 & mask);
        row[3] = (byte)(encodedTabletId >>> 8 & mask);
        row[4] = (byte)(encodedTabletId & mask);
        row[5] = (byte)(seq >>> 56 & (long)mask);
        row[6] = (byte)(seq >>> 48 & (long)mask);
        row[7] = (byte)(seq >>> 40 & (long)mask);
        row[8] = (byte)(seq >>> 32 & (long)mask);
        row[9] = (byte)(seq >>> 24 & (long)mask);
        row[10] = (byte)(seq >>> 16 & (long)mask);
        row[11] = (byte)(seq >>> 8 & (long)mask);
        row[12] = (byte)(seq & (long)mask);
        log.trace("Convert {} {} {} to row {}", new Object[]{this.event, tabletId, seq, Arrays.toString(row)});
        return row;
    }

    private static int getTabletId(byte[] row) {
        int mask = 255;
        int encoded = (row[1] & mask) << 24 | (row[2] & mask) << 16 | (row[3] & mask) << 8 | row[4] & mask;
        return encoded ^ Integer.MIN_VALUE;
    }

    private static long getSequence(byte[] row) {
        long mask = 255L;
        return ((long)row[5] & mask) << 56 | ((long)row[6] & mask) << 48 | ((long)row[7] & mask) << 40 | ((long)row[8] & mask) << 32 | ((long)row[9] & mask) << 24 | ((long)row[10] & mask) << 16 | ((long)row[11] & mask) << 8 | (long)row[12] & mask;
    }

    public static Range toRange(LogFileKey start, LogFileKey end) {
        return new Range(start.formatRow(), end.formatRow());
    }

    public static LogFileKey fromKey(Key key) {
        LogFileKey logFileKey = new LogFileKey();
        byte[] rowParts = key.getRow().getBytes();
        logFileKey.tabletId = LogFileKey.getTabletId(rowParts);
        logFileKey.seq = LogFileKey.getSequence(rowParts);
        logFileKey.event = LogEvents.valueOf(key.getColumnFamilyData().toString());
        if (LogFileKey.eventType(logFileKey.event) != rowParts[0]) {
            throw new AssertionError((Object)("Event in row differs from column family. Key: " + key));
        }
        log.trace("From row {} get {} {} {}", new Object[]{Arrays.toString(rowParts), logFileKey.event, logFileKey.tabletId, logFileKey.seq});
        switch (logFileKey.event) {
            case OPEN: {
                logFileKey.tserverSession = key.getColumnQualifierData().toString();
                break;
            }
            case COMPACTION_START: {
                logFileKey.filename = key.getColumnQualifierData().toString();
                break;
            }
            case DEFINE_TABLET: {
                try (DataInputBuffer buffer = new DataInputBuffer();){
                    byte[] bytes = key.getColumnQualifierData().toArray();
                    buffer.reset(bytes, bytes.length);
                    logFileKey.tablet = KeyExtent.readFrom((DataInput)buffer);
                    break;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            case COMPACTION_FINISH: 
            case MANY_MUTATIONS: 
            case MUTATION: {
                break;
            }
            default: {
                throw new AssertionError((Object)("Invalid event type in key: " + key));
            }
        }
        return logFileKey;
    }
}

