/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.util;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.google.auto.service.AutoService;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.accumulo.core.cli.Help;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.conf.SiteConfiguration;
import org.apache.accumulo.core.data.InstanceId;
import org.apache.accumulo.core.fate.zookeeper.ServiceLock;
import org.apache.accumulo.core.fate.zookeeper.ZooReaderWriter;
import org.apache.accumulo.core.fate.zookeeper.ZooUtil;
import org.apache.accumulo.core.singletons.SingletonManager;
import org.apache.accumulo.core.util.HostAndPort;
import org.apache.accumulo.core.volume.VolumeConfiguration;
import org.apache.accumulo.server.fs.VolumeManager;
import org.apache.accumulo.server.security.SecurityUtil;
import org.apache.accumulo.start.spi.KeywordExecutable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@AutoService(value={KeywordExecutable.class})
public class ZooZap
implements KeywordExecutable {
    private static final Logger log = LoggerFactory.getLogger(ZooZap.class);

    private static void message(String msg, Opts opts) {
        if (opts.verbose) {
            System.out.println(msg);
        }
    }

    public String keyword() {
        return "zoo-zap";
    }

    public String description() {
        return "Utility for zapping Zookeeper locks";
    }

    public static void main(String[] args) throws Exception {
        new ZooZap().execute(args);
    }

    public void execute(String[] args) throws Exception {
        try {
            SiteConfiguration siteConf = SiteConfiguration.auto();
            if (siteConf.getBoolean(Property.INSTANCE_RPC_SASL_ENABLED)) {
                SecurityUtil.serverLogin((AccumuloConfiguration)siteConf);
            }
            this.zap(siteConf, args);
        }
        finally {
            SingletonManager.setMode((SingletonManager.Mode)SingletonManager.Mode.CLOSED);
        }
    }

    public void zap(SiteConfiguration siteConf, String ... args) {
        Predicate<String> groupPredicate;
        Predicate<HostAndPort> hostPortPredicate;
        Opts opts = new Opts();
        opts.parseArgs(this.keyword(), args, new Object[0]);
        if (opts.hostPortExcludeFile != null) {
            try {
                Set hostPorts = Files.lines(java.nio.file.Path.of(opts.hostPortExcludeFile, new String[0])).map(String::trim).map(HostAndPort::fromString).collect(Collectors.toSet());
                hostPortPredicate = hp -> !hostPorts.contains(hp);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        } else {
            hostPortPredicate = hp -> true;
        }
        if (opts.includeGroups != null) {
            Set groups = Arrays.stream(opts.includeGroups.split(",")).map(String::trim).collect(Collectors.toSet());
            groupPredicate = groups::contains;
        } else {
            groupPredicate = g -> true;
        }
        if (!(opts.zapMaster || opts.zapManager || opts.zapTservers || opts.zapCompactors || opts.zapCoordinators || opts.zapScanServers || opts.zapGc || opts.zapMonitor)) {
            new JCommander((Object)opts).usage();
            return;
        }
        String volDir = (String)VolumeConfiguration.getVolumeUris((AccumuloConfiguration)siteConf).iterator().next();
        Path instanceDir = new Path(volDir, "instance_id");
        InstanceId iid = VolumeManager.getInstanceIDFromHdfs(instanceDir, new Configuration());
        ZooReaderWriter zoo = new ZooReaderWriter((AccumuloConfiguration)siteConf);
        if (opts.zapMaster) {
            log.warn("The -master option is deprecated. Please use -manager instead.");
        }
        if (opts.zapManager || opts.zapMaster) {
            String managerLockPath = "/accumulo/" + String.valueOf(iid) + "/managers/lock";
            try {
                ZooZap.removeSingletonLock(zoo, managerLockPath, hostPortPredicate, opts);
            }
            catch (InterruptedException | KeeperException e) {
                log.error("Error deleting manager lock", e);
            }
        }
        if (opts.zapGc) {
            String gcLockPath = "/accumulo/" + String.valueOf(iid) + "/gc/lock";
            try {
                ZooZap.removeSingletonLock(zoo, gcLockPath, hostPortPredicate, opts);
            }
            catch (InterruptedException | KeeperException e) {
                log.error("Error deleting manager lock", e);
            }
        }
        if (opts.zapMonitor) {
            String monitorLockPath = "/accumulo/" + String.valueOf(iid) + "/monitor/lock";
            try {
                ZooZap.removeSingletonLock(zoo, monitorLockPath, hostPortPredicate, opts);
            }
            catch (InterruptedException | KeeperException e) {
                log.error("Error deleting monitor lock", e);
            }
        }
        if (opts.zapTservers) {
            String tserversPath = "/accumulo/" + String.valueOf(iid) + "/tservers";
            try {
                if ((opts.zapManager || opts.zapMaster) && opts.hostPortExcludeFile == null && !opts.dryRun) {
                    List children = zoo.getChildren(tserversPath);
                    for (String child : children) {
                        ZooZap.message("Deleting " + tserversPath + "/" + child + " from zookeeper", opts);
                        zoo.recursiveDelete(tserversPath + "/" + child, ZooUtil.NodeMissingPolicy.SKIP);
                    }
                } else {
                    ZooZap.removeLocks(zoo, tserversPath, hostPortPredicate, opts);
                }
            }
            catch (InterruptedException | KeeperException e) {
                log.error("Error deleting tserver locks", e);
            }
        }
        String path = siteConf.get(Property.TRACE_ZK_PATH);
        try {
            ZooZap.zapDirectory(zoo, path, opts);
        }
        catch (Exception e) {
            // empty catch block
        }
        if (opts.zapCoordinators) {
            String coordinatorPath = "/accumulo/" + String.valueOf(iid) + "/coordinators/lock";
            try {
                ZooZap.removeSingletonLock(zoo, coordinatorPath, hostPortPredicate, opts);
            }
            catch (InterruptedException | KeeperException e) {
                log.error("Error deleting coordinator from zookeeper", e);
            }
        }
        if (opts.zapCompactors) {
            String compactorsBasepath = "/accumulo/" + String.valueOf(iid) + "/compactors";
            try {
                ZooZap.removeGroupedLocks(zoo, compactorsBasepath, groupPredicate, hostPortPredicate, opts);
            }
            catch (InterruptedException | KeeperException e) {
                log.error("Error deleting compactors from zookeeper", e);
            }
        }
        if (opts.zapScanServers) {
            String sserversPath = "/accumulo/" + String.valueOf(iid) + "/sservers";
            try {
                ZooZap.removeGroupedLocks(zoo, sserversPath, groupPredicate, hostPortPredicate, opts);
            }
            catch (InterruptedException | KeeperException e) {
                log.error("Error deleting scan server locks", e);
            }
        }
    }

    private static void zapDirectory(ZooReaderWriter zoo, String path, Opts opts) throws KeeperException, InterruptedException {
        List children = zoo.getChildren(path);
        for (String child : children) {
            ZooZap.message("Deleting " + path + "/" + child + " from zookeeper", opts);
            if (opts.dryRun) continue;
            zoo.recursiveDelete(path + "/" + child, ZooUtil.NodeMissingPolicy.SKIP);
        }
    }

    static void removeGroupedLocks(ZooReaderWriter zoo, String path, Predicate<String> groupPredicate, Predicate<HostAndPort> hostPortPredicate, Opts opts) throws KeeperException, InterruptedException {
        if (zoo.exists(path)) {
            List groups = zoo.getChildren(path);
            for (String group : groups) {
                if (!groupPredicate.test(group)) continue;
                ZooZap.removeLocks(zoo, path + "/" + group, hostPortPredicate, opts);
            }
        }
    }

    static void removeLocks(ZooReaderWriter zoo, String path, Predicate<HostAndPort> hostPortPredicate, Opts opts) throws KeeperException, InterruptedException {
        if (zoo.exists(path)) {
            List children = zoo.getChildren(path);
            for (String child : children) {
                if (!hostPortPredicate.test(HostAndPort.fromString((String)child))) continue;
                ZooZap.message("Deleting " + path + "/" + child + " from zookeeper", opts);
                if (opts.dryRun) continue;
                zoo.recursiveDelete(path + "/" + child, ZooUtil.NodeMissingPolicy.SKIP);
            }
        }
    }

    static void removeSingletonLock(ZooReaderWriter zoo, String path, Predicate<HostAndPort> hostPortPredicate, Opts ops) throws KeeperException, InterruptedException {
        byte[] lockData = ServiceLock.getLockData((ZooKeeper)zoo.getZooKeeper(), (ServiceLock.ServiceLockPath)ServiceLock.path((String)path));
        if (lockData != null && hostPortPredicate.test(HostAndPort.fromString((String)new String(lockData, StandardCharsets.UTF_8)))) {
            ZooZap.zapDirectory(zoo, path, ops);
        }
    }

    static class Opts
    extends Help {
        @Deprecated(since="2.1.0")
        @Parameter(names={"-master"}, description="remove master locks (deprecated -- user -manager instead")
        boolean zapMaster = false;
        @Parameter(names={"-manager"}, description="remove manager locks")
        boolean zapManager = false;
        @Parameter(names={"-tservers"}, description="remove tablet server locks")
        boolean zapTservers = false;
        @Parameter(names={"-compaction-coordinators"}, description="remove compaction coordinator locks")
        boolean zapCoordinators = false;
        @Parameter(names={"-compactors"}, description="remove compactor locks")
        boolean zapCompactors = false;
        @Parameter(names={"-sservers"}, description="remove scan server locks")
        boolean zapScanServers = false;
        @Parameter(names={"--gc"}, description="remove gc server locks")
        boolean zapGc = false;
        @Parameter(names={"--monitor"}, description="remove monitor server locks")
        boolean zapMonitor = false;
        @Parameter(names={"-verbose"}, description="print out messages about progress")
        boolean verbose = false;
        @Parameter(names={"--include-groups"}, description="Comma seperated list of resource groups to include")
        String includeGroups;
        @Parameter(names={"--exclude-host-ports"}, description="File with lines of <host>:<port> to exclude from removal")
        String hostPortExcludeFile;
        @Parameter(names={"--dry-run"}, description="Only print changes that would be made w/o actually making any change")
        boolean dryRun = false;

        Opts() {
        }
    }
}

