/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.manager.upgrade;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;
import org.apache.accumulo.core.client.AccumuloException;
import org.apache.accumulo.core.client.AccumuloSecurityException;
import org.apache.accumulo.core.client.NamespaceNotFoundException;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.ConfigCheckUtil;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.fate.ZooStore;
import org.apache.accumulo.core.util.threads.ThreadPoolNames;
import org.apache.accumulo.core.util.threads.ThreadPools;
import org.apache.accumulo.core.volume.Volume;
import org.apache.accumulo.manager.EventCoordinator;
import org.apache.accumulo.manager.upgrade.Upgrader;
import org.apache.accumulo.manager.upgrade.Upgrader8to9;
import org.apache.accumulo.manager.upgrade.Upgrader9to10;
import org.apache.accumulo.server.AccumuloDataVersion;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.ServerDirs;
import org.apache.accumulo.server.conf.CheckCompactionConfig;
import org.apache.accumulo.server.fs.VolumeManager;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UpgradeCoordinator {
    private static final Logger log = LoggerFactory.getLogger(UpgradeCoordinator.class);
    private int currentVersion;
    private final Map<Integer, Upgrader> upgraders = Collections.unmodifiableMap(new TreeMap<Integer, Upgrader9to10>(Map.of(8, new Upgrader8to9(), 9, new Upgrader9to10())));
    private volatile UpgradeStatus status = UpgradeStatus.INITIAL;

    private void setStatus(UpgradeStatus status, EventCoordinator eventCoordinator) {
        UpgradeStatus oldStatus = this.status;
        this.status = status;
        eventCoordinator.event("Upgrade status changed from %s to %s", new Object[]{oldStatus, status});
    }

    @SuppressFBWarnings(value={"DM_EXIT"}, justification="Want to immediately stop all manager threads on upgrade error")
    private void handleFailure(Exception e) {
        log.error("FATAL: Error performing upgrade", (Throwable)e);
        this.status = UpgradeStatus.FAILED;
        System.exit(1);
    }

    public synchronized void upgradeZookeeper(ServerContext context, EventCoordinator eventCoordinator) {
        Preconditions.checkState((this.status == UpgradeStatus.INITIAL ? 1 : 0) != 0, (String)"Not currently in a suitable state to do zookeeper upgrade %s", (Object)((Object)this.status));
        try {
            int cv;
            this.currentVersion = cv = AccumuloDataVersion.getCurrentVersion((ServerContext)context);
            if (cv == AccumuloDataVersion.get()) {
                this.status = UpgradeStatus.COMPLETE;
                return;
            }
            if (this.currentVersion < AccumuloDataVersion.get()) {
                this.abortIfFateTransactions(context);
                for (int v = this.currentVersion; v < AccumuloDataVersion.get(); ++v) {
                    log.info("Upgrading Zookeeper from data version {}", (Object)v);
                    this.upgraders.get(v).upgradeZookeeper(context);
                }
            }
            this.setStatus(UpgradeStatus.UPGRADED_ZOOKEEPER, eventCoordinator);
        }
        catch (Exception e) {
            this.handleFailure(e);
        }
    }

    public synchronized Future<Void> upgradeMetadata(ServerContext context, EventCoordinator eventCoordinator) {
        if (this.status == UpgradeStatus.COMPLETE) {
            return CompletableFuture.completedFuture(null);
        }
        Preconditions.checkState((this.status == UpgradeStatus.UPGRADED_ZOOKEEPER ? 1 : 0) != 0, (String)"Not currently in a suitable state to do metadata upgrade %s", (Object)((Object)this.status));
        if (this.currentVersion < AccumuloDataVersion.get()) {
            return ThreadPools.getServerThreadPools().getPoolBuilder(ThreadPoolNames.MANAGER_UPGRADE_COORDINATOR_METADATA_POOL).numCoreThreads(0).numMaxThreads(Integer.MAX_VALUE).withTimeOut(60L, TimeUnit.SECONDS).withQueue(new SynchronousQueue()).build().submit(() -> {
                try {
                    int v;
                    for (v = this.currentVersion; v < AccumuloDataVersion.get(); ++v) {
                        log.info("Upgrading Root from data version {}", (Object)v);
                        this.upgraders.get(v).upgradeRoot(context);
                    }
                    this.setStatus(UpgradeStatus.UPGRADED_ROOT, eventCoordinator);
                    for (v = this.currentVersion; v < AccumuloDataVersion.get(); ++v) {
                        log.info("Upgrading Metadata from data version {}", (Object)v);
                        this.upgraders.get(v).upgradeMetadata(context);
                    }
                    this.setStatus(UpgradeStatus.UPGRADED_METADATA, eventCoordinator);
                    log.info("Validating configuration properties.");
                    this.validateProperties(context);
                    log.info("Updating persistent data version.");
                    this.updateAccumuloVersion(context.getServerDirs(), context.getVolumeManager(), this.currentVersion);
                    log.info("Upgrade complete");
                    this.setStatus(UpgradeStatus.COMPLETE, eventCoordinator);
                }
                catch (Exception e) {
                    this.handleFailure(e);
                }
                return null;
            });
        }
        return CompletableFuture.completedFuture(null);
    }

    private void validateProperties(ServerContext context) {
        ConfigCheckUtil.validate((Iterable)context.getSiteConfiguration(), (String)"site configuration");
        ConfigCheckUtil.validate((Iterable)context.getConfiguration(), (String)"system configuration");
        try {
            for (String ns : context.namespaceOperations().list()) {
                ConfigCheckUtil.validate(context.namespaceOperations().getNamespaceProperties(ns).entrySet(), (String)(ns + " namespace configuration"));
            }
            for (String table : context.tableOperations().list()) {
                ConfigCheckUtil.validate(context.tableOperations().getTableProperties(table).entrySet(), (String)(table + " table configuration"));
            }
        }
        catch (AccumuloException | AccumuloSecurityException | NamespaceNotFoundException | TableNotFoundException e) {
            throw new IllegalStateException("Error checking properties", e);
        }
        try {
            CheckCompactionConfig.validate((AccumuloConfiguration)context.getConfiguration());
        }
        catch (IllegalArgumentException | ReflectiveOperationException | SecurityException e) {
            throw new IllegalStateException("Error validating compaction configuration", e);
        }
    }

    synchronized void updateAccumuloVersion(ServerDirs serverDirs, VolumeManager fs, int oldVersion) {
        for (Volume volume : fs.getVolumes()) {
            try {
                if (serverDirs.getAccumuloPersistentVersion(volume) != oldVersion) continue;
                log.debug("Attempting to upgrade {}", (Object)volume);
                Path dataVersionLocation = serverDirs.getDataVersionLocation(volume);
                fs.create(new Path(dataVersionLocation, Integer.toString(AccumuloDataVersion.get()))).close();
                Path prevDataVersionLoc = new Path(dataVersionLocation, Integer.toString(oldVersion));
                if (fs.delete(prevDataVersionLoc)) continue;
                throw new RuntimeException("Could not delete previous data version location (" + String.valueOf(prevDataVersionLoc) + ") for " + String.valueOf(volume));
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to set accumulo version: an error occurred.", e);
            }
        }
    }

    public UpgradeStatus getStatus() {
        return this.status;
    }

    @SuppressFBWarnings(value={"DM_EXIT"}, justification="Want to immediately stop all manager threads on upgrade error")
    private void abortIfFateTransactions(ServerContext context) {
        try {
            ZooStore fate = new ZooStore(context.getZooKeeperRoot() + "/fate", context.getZooReaderWriter(), context.getZooCache());
            if (!fate.list().isEmpty()) {
                throw new AccumuloException("Aborting upgrade because there are outstanding FATE transactions from a previous Accumulo version. You can start the tservers and then use the shell to delete completed  transactions. If there are uncomplete transactions, you will need to roll back and fix those issues. Please see the following page for more information:  https://accumulo.apache.org/docs/2.x/troubleshooting/advanced#upgrade-issues");
            }
        }
        catch (Exception exception) {
            log.error("Problem verifying Fate readiness", (Throwable)exception);
            System.exit(1);
        }
    }

    public static enum UpgradeStatus {
        INITIAL{

            @Override
            public boolean isParentLevelUpgraded(KeyExtent extent) {
                return false;
            }
        }
        ,
        UPGRADED_ZOOKEEPER{

            @Override
            public boolean isParentLevelUpgraded(KeyExtent extent) {
                return extent.isRootTablet();
            }
        }
        ,
        UPGRADED_ROOT{

            @Override
            public boolean isParentLevelUpgraded(KeyExtent extent) {
                return extent.isMeta();
            }
        }
        ,
        UPGRADED_METADATA{

            @Override
            public boolean isParentLevelUpgraded(KeyExtent extent) {
                return extent.isMeta();
            }
        }
        ,
        COMPLETE{

            @Override
            public boolean isParentLevelUpgraded(KeyExtent extent) {
                return true;
            }
        }
        ,
        FAILED{

            @Override
            public boolean isParentLevelUpgraded(KeyExtent extent) {
                return false;
            }
        };


        public abstract boolean isParentLevelUpgraded(KeyExtent var1);
    }
}

