/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.gc;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterators;
import com.google.common.collect.Maps;
import com.google.protobuf.InvalidProtocolBufferException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.apache.accumulo.core.client.AccumuloClient;
import org.apache.accumulo.core.client.IsolatedScanner;
import org.apache.accumulo.core.client.Scanner;
import org.apache.accumulo.core.client.ScannerBase;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.fate.zookeeper.ZooReader;
import org.apache.accumulo.core.gc.GcCandidate;
import org.apache.accumulo.core.gc.Reference;
import org.apache.accumulo.core.gc.ReferenceFile;
import org.apache.accumulo.core.manager.state.tables.TableState;
import org.apache.accumulo.core.metadata.MetadataTable;
import org.apache.accumulo.core.metadata.RootTable;
import org.apache.accumulo.core.metadata.ValidationUtil;
import org.apache.accumulo.core.metadata.schema.Ample;
import org.apache.accumulo.core.metadata.schema.MetadataSchema;
import org.apache.accumulo.core.metadata.schema.TabletMetadata;
import org.apache.accumulo.core.metadata.schema.TabletsMetadata;
import org.apache.accumulo.core.replication.ReplicationSchema;
import org.apache.accumulo.core.replication.ReplicationTable;
import org.apache.accumulo.core.replication.ReplicationTableOfflineException;
import org.apache.accumulo.core.security.Authorizations;
import org.apache.accumulo.core.util.UtilWaitThread;
import org.apache.accumulo.core.util.threads.ThreadPools;
import org.apache.accumulo.core.volume.Volume;
import org.apache.accumulo.gc.GarbageCollectionEnvironment;
import org.apache.accumulo.gc.SimpleGarbageCollector;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.fs.VolumeManager;
import org.apache.accumulo.server.fs.VolumeUtil;
import org.apache.accumulo.server.gc.GcVolumeUtil;
import org.apache.accumulo.server.replication.proto.Replication;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GCRun
implements GarbageCollectionEnvironment {
    private final Logger log;
    private static final String fileActionPrefix = "FILE-ACTION:";
    private final Ample.DataLevel level;
    private final ServerContext context;
    private final AccumuloConfiguration config;
    private long candidates = 0L;
    private long inUse = 0L;
    private long deleted = 0L;
    private long errors = 0L;

    public GCRun(Ample.DataLevel level, ServerContext context) {
        this.log = LoggerFactory.getLogger((String)(GCRun.class.getName() + "." + level.name()));
        this.level = level;
        this.context = context;
        this.config = context.getConfiguration();
    }

    @Override
    public Iterator<GcCandidate> getCandidates() {
        return this.context.getAmple().getGcCandidates(this.level);
    }

    @Override
    public void deleteGcCandidates(Collection<GcCandidate> gcCandidates, Ample.GcCandidateType type) {
        if (this.inSafeMode()) {
            System.out.println("SAFEMODE: There are " + gcCandidates.size() + " reference file gcCandidates entries marked for deletion from " + this.level + " of type: " + type + ".\n          Examine the log files to identify them.\n");
            this.log.info("SAFEMODE: Listing all ref file gcCandidates for deletion");
            for (GcCandidate gcCandidate : gcCandidates) {
                this.log.info("SAFEMODE: {}", (Object)gcCandidate);
            }
            this.log.info("SAFEMODE: End reference candidates for deletion");
            return;
        }
        this.log.info("Attempting to delete gcCandidates of type {} from metadata", (Object)type);
        this.context.getAmple().deleteGcCandidates(this.level, gcCandidates, type);
    }

    @Override
    public List<GcCandidate> readCandidatesThatFitInMemory(Iterator<GcCandidate> candidates) {
        long candidateLength = 0L;
        long candidateBatchSize = this.getCandidateBatchSize() / 2L;
        ArrayList<GcCandidate> candidatesBatch = new ArrayList<GcCandidate>();
        while (candidates.hasNext()) {
            GcCandidate candidate = candidates.next();
            candidatesBatch.add(candidate);
            if ((candidateLength += (long)candidate.getPath().length()) <= candidateBatchSize) continue;
            this.log.info("Candidate batch of size {} has exceeded the threshold. Attempting to delete what has been gathered so far.", (Object)candidateLength);
            return candidatesBatch;
        }
        return candidatesBatch;
    }

    @Override
    public Stream<String> getBlipPaths() throws TableNotFoundException {
        if (this.level == Ample.DataLevel.ROOT) {
            return Stream.empty();
        }
        int blipPrefixLen = MetadataSchema.BlipSection.getRowPrefix().length();
        IsolatedScanner scanner = new IsolatedScanner(this.context.createScanner(this.level.metaTable(), Authorizations.EMPTY));
        scanner.setRange(MetadataSchema.BlipSection.getRange());
        return (Stream)scanner.stream().map(entry -> ((Key)entry.getKey()).getRow().toString().substring(blipPrefixLen)).onClose(() -> ((IsolatedScanner)scanner).close());
    }

    @Override
    public Stream<Reference> getReferences() {
        Stream tabletStream;
        if (this.level == Ample.DataLevel.ROOT) {
            tabletStream = Stream.of(this.context.getAmple().readTablet(RootTable.EXTENT, new TabletMetadata.ColumnType[]{TabletMetadata.ColumnType.DIR, TabletMetadata.ColumnType.FILES, TabletMetadata.ColumnType.SCANS}));
        } else {
            TabletsMetadata tabletsMetadata = TabletsMetadata.builder((AccumuloClient)this.context).scanTable(this.level.metaTable()).checkConsistency().fetch(new TabletMetadata.ColumnType[]{TabletMetadata.ColumnType.DIR, TabletMetadata.ColumnType.FILES, TabletMetadata.ColumnType.SCANS}).build();
            tabletStream = tabletsMetadata.stream();
        }
        Stream tabletReferences = tabletStream.flatMap(tm -> {
            TableId tableId = tm.getTableId();
            this.log.trace("tablet metadata table id: {}, end row:{}, dir:{}, saw: {}, prev row: {}", new Object[]{tableId, tm.getEndRow(), tm.getDirName(), tm.sawPrevEndRow(), tm.getPrevEndRow()});
            if (tm.getDirName() == null || tm.getDirName().isEmpty() || !tm.sawPrevEndRow()) {
                throw new IllegalStateException("possible incomplete metadata scan for table id: " + tableId + ", end row: " + tm.getEndRow() + ", dir: " + tm.getDirName() + ", saw prev row: " + tm.sawPrevEndRow());
            }
            Stream stfStream = tm.getFiles().stream();
            Stream<ReferenceFile> fileStream = stfStream.map(f -> ReferenceFile.forFile((TableId)tableId, (String)f.getMetaUpdateDelete()));
            List tmScans = tm.getScans();
            if (!tmScans.isEmpty()) {
                Stream<ReferenceFile> scanStream = tmScans.stream().map(s -> ReferenceFile.forScan((TableId)tableId, (String)s.getMetaUpdateDelete()));
                fileStream = Stream.concat(fileStream, scanStream);
            }
            if (tm.getDirName() != null) {
                ReferenceFile tabletDir = ReferenceFile.forDirectory((TableId)tableId, (String)tm.getDirName());
                fileStream = Stream.concat(fileStream, Stream.of(tabletDir));
            }
            return fileStream;
        });
        Stream<ReferenceFile> scanServerRefs = this.context.getAmple().getScanServerFileReferences().map(sfr -> ReferenceFile.forScan((TableId)sfr.getTableId(), (String)sfr.getPathStr()));
        return Stream.concat(tabletReferences, scanServerRefs);
    }

    @Override
    public Map<TableId, TableState> getTableIDs() throws InterruptedException {
        String tablesPath = this.context.getZooKeeperRoot() + "/tables";
        ZooReader zr = this.context.getZooReader();
        int retries = 1;
        Throwable ioe = null;
        while (retries <= 10) {
            try {
                zr.sync(tablesPath);
                HashMap<TableId, TableState> tids = new HashMap<TableId, TableState>();
                for (String table : zr.getChildren(tablesPath)) {
                    TableId tableId = TableId.of((String)table);
                    TableState tableState = null;
                    String statePath = this.context.getZooKeeperRoot() + "/tables/" + tableId.canonical() + "/state";
                    try {
                        byte[] state = zr.getData(statePath);
                        tableState = state == null ? TableState.UNKNOWN : TableState.valueOf((String)new String(state, StandardCharsets.UTF_8));
                    }
                    catch (KeeperException.NoNodeException e) {
                        tableState = TableState.UNKNOWN;
                    }
                    tids.put(tableId, tableState);
                }
                return tids;
            }
            catch (KeeperException e) {
                ++retries;
                if (ioe == null) {
                    ioe = new IllegalStateException("Error getting table ids from ZooKeeper");
                }
                ioe.addSuppressed(e);
                this.log.error("Error getting tables from ZooKeeper, retrying in {} seconds", (Object)retries, (Object)e);
                UtilWaitThread.sleepUninterruptibly((long)retries, (TimeUnit)TimeUnit.SECONDS);
            }
        }
        throw ioe;
    }

    @Override
    public void deleteConfirmedCandidates(SortedMap<String, GcCandidate> confirmedDeletes) throws TableNotFoundException {
        String metadataLocation;
        VolumeManager fs = this.context.getVolumeManager();
        String string = metadataLocation = this.level == Ample.DataLevel.ROOT ? this.context.getZooKeeperRoot() + " for " + RootTable.NAME : this.level.metaTable();
        if (this.inSafeMode()) {
            System.out.println("SAFEMODE: There are " + confirmedDeletes.size() + " data file candidates marked for deletion in " + metadataLocation + ".\n          Examine the log files to identify them.\n");
            this.log.info("{} SAFEMODE: Listing all data file candidates for deletion", (Object)fileActionPrefix);
            for (GcCandidate candidate : confirmedDeletes.values()) {
                this.log.info("{} SAFEMODE: {}", (Object)fileActionPrefix, (Object)candidate);
            }
            this.log.info("SAFEMODE: End candidates for deletion");
            return;
        }
        List<GcCandidate> processedDeletes = Collections.synchronizedList(new ArrayList());
        GCRun.minimizeDeletes(confirmedDeletes, processedDeletes, fs, this.log);
        ThreadPoolExecutor deleteThreadPool = ThreadPools.getServerThreadPools().createExecutorService(this.config, Property.GC_DELETE_THREADS);
        List replacements = this.context.getVolumeReplacements();
        for (GcCandidate delete : confirmedDeletes.values()) {
            Runnable deleteTask = () -> {
                boolean removeFlag = false;
                try {
                    Path fullPath;
                    Path switchedDelete = VolumeUtil.switchVolume((String)delete.getPath(), (VolumeManager.FileType)VolumeManager.FileType.TABLE, (List)replacements);
                    if (switchedDelete != null) {
                        this.log.debug("Volume replaced {} -> {}", (Object)delete.getPath(), (Object)switchedDelete);
                        fullPath = ValidationUtil.validate((Path)switchedDelete);
                    } else {
                        fullPath = new Path(ValidationUtil.validate((String)delete.getPath()));
                    }
                    for (Path pathToDel : GcVolumeUtil.expandAllVolumesUri((VolumeManager)fs, (Path)fullPath)) {
                        this.log.debug("{} Deleting {}", (Object)fileActionPrefix, (Object)pathToDel);
                        if (this.moveToTrash(pathToDel) || fs.deleteRecursively(pathToDel)) {
                            removeFlag = true;
                            ++this.deleted;
                            continue;
                        }
                        if (fs.exists(pathToDel)) {
                            removeFlag = false;
                            ++this.errors;
                            this.log.warn("{} File exists, but was not deleted for an unknown reason: {}", (Object)fileActionPrefix, (Object)pathToDel);
                            break;
                        }
                        removeFlag = true;
                        ++this.errors;
                        String[] parts = pathToDel.toString().split("/tables")[1].split("/");
                        if (parts.length > 2) {
                            TableId tableId = TableId.of((String)parts[1]);
                            String tabletDir = parts[2];
                            this.context.getTableManager().updateTableStateCache(tableId);
                            TableState tableState = this.context.getTableManager().getTableState(tableId);
                            if (tableState == null || tableState == TableState.DELETING || tabletDir.startsWith("c-")) continue;
                            this.log.debug("{} File doesn't exist: {}", (Object)fileActionPrefix, (Object)pathToDel);
                            continue;
                        }
                        this.log.warn("{} Delete failed due to invalid file path format: {}", (Object)fileActionPrefix, (Object)delete.getPath());
                    }
                    if (removeFlag) {
                        processedDeletes.add(delete);
                    }
                }
                catch (Exception e) {
                    this.log.error("{} Exception while deleting files ", (Object)fileActionPrefix, (Object)e);
                }
            };
            deleteThreadPool.execute(deleteTask);
        }
        deleteThreadPool.shutdown();
        try {
            while (!deleteThreadPool.awaitTermination(1000L, TimeUnit.MILLISECONDS)) {
            }
        }
        catch (InterruptedException e1) {
            this.log.error("{}", (Object)e1.getMessage(), (Object)e1);
        }
        this.deleteGcCandidates(processedDeletes, Ample.GcCandidateType.VALID);
    }

    @Override
    public void deleteTableDirIfEmpty(TableId tableID) throws IOException {
        VolumeManager fs = this.context.getVolumeManager();
        for (String dir : this.context.getTablesDirs()) {
            FileStatus[] tabletDirs;
            try {
                tabletDirs = fs.listStatus(new Path(dir + "/" + tableID));
            }
            catch (FileNotFoundException ex) {
                continue;
            }
            if (tabletDirs.length != 0) continue;
            Path p = new Path(dir + "/" + tableID);
            this.log.debug("{} Removing table dir {}", (Object)fileActionPrefix, (Object)p);
            if (this.moveToTrash(p)) continue;
            fs.delete(p);
        }
    }

    @Override
    public void incrementCandidatesStat(long i) {
        this.candidates += i;
    }

    @Override
    public void incrementInUseStat(long i) {
        this.inUse += i;
    }

    @Override
    @Deprecated
    public Iterator<Map.Entry<String, Replication.Status>> getReplicationNeededIterator() {
        ServerContext client = this.context;
        try {
            Scanner s = ReplicationTable.getScanner((AccumuloClient)client);
            ReplicationSchema.StatusSection.limit((ScannerBase)s);
            return Iterators.transform((Iterator)s.iterator(), input -> {
                Replication.Status stat;
                String file = ((Key)input.getKey()).getRow().toString();
                try {
                    stat = Replication.Status.parseFrom((byte[])((Value)input.getValue()).get());
                }
                catch (InvalidProtocolBufferException e) {
                    this.log.warn("Could not deserialize protobuf for: {}", input.getKey());
                    stat = null;
                }
                return Maps.immutableEntry((Object)file, (Object)stat);
            });
        }
        catch (ReplicationTableOfflineException e) {
            return Collections.emptyIterator();
        }
    }

    @VisibleForTesting
    static void minimizeDeletes(SortedMap<String, GcCandidate> confirmedDeletes, List<GcCandidate> processedDeletes, VolumeManager fs, Logger logger) {
        HashSet<Path> seenVolumes = new HashSet<Path>();
        Iterator<Map.Entry<String, GcCandidate>> cdIter = confirmedDeletes.entrySet().iterator();
        String lastDirRel = null;
        Path lastDirAbs = null;
        while (cdIter.hasNext()) {
            Map.Entry<String, GcCandidate> entry = cdIter.next();
            String relPath = entry.getKey();
            Path absPath = new Path(entry.getValue().getPath());
            if (SimpleGarbageCollector.isDir(relPath)) {
                lastDirRel = relPath;
                lastDirAbs = absPath;
                continue;
            }
            if (lastDirRel == null) continue;
            if (relPath.startsWith(lastDirRel)) {
                Path vol = VolumeManager.FileType.TABLE.getVolume(absPath);
                boolean sameVol = false;
                if (GcVolumeUtil.isAllVolumesUri((Path)lastDirAbs)) {
                    if (seenVolumes.contains(vol)) {
                        sameVol = true;
                    } else {
                        for (Volume cvol : fs.getVolumes()) {
                            if (!cvol.containsPath(vol)) continue;
                            seenVolumes.add(vol);
                            sameVol = true;
                        }
                    }
                } else {
                    sameVol = Objects.equals(VolumeManager.FileType.TABLE.getVolume(lastDirAbs), vol);
                }
                if (!sameVol) continue;
                logger.info("{} Ignoring {} because {} exist", new Object[]{fileActionPrefix, entry.getValue().getPath(), lastDirAbs});
                processedDeletes.add(entry.getValue());
                cdIter.remove();
                continue;
            }
            lastDirRel = null;
            lastDirAbs = null;
        }
    }

    boolean inSafeMode() {
        return this.context.getConfiguration().getBoolean(Property.GC_SAFEMODE);
    }

    @Override
    public boolean canRemoveInUseCandidates() {
        return this.context.getConfiguration().getBoolean(Property.GC_REMOVE_IN_USE_CANDIDATES);
    }

    boolean moveToTrash(Path path) throws IOException {
        VolumeManager fs = this.context.getVolumeManager();
        if (!this.isUsingTrash()) {
            this.log.trace("Accumulo Trash is disabled. Skipped for {}", (Object)path);
            return false;
        }
        try {
            boolean success = fs.moveToTrash(path);
            this.log.trace("Accumulo Trash enabled, moving to trash succeeded?: {}", (Object)success);
            return success;
        }
        catch (FileNotFoundException ex) {
            this.log.error("Error moving {} to trash", (Object)path, (Object)ex);
            return false;
        }
    }

    boolean isUsingTrash() {
        Property p = Property.GC_TRASH_IGNORE;
        return !this.config.getBoolean(p);
    }

    long getCandidateBatchSize() {
        return this.config.getAsBytes(Property.GC_CANDIDATE_BATCH_SIZE);
    }

    public long getInUseStat() {
        return this.inUse;
    }

    public long getDeletedStat() {
        return this.deleted;
    }

    public long getErrorsStat() {
        return this.errors;
    }

    public long getCandidatesStat() {
        return this.candidates;
    }

    @Override
    public Set<TableId> getCandidateTableIDs() throws InterruptedException {
        if (this.level == Ample.DataLevel.ROOT) {
            return Set.of(RootTable.ID);
        }
        if (this.level == Ample.DataLevel.METADATA) {
            return Set.of(MetadataTable.ID);
        }
        if (this.level == Ample.DataLevel.USER) {
            HashSet<TableId> tableIds = new HashSet<TableId>();
            this.getTableIDs().forEach((k, v) -> {
                if (v == TableState.ONLINE || v == TableState.OFFLINE) {
                    tableIds.add((TableId)k);
                }
            });
            tableIds.remove(MetadataTable.ID);
            tableIds.remove(RootTable.ID);
            return tableIds;
        }
        throw new IllegalArgumentException("Unexpected level in GC Env: " + this.level.name());
    }
}

