/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.util;

import com.beust.jcommander.Parameter;
import com.beust.jcommander.Parameters;
import com.google.common.annotations.VisibleForTesting;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.accumulo.core.fate.zookeeper.ZooReader;
import org.apache.accumulo.core.util.Pair;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.util.serviceStatus.ServiceStatusReport;
import org.apache.accumulo.server.util.serviceStatus.StatusSummary;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServiceStatusCmd {
    public static final String NO_GROUP_TAG = "NO_GROUP";
    private static final Logger LOG = LoggerFactory.getLogger(ServiceStatusCmd.class);

    public void execute(ServerContext context, Opts opts) {
        ZooReader zooReader = context.getZooReader();
        String zooRoot = context.getZooKeeperRoot();
        LOG.trace("zooRoot: {}", (Object)zooRoot);
        TreeMap<ServiceStatusReport.ReportKey, StatusSummary> services = new TreeMap<ServiceStatusReport.ReportKey, StatusSummary>();
        services.put(ServiceStatusReport.ReportKey.MANAGER, this.getManagerStatus(zooReader, zooRoot));
        services.put(ServiceStatusReport.ReportKey.MONITOR, this.getMonitorStatus(zooReader, zooRoot));
        services.put(ServiceStatusReport.ReportKey.T_SERVER, this.getTServerStatus(zooReader, zooRoot));
        services.put(ServiceStatusReport.ReportKey.S_SERVER, this.getScanServerStatus(zooReader, zooRoot));
        services.put(ServiceStatusReport.ReportKey.COORDINATOR, this.getCoordinatorStatus(zooReader, zooRoot));
        services.put(ServiceStatusReport.ReportKey.COMPACTOR, this.getCompactorStatus(zooReader, zooRoot));
        services.put(ServiceStatusReport.ReportKey.GC, this.getGcStatus(zooReader, zooRoot));
        ServiceStatusReport report = new ServiceStatusReport(services, opts.noHosts);
        if (opts.json) {
            System.out.println(report.toJson());
        } else {
            StringBuilder sb = new StringBuilder(8192);
            report.report(sb);
            System.out.println(sb);
        }
    }

    @VisibleForTesting
    StatusSummary getManagerStatus(ZooReader zooReader, String zRootPath) {
        String lockPath = zRootPath + "/managers/lock";
        return this.getStatusSummary(ServiceStatusReport.ReportKey.MANAGER, zooReader, lockPath);
    }

    @VisibleForTesting
    StatusSummary getMonitorStatus(ZooReader zooReader, String zRootPath) {
        String lockPath = zRootPath + "/monitor/lock";
        return this.getStatusSummary(ServiceStatusReport.ReportKey.MONITOR, zooReader, lockPath);
    }

    @VisibleForTesting
    StatusSummary getTServerStatus(ZooReader zooReader, String zRootPath) {
        String lockPath = zRootPath + "/tservers";
        return this.getServerHostStatus(zooReader, lockPath, ServiceStatusReport.ReportKey.T_SERVER);
    }

    @VisibleForTesting
    StatusSummary getScanServerStatus(ZooReader zooReader, String zRootPath) {
        String lockPath = zRootPath + "/sservers";
        return this.getServerHostStatus(zooReader, lockPath, ServiceStatusReport.ReportKey.S_SERVER);
    }

    private StatusSummary getServerHostStatus(ZooReader zooReader, String basePath, ServiceStatusReport.ReportKey displayNames) {
        AtomicInteger errorSum = new AtomicInteger(0);
        TreeSet<String> groupNames = new TreeSet<String>();
        TreeMap<String, Set<String>> hostsByGroups = new TreeMap<String, Set<String>>();
        Result<Set<String>> nodeNames = this.readNodeNames(zooReader, basePath);
        nodeNames.getHosts().forEach(host -> {
            Result<Set<String>> lock = this.readNodeNames(zooReader, basePath + "/" + host);
            lock.getHosts().forEach(l -> {
                Result<String> nodeData = this.readNodeData(zooReader, basePath + "/" + host + "/" + l);
                int err = nodeData.getErrorCount();
                if (err > 0) {
                    errorSum.addAndGet(nodeData.getErrorCount());
                } else {
                    String[] tokens = nodeData.getHosts().split(",");
                    if (tokens.length == 2) {
                        String groupName = tokens[1];
                        groupNames.add(groupName);
                        hostsByGroups.computeIfAbsent(groupName, s -> new TreeSet()).add(host);
                    } else {
                        hostsByGroups.computeIfAbsent(NO_GROUP_TAG, s -> new TreeSet()).add(host);
                    }
                }
            });
            errorSum.addAndGet((Integer)lock.getFirst());
        });
        return new StatusSummary(displayNames, groupNames, hostsByGroups, errorSum.get());
    }

    @VisibleForTesting
    StatusSummary getGcStatus(ZooReader zooReader, String zRootPath) {
        String lockPath = zRootPath + "/gc/lock";
        StatusSummary temp = this.getStatusSummary(ServiceStatusReport.ReportKey.GC, zooReader, lockPath);
        TreeSet<String> hosts = new TreeSet<String>(this.stripServiceName(temp.getServiceByGroups().get(NO_GROUP_TAG)));
        TreeMap<String, Set<String>> hostByGroup = new TreeMap<String, Set<String>>();
        hostByGroup.put(NO_GROUP_TAG, hosts);
        return new StatusSummary(temp.getServiceType(), temp.getResourceGroups(), hostByGroup, temp.getErrorCount());
    }

    private Set<String> stripServiceName(Set<String> hostnames) {
        return hostnames.stream().map(h -> {
            if (h.contains("=")) {
                return h.substring(h.indexOf("=") + 1);
            }
            return h;
        }).collect(Collectors.toCollection(TreeSet::new));
    }

    @VisibleForTesting
    StatusSummary getCoordinatorStatus(ZooReader zooReader, String zRootPath) {
        String lockPath = zRootPath + "/coordinators/lock";
        return this.getStatusSummary(ServiceStatusReport.ReportKey.COORDINATOR, zooReader, lockPath);
    }

    @VisibleForTesting
    StatusSummary getCompactorStatus(ZooReader zooReader, String zRootPath) {
        String lockPath = zRootPath + "/compactors";
        return this.getCompactorHosts(zooReader, lockPath);
    }

    private StatusSummary getStatusSummary(ServiceStatusReport.ReportKey displayNames, ZooReader zooReader, String lockPath) {
        Result<Set<String>> result = this.readAllNodesData(zooReader, lockPath);
        TreeMap<String, Set<String>> byGroup = new TreeMap<String, Set<String>>();
        byGroup.put(NO_GROUP_TAG, result.getHosts());
        return new StatusSummary(displayNames, Set.of(), byGroup, result.getErrorCount());
    }

    private StatusSummary getCompactorHosts(ZooReader zooReader, String zRootPath) {
        AtomicInteger errors = new AtomicInteger(0);
        TreeMap<String, Set<String>> hostsByGroups = new TreeMap<String, Set<String>>();
        Result<Set<String>> queueNodes = this.readNodeNames(zooReader, zRootPath);
        errors.addAndGet(queueNodes.getErrorCount());
        TreeSet<String> queues = new TreeSet<String>((Collection)queueNodes.getHosts());
        queues.forEach(group -> {
            Result<Set<String>> hostNames = this.readNodeNames(zooReader, zRootPath + "/" + group);
            errors.addAndGet(hostNames.getErrorCount());
            Collection hosts = hostNames.getHosts();
            hosts.forEach(host -> hostsByGroups.computeIfAbsent((String)group, set -> new TreeSet()).add(host));
        });
        return new StatusSummary(ServiceStatusReport.ReportKey.COMPACTOR, queues, hostsByGroups, errors.get());
    }

    @VisibleForTesting
    Result<Set<String>> readNodeNames(ZooReader zooReader, String path) {
        TreeSet nodeNames = new TreeSet();
        AtomicInteger errorCount = new AtomicInteger(0);
        try {
            List children = zooReader.getChildren(path);
            if (children != null) {
                nodeNames.addAll(children);
            }
        }
        catch (InterruptedException | KeeperException ex) {
            if (Thread.currentThread().isInterrupted()) {
                Thread.currentThread().interrupt();
                throw new IllegalStateException(ex);
            }
            errorCount.incrementAndGet();
        }
        return new Result<Set<String>>(errorCount.get(), nodeNames);
    }

    @VisibleForTesting
    Result<String> readNodeData(ZooReader zooReader, String path) {
        try {
            byte[] data = zooReader.getData(path);
            return new Result<String>(0, new String(data, StandardCharsets.UTF_8));
        }
        catch (InterruptedException | KeeperException ex) {
            if (Thread.currentThread().isInterrupted()) {
                Thread.currentThread().interrupt();
                throw new IllegalStateException(ex);
            }
            LOG.info("Could not read locks from ZooKeeper for path {}", (Object)path, (Object)ex);
            return new Result<String>(1, "");
        }
    }

    @VisibleForTesting
    Result<Set<String>> readAllNodesData(ZooReader zooReader, String path) {
        TreeSet hosts = new TreeSet();
        AtomicInteger errorCount = new AtomicInteger(0);
        try {
            List locks = zooReader.getChildren(path);
            locks.forEach(lock -> {
                Result<String> nodeData = this.readNodeData(zooReader, path + "/" + lock);
                int err = nodeData.getErrorCount();
                if (err > 0) {
                    errorCount.addAndGet(nodeData.getErrorCount());
                } else {
                    hosts.add(nodeData.getHosts());
                }
            });
        }
        catch (InterruptedException | KeeperException ex) {
            if (Thread.currentThread().isInterrupted()) {
                Thread.currentThread().interrupt();
                throw new IllegalStateException(ex);
            }
            LOG.info("Could not read node names from ZooKeeper for path {}", (Object)path, (Object)ex);
            errorCount.incrementAndGet();
        }
        return new Result<Set<String>>(errorCount.get(), hosts);
    }

    @Parameters(commandDescription="show service status")
    public static class Opts {
        @Parameter(names={"--json"}, description="provide output in json format (--noHosts ignored)")
        boolean json = false;
        @Parameter(names={"--noHosts"}, description="provide a summary of service counts without host details")
        boolean noHosts = false;
    }

    private static class Result<B>
    extends Pair<Integer, B> {
        public Result(Integer errorCount, B hosts) {
            super((Object)errorCount, hosts);
        }

        public Integer getErrorCount() {
            return (Integer)this.getFirst();
        }

        public B getHosts() {
            return (B)this.getSecond();
        }
    }
}

