/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.tables;

import com.google.common.base.Preconditions;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import org.apache.accumulo.core.client.NamespaceNotFoundException;
import org.apache.accumulo.core.data.InstanceId;
import org.apache.accumulo.core.data.NamespaceId;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.fate.zookeeper.ZooCache;
import org.apache.accumulo.core.fate.zookeeper.ZooReader;
import org.apache.accumulo.core.fate.zookeeper.ZooReaderWriter;
import org.apache.accumulo.core.fate.zookeeper.ZooUtil;
import org.apache.accumulo.core.manager.state.tables.TableState;
import org.apache.accumulo.core.util.Pair;
import org.apache.accumulo.core.util.tables.TableNameUtil;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.conf.store.NamespacePropKey;
import org.apache.accumulo.server.conf.store.PropStore;
import org.apache.accumulo.server.conf.store.TablePropKey;
import org.apache.accumulo.server.tables.IllegalTableTransitionException;
import org.apache.accumulo.server.tables.TableObserver;
import org.apache.accumulo.server.util.PropUtil;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TableManager {
    private static final Logger log = LoggerFactory.getLogger(TableManager.class);
    private static final Set<TableObserver> observers = Collections.synchronizedSet(new HashSet());
    private static final Map<TableId, TableState> tableStateCache = Collections.synchronizedMap(new HashMap());
    private static final byte[] ZERO_BYTE = new byte[]{48};
    private final ServerContext context;
    private final String zkRoot;
    private final InstanceId instanceID;
    private final ZooReaderWriter zoo;
    private final ZooCache zooStateCache;

    public static void prepareNewNamespaceState(ZooReaderWriter zoo, PropStore propStore, InstanceId instanceId, NamespaceId namespaceId, String namespace, ZooUtil.NodeExistsPolicy existsPolicy) throws KeeperException, InterruptedException {
        log.debug("Creating ZooKeeper entries for new namespace {} (ID: {})", (Object)namespace, (Object)namespaceId);
        String zPath = "/accumulo/" + instanceId + "/namespaces/" + namespaceId;
        zoo.putPersistentData(zPath, new byte[0], existsPolicy);
        zoo.putPersistentData(zPath + "/name", namespace.getBytes(StandardCharsets.UTF_8), existsPolicy);
        NamespacePropKey propKey = NamespacePropKey.of(instanceId, namespaceId);
        if (!propStore.exists(propKey)) {
            propStore.create(propKey, Map.of());
        }
    }

    public static void prepareNewNamespaceState(ServerContext context, NamespaceId namespaceId, String namespace, ZooUtil.NodeExistsPolicy existsPolicy) throws KeeperException, InterruptedException {
        TableManager.prepareNewNamespaceState(context.getZooReaderWriter(), context.getPropStore(), context.getInstanceID(), namespaceId, namespace, existsPolicy);
    }

    public static void prepareNewTableState(ZooReaderWriter zoo, PropStore propStore, InstanceId instanceId, TableId tableId, NamespaceId namespaceId, String tableName, TableState state, ZooUtil.NodeExistsPolicy existsPolicy) throws KeeperException, InterruptedException {
        log.debug("Creating ZooKeeper entries for new table {} (ID: {}) in namespace (ID: {})", new Object[]{tableName, tableId, namespaceId});
        Pair qualifiedTableName = TableNameUtil.qualify((String)tableName);
        tableName = (String)qualifiedTableName.getSecond();
        String zTablePath = "/accumulo/" + instanceId + "/tables/" + tableId;
        zoo.putPersistentData(zTablePath, new byte[0], existsPolicy);
        zoo.putPersistentData(zTablePath + "/namespace", namespaceId.canonical().getBytes(StandardCharsets.UTF_8), existsPolicy);
        zoo.putPersistentData(zTablePath + "/name", tableName.getBytes(StandardCharsets.UTF_8), existsPolicy);
        zoo.putPersistentData(zTablePath + "/flush-id", ZERO_BYTE, existsPolicy);
        zoo.putPersistentData(zTablePath + "/compact-id", ZERO_BYTE, existsPolicy);
        zoo.putPersistentData(zTablePath + "/compact-cancel-id", ZERO_BYTE, existsPolicy);
        zoo.putPersistentData(zTablePath + "/state", state.name().getBytes(StandardCharsets.UTF_8), existsPolicy);
        TablePropKey propKey = TablePropKey.of(instanceId, tableId);
        if (!propStore.exists(propKey)) {
            propStore.create(propKey, Map.of());
        }
    }

    public static void prepareNewTableState(ServerContext context, TableId tableId, NamespaceId namespaceId, String tableName, TableState state, ZooUtil.NodeExistsPolicy existsPolicy) throws KeeperException, InterruptedException {
        TableManager.prepareNewTableState(context.getZooReaderWriter(), context.getPropStore(), context.getInstanceID(), tableId, namespaceId, tableName, state, existsPolicy);
    }

    public TableManager(ServerContext context) {
        this.context = context;
        this.zkRoot = context.getZooKeeperRoot();
        this.instanceID = context.getInstanceID();
        this.zoo = context.getZooReaderWriter();
        this.zooStateCache = new ZooCache((ZooReader)this.zoo, (Watcher)new TableStateWatcher());
        this.updateTableStateCache();
    }

    public TableState getTableState(TableId tableId) {
        return tableStateCache.get(tableId);
    }

    public synchronized void transitionTableState(TableId tableId, TableState newState, EnumSet<TableState> expectedCurrStates) {
        Preconditions.checkArgument((newState != TableState.UNKNOWN ? 1 : 0) != 0);
        String statePath = this.zkRoot + "/tables/" + tableId + "/state";
        try {
            this.zoo.mutateOrCreate(statePath, newState.name().getBytes(StandardCharsets.UTF_8), currData -> {
                TableState currState = TableState.UNKNOWN;
                if (currData != null) {
                    currState = TableState.valueOf((String)new String(currData, StandardCharsets.UTF_8));
                }
                if (currState == newState) {
                    return null;
                }
                boolean transition = true;
                switch (currState) {
                    case NEW: {
                        transition = newState == TableState.OFFLINE || newState == TableState.ONLINE;
                        break;
                    }
                    case ONLINE: 
                    case UNKNOWN: 
                    case OFFLINE: {
                        transition = newState != TableState.NEW;
                        break;
                    }
                    case DELETING: {
                        transition = false;
                    }
                }
                if (!transition || !expectedCurrStates.contains(currState)) {
                    throw new IllegalTableTransitionException(currState, newState);
                }
                log.debug("Transitioning state for table {} from {} to {}", new Object[]{tableId, currState, newState});
                return newState.name().getBytes(StandardCharsets.UTF_8);
            });
        }
        catch (Exception e) {
            log.error("FATAL Failed to transition table to state {}", (Object)newState);
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateTableStateCache() {
        Map<TableId, TableState> map = tableStateCache;
        synchronized (map) {
            for (String tableId : this.zooStateCache.getChildren(this.zkRoot + "/tables")) {
                if (this.zooStateCache.get(this.zkRoot + "/tables/" + tableId + "/state") == null) continue;
                this.updateTableStateCache(TableId.of((String)tableId));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TableState updateTableStateCache(TableId tableId) {
        Map<TableId, TableState> map = tableStateCache;
        synchronized (map) {
            TableState tState = TableState.UNKNOWN;
            byte[] data = this.zooStateCache.get(this.zkRoot + "/tables/" + tableId + "/state");
            if (data != null) {
                String sState = new String(data, StandardCharsets.UTF_8);
                try {
                    tState = TableState.valueOf((String)sState);
                }
                catch (IllegalArgumentException e) {
                    log.error("Unrecognized state for table with tableId={}: {}", (Object)tableId, (Object)sState);
                }
                tableStateCache.put(tableId, tState);
            }
            return tState;
        }
    }

    public void addTable(TableId tableId, NamespaceId namespaceId, String tableName) throws KeeperException, InterruptedException, NamespaceNotFoundException {
        TableManager.prepareNewTableState(this.zoo, this.context.getPropStore(), this.instanceID, tableId, namespaceId, tableName, TableState.NEW, ZooUtil.NodeExistsPolicy.OVERWRITE);
        this.updateTableStateCache(tableId);
    }

    public void cloneTable(TableId srcTableId, TableId tableId, String tableName, NamespaceId namespaceId, Map<String, String> propertiesToSet, Set<String> propertiesToExclude) throws KeeperException, InterruptedException {
        TableManager.prepareNewTableState(this.zoo, this.context.getPropStore(), this.instanceID, tableId, namespaceId, tableName, TableState.NEW, ZooUtil.NodeExistsPolicy.OVERWRITE);
        String srcTablePath = "/accumulo/" + this.instanceID + "/tables/" + srcTableId + "/config";
        String newTablePath = "/accumulo/" + this.instanceID + "/tables/" + tableId + "/config";
        this.zoo.recursiveCopyPersistentOverwrite(srcTablePath, newTablePath);
        PropUtil.setProperties(this.context, TablePropKey.of(this.context, tableId), propertiesToSet);
        PropUtil.removeProperties(this.context, TablePropKey.of(this.context, tableId), propertiesToExclude);
        this.updateTableStateCache(tableId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeTable(TableId tableId) throws KeeperException, InterruptedException {
        Map<TableId, TableState> map = tableStateCache;
        synchronized (map) {
            tableStateCache.remove(tableId);
            this.zoo.recursiveDelete(this.zkRoot + "/tables/" + tableId + "/state", ZooUtil.NodeMissingPolicy.SKIP);
            this.zoo.recursiveDelete(this.zkRoot + "/tables/" + tableId, ZooUtil.NodeMissingPolicy.SKIP);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean addObserver(TableObserver to) {
        Set<TableObserver> set = observers;
        synchronized (set) {
            Map<TableId, TableState> map = tableStateCache;
            synchronized (map) {
                to.initialize();
                return observers.add(to);
            }
        }
    }

    public void removeNamespace(NamespaceId namespaceId) throws KeeperException, InterruptedException {
        this.zoo.recursiveDelete(this.zkRoot + "/namespaces/" + namespaceId, ZooUtil.NodeMissingPolicy.SKIP);
    }

    private class TableStateWatcher
    implements Watcher {
        private TableStateWatcher() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Recovered potentially malformed switches.  Disable with '--allowmalformedswitch false'
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        public void process(WatchedEvent event) {
            if (log.isTraceEnabled()) {
                log.trace("{}", (Object)event);
            }
            String zPath = event.getPath();
            Watcher.Event.EventType zType = event.getType();
            String tablesPrefix = TableManager.this.zkRoot + "/tables";
            TableId tableId = null;
            if (zPath != null && zPath.startsWith(tablesPrefix + "/")) {
                String[] sa;
                String suffix = zPath.substring(tablesPrefix.length() + 1);
                if (suffix.contains("/") && "/state".equals("/" + (sa = suffix.split("/", 2))[1])) {
                    tableId = TableId.of((String)sa[0]);
                }
                if (tableId == null) {
                    log.warn("Unknown path in {}", (Object)event);
                    return;
                }
            }
            switch (zType) {
                case NodeChildrenChanged: {
                    if (zPath != null && zPath.equals(tablesPrefix)) {
                        TableManager.this.updateTableStateCache();
                        return;
                    }
                    log.warn("Unexpected path {} in {}", (Object)zPath, (Object)event);
                    return;
                }
                case NodeCreated: 
                case NodeDataChanged: {
                    TableState tState = TableManager.this.updateTableStateCache(tableId);
                    log.debug("State transition to {} @ {}", (Object)tState, (Object)event);
                    Set<TableObserver> set = observers;
                    synchronized (set) {
                        Iterator<TableObserver> iterator = observers.iterator();
                        while (iterator.hasNext()) {
                            TableObserver to = iterator.next();
                            to.stateChanged(tableId, tState);
                        }
                        return;
                    }
                }
                case NodeDeleted: {
                    if (zPath == null) return;
                    if (tableId == null) return;
                    if (!zPath.equals(tablesPrefix + "/" + tableId + "/state") && !zPath.equals(tablesPrefix + "/" + tableId + "/config")) {
                        if (!zPath.equals(tablesPrefix + "/" + tableId + "/name")) return;
                    }
                    tableStateCache.remove(tableId);
                    return;
                }
                case None: {
                    Set<TableObserver> set;
                    switch (event.getState()) {
                        case Expired: {
                            log.trace("Session expired; {}", (Object)event);
                            set = observers;
                            synchronized (set) {
                                Iterator<TableObserver> iterator = observers.iterator();
                                while (iterator.hasNext()) {
                                    TableObserver to = iterator.next();
                                    to.sessionExpired();
                                }
                                return;
                            }
                        }
                    }
                    log.trace("Ignored {}", (Object)event);
                    return;
                }
            }
            log.warn("Unhandled {}", (Object)event);
        }
    }
}

