/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.util.threads;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;
import java.util.Iterator;
import java.util.List;
import java.util.OptionalInt;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.IntSupplier;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.trace.TraceUtil;
import org.apache.accumulo.core.util.threads.NamedThreadFactory;
import org.apache.accumulo.core.util.threads.ThreadPoolNames;
import org.apache.accumulo.core.util.threads.Threads;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressFBWarnings(value={"RV_EXCEPTION_NOT_THROWN"}, justification="Throwing Error for it to be caught by AccumuloUncaughtExceptionHandler")
public class ThreadPools {
    private static final Logger LOG = LoggerFactory.getLogger(ThreadPools.class);
    public static final long DEFAULT_TIMEOUT_MILLISECS = TimeUnit.MINUTES.toMillis(3L);
    private static final ThreadPools SERVER_INSTANCE = new ThreadPools(Threads.UEH);
    private static final ThreadPoolExecutor SCHEDULED_FUTURE_CHECKER_POOL = ThreadPools.getServerThreadPools().getPoolBuilder(ThreadPoolNames.SCHED_FUTURE_CHECKER_POOL).numCoreThreads(1).build();
    private static final ConcurrentLinkedQueue<ScheduledFuture<?>> CRITICAL_RUNNING_TASKS = new ConcurrentLinkedQueue();
    private static final ConcurrentLinkedQueue<ScheduledFuture<?>> NON_CRITICAL_RUNNING_TASKS = new ConcurrentLinkedQueue();
    private static final Runnable TASK_CHECKER = new Runnable(){

        @Override
        public void run() {
            List<ConcurrentLinkedQueue<ScheduledFuture<?>>> queues = List.of(CRITICAL_RUNNING_TASKS, NON_CRITICAL_RUNNING_TASKS);
            while (true) {
                queues.forEach(q -> {
                    Iterator tasks = q.iterator();
                    while (tasks.hasNext()) {
                        if (!ThreadPools.checkTaskFailed((ScheduledFuture)tasks.next(), q)) continue;
                        tasks.remove();
                    }
                });
                try {
                    TimeUnit.MINUTES.sleep(1L);
                    continue;
                }
                catch (InterruptedException ie) {
                    Thread.interrupted();
                    continue;
                }
                break;
            }
        }
    };
    private final Thread.UncaughtExceptionHandler handler;

    public static final ThreadPools getServerThreadPools() {
        return SERVER_INSTANCE;
    }

    public static final ThreadPools getClientThreadPools(Thread.UncaughtExceptionHandler ueh) {
        return new ThreadPools(ueh);
    }

    private static boolean checkTaskFailed(ScheduledFuture<?> future, ConcurrentLinkedQueue<ScheduledFuture<?>> taskQueue) {
        if (future.isDone()) {
            try {
                future.get();
                return true;
            }
            catch (ExecutionException ee) {
                if (taskQueue == CRITICAL_RUNNING_TASKS) {
                    throw new ExecutionError("Critical scheduled background task failed.", ee);
                }
                LOG.error("Non-critical scheduled background task failed", (Throwable)ee);
                return true;
            }
            catch (CancellationException ce) {
                return true;
            }
            catch (InterruptedException ie) {
                LOG.info("Interrupted while waiting to check on scheduled background task.");
                Thread.interrupted();
            }
        }
        return false;
    }

    public static void watchCriticalScheduledTask(ScheduledFuture<?> future) {
        CRITICAL_RUNNING_TASKS.add(future);
    }

    public static void watchCriticalFixedDelay(AccumuloConfiguration aconf, long intervalMillis, Runnable runnable) {
        ScheduledFuture<?> future = ThreadPools.getServerThreadPools().createGeneralScheduledExecutorService(aconf).scheduleWithFixedDelay(runnable, intervalMillis, intervalMillis, TimeUnit.MILLISECONDS);
        CRITICAL_RUNNING_TASKS.add(future);
    }

    public static void watchNonCriticalScheduledTask(ScheduledFuture<?> future) {
        NON_CRITICAL_RUNNING_TASKS.add(future);
    }

    public static void ensureRunning(ScheduledFuture<?> future, String message) {
        if (future.isDone()) {
            try {
                future.get();
            }
            catch (Exception e) {
                throw new IllegalStateException(message, e);
            }
            throw new IllegalStateException(message);
        }
    }

    public static void resizePool(ThreadPoolExecutor pool, IntSupplier maxThreads, String poolName) {
        int newCount;
        int count = pool.getMaximumPoolSize();
        if (count == (newCount = maxThreads.getAsInt())) {
            return;
        }
        LOG.info("Changing max threads for {} from {} to {}", new Object[]{poolName, count, newCount});
        if (newCount > count) {
            pool.setMaximumPoolSize(newCount);
            pool.setCorePoolSize(newCount);
        } else {
            pool.setCorePoolSize(newCount);
            pool.setMaximumPoolSize(newCount);
        }
    }

    public static void resizePool(ThreadPoolExecutor pool, AccumuloConfiguration conf, Property p) {
        ThreadPools.resizePool(pool, () -> conf.getCount(p), p.getKey());
    }

    private ThreadPools(Thread.UncaughtExceptionHandler ueh) {
        this.handler = ueh;
    }

    public ThreadPoolExecutor createExecutorService(AccumuloConfiguration conf, Property p) {
        return this.createExecutorService(conf, p, false);
    }

    public ThreadPoolExecutor createExecutorService(AccumuloConfiguration conf, Property p, boolean emitThreadPoolMetrics) {
        switch (p) {
            case GENERAL_SIMPLETIMER_THREADPOOL_SIZE: {
                return this.createScheduledExecutorService(conf.getCount(p), ThreadPoolNames.GENERAL_SERVER_SIMPLETIMER_POOL.poolName);
            }
            case GENERAL_THREADPOOL_SIZE: {
                return this.createScheduledExecutorService(conf.getCount(p), ThreadPoolNames.GENERAL_SERVER_POOL.poolName, emitThreadPoolMetrics);
            }
            case MANAGER_BULK_THREADPOOL_SIZE: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.MANAGER_BULK_IMPORT_POOL).numCoreThreads(conf.getCount(p)).withTimeOut(conf.getTimeInMillis(Property.MANAGER_BULK_THREADPOOL_TIMEOUT), TimeUnit.MILLISECONDS);
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case MANAGER_RENAME_THREADS: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.MANAGER_RENAME_POOL).numCoreThreads(conf.getCount(p));
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case MANAGER_FATE_THREADPOOL_SIZE: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.MANAGER_FATE_POOL).numCoreThreads(conf.getCount(p));
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case MANAGER_STATUS_THREAD_POOL_SIZE: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.MANAGER_STATUS_POOL);
                int threads = conf.getCount(p);
                if (threads == 0) {
                    builder.numCoreThreads(0).numMaxThreads(Integer.MAX_VALUE).withTimeOut(60L, TimeUnit.SECONDS).withQueue(new SynchronousQueue<Runnable>());
                } else {
                    builder.numCoreThreads(threads);
                }
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case TSERV_WORKQ_THREADS: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.TSERVER_WORKQ_POOL).numCoreThreads(conf.getCount(p));
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case TSERV_MINC_MAXCONCURRENT: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.TSERVER_MINOR_COMPACTOR_POOL).numCoreThreads(conf.getCount(p)).withTimeOut(0L, TimeUnit.MILLISECONDS);
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case TSERV_MIGRATE_MAXCONCURRENT: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.TSERVER_MIGRATIONS_POOL).numCoreThreads(conf.getCount(p)).withTimeOut(0L, TimeUnit.MILLISECONDS);
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case TSERV_ASSIGNMENT_MAXCONCURRENT: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.TSERVER_ASSIGNMENT_POOL).numCoreThreads(conf.getCount(p)).withTimeOut(0L, TimeUnit.MILLISECONDS);
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case TSERV_SUMMARY_RETRIEVAL_THREADS: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.TSERVER_SUMMARY_RETRIEVAL_POOL).numCoreThreads(conf.getCount(p)).withTimeOut(60L, TimeUnit.MILLISECONDS);
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case TSERV_SUMMARY_REMOTE_THREADS: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.TSERVER_SUMMARY_REMOTE_POOL).numCoreThreads(conf.getCount(p)).withTimeOut(60L, TimeUnit.MILLISECONDS);
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case TSERV_SUMMARY_PARTITION_THREADS: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.TSERVER_SUMMARY_PARTITION_POOL).numCoreThreads(conf.getCount(p)).withTimeOut(60L, TimeUnit.MILLISECONDS);
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
            case GC_DELETE_THREADS: {
                return this.getPoolBuilder(ThreadPoolNames.GC_DELETE_POOL).numCoreThreads(conf.getCount(p)).build();
            }
            case REPLICATION_WORKER_THREADS: {
                ThreadPoolExecutorBuilder builder = this.getPoolBuilder(ThreadPoolNames.REPLICATION_WORKER_POOL).numCoreThreads(conf.getCount(p));
                if (emitThreadPoolMetrics) {
                    builder.enableThreadPoolMetrics();
                }
                return builder.build();
            }
        }
        throw new IllegalArgumentException("Unhandled thread pool property: " + p);
    }

    public ThreadPoolExecutorBuilder getPoolBuilder(@NonNull ThreadPoolNames pool) {
        return new ThreadPoolExecutorBuilder(pool.poolName);
    }

    public ThreadPoolExecutorBuilder getPoolBuilder(@NonNull String name) {
        String trimmed = name.trim();
        if (trimmed.startsWith(ThreadPoolNames.ACCUMULO_POOL_PREFIX.poolName)) {
            return new ThreadPoolExecutorBuilder(trimmed);
        }
        return new ThreadPoolExecutorBuilder(ThreadPoolNames.ACCUMULO_POOL_PREFIX.poolName + trimmed);
    }

    private ThreadPoolExecutor createThreadPool(int coreThreads, int maxThreads, long timeOut, TimeUnit units, String name, BlockingQueue<Runnable> queue, OptionalInt priority, boolean emitThreadPoolMetrics) {
        LOG.trace("Creating ThreadPoolExecutor for {} with {} core threads and {} max threads {} {} timeout", new Object[]{name, coreThreads, maxThreads, timeOut, units});
        ThreadPoolExecutor result = new ThreadPoolExecutor(coreThreads, maxThreads, timeOut, units, queue, new NamedThreadFactory(name, priority, this.handler)){

            @Override
            public void execute(@NonNull Runnable arg0) {
                super.execute(TraceUtil.wrap(arg0));
            }

            @Override
            public boolean remove(Runnable task) {
                return super.remove(TraceUtil.wrap(task));
            }

            @Override
            public <T> @NonNull Future<T> submit(@NonNull Callable<T> task) {
                return super.submit(TraceUtil.wrap(task));
            }

            @Override
            public <T> @NonNull Future<T> submit(@NonNull Runnable task, T result) {
                return super.submit(TraceUtil.wrap(task), result);
            }

            @Override
            public @NonNull Future<?> submit(@NonNull Runnable task) {
                return super.submit(TraceUtil.wrap(task));
            }
        };
        if (timeOut > 0L) {
            result.allowCoreThreadTimeOut(true);
        }
        if (emitThreadPoolMetrics) {
            ThreadPools.addExecutorServiceMetrics(result, name);
        }
        return result;
    }

    public ScheduledThreadPoolExecutor createGeneralScheduledExecutorService(AccumuloConfiguration conf) {
        Property oldProp = Property.GENERAL_SIMPLETIMER_THREADPOOL_SIZE;
        Property prop = conf.resolve(Property.GENERAL_THREADPOOL_SIZE, oldProp);
        return (ScheduledThreadPoolExecutor)this.createExecutorService(conf, prop, true);
    }

    public ScheduledThreadPoolExecutor createScheduledExecutorService(int numThreads, String name) {
        return this.createScheduledExecutorService(numThreads, name, false);
    }

    private ScheduledThreadPoolExecutor createScheduledExecutorService(int numThreads, String name, boolean emitThreadPoolMetrics) {
        LOG.trace("Creating ScheduledThreadPoolExecutor for {} with {} threads", (Object)name, (Object)numThreads);
        ScheduledThreadPoolExecutor result = new ScheduledThreadPoolExecutor(numThreads, new NamedThreadFactory(name, this.handler)){

            @Override
            public void execute(@NonNull Runnable command) {
                super.execute(TraceUtil.wrap(command));
            }

            @Override
            public <V> @NonNull ScheduledFuture<V> schedule(@NonNull Callable<V> callable, long delay, @NonNull TimeUnit unit) {
                return super.schedule(TraceUtil.wrap(callable), delay, unit);
            }

            @Override
            public @NonNull ScheduledFuture<?> schedule(@NonNull Runnable command, long delay, @NonNull TimeUnit unit) {
                return super.schedule(TraceUtil.wrap(command), delay, unit);
            }

            @Override
            public @NonNull ScheduledFuture<?> scheduleAtFixedRate(@NonNull Runnable command, long initialDelay, long period, @NonNull TimeUnit unit) {
                return super.scheduleAtFixedRate(TraceUtil.wrap(command), initialDelay, period, unit);
            }

            @Override
            public @NonNull ScheduledFuture<?> scheduleWithFixedDelay(@NonNull Runnable command, long initialDelay, long delay, @NonNull TimeUnit unit) {
                return super.scheduleWithFixedDelay(TraceUtil.wrap(command), initialDelay, delay, unit);
            }

            @Override
            public <T> @NonNull Future<T> submit(@NonNull Callable<T> task) {
                return super.submit(TraceUtil.wrap(task));
            }

            @Override
            public <T> @NonNull Future<T> submit(@NonNull Runnable task, T result) {
                return super.submit(TraceUtil.wrap(task), result);
            }

            @Override
            public @NonNull Future<?> submit(@NonNull Runnable task) {
                return super.submit(TraceUtil.wrap(task));
            }

            @Override
            public boolean remove(Runnable task) {
                return super.remove(TraceUtil.wrap(task));
            }
        };
        if (emitThreadPoolMetrics) {
            ThreadPools.addExecutorServiceMetrics(result, name);
        }
        return result;
    }

    private static void addExecutorServiceMetrics(ExecutorService executor, String name) {
        new ExecutorServiceMetrics(executor, name, List.of()).bindTo((MeterRegistry)Metrics.globalRegistry);
    }

    static {
        SCHEDULED_FUTURE_CHECKER_POOL.execute(TASK_CHECKER);
    }

    public static class ExecutionError
    extends Error {
        private static final long serialVersionUID = 1L;

        public ExecutionError(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public class ThreadPoolExecutorBuilder {
        final String name;
        int coreThreads = 0;
        int maxThreads = -1;
        long timeOut = DEFAULT_TIMEOUT_MILLISECS;
        TimeUnit units = TimeUnit.MILLISECONDS;
        BlockingQueue<Runnable> queue = new LinkedBlockingQueue<Runnable>();
        OptionalInt priority = OptionalInt.empty();
        boolean emitThreadPoolMetrics = false;

        ThreadPoolExecutorBuilder(String name) {
            this.name = name;
        }

        public ThreadPoolExecutor build() {
            Preconditions.checkArgument((this.coreThreads >= 0 ? 1 : 0) != 0, (Object)"The number of core threads must be 0 or larger");
            if (this.maxThreads < 0) {
                this.maxThreads = this.coreThreads == 0 ? 1 : this.coreThreads;
            }
            Preconditions.checkArgument((this.maxThreads >= this.coreThreads ? 1 : 0) != 0, (Object)"The number of max threads must be greater than 0 and greater than or equal to the number of core threads");
            Preconditions.checkArgument((this.priority.orElse(1) >= 1 && this.priority.orElse(1) <= 10 ? 1 : 0) != 0, (Object)"invalid thread priority, range must be Thread.MIN_PRIORITY <= priority <= Thread.MAX_PRIORITY");
            return ThreadPools.this.createThreadPool(this.coreThreads, this.maxThreads, this.timeOut, this.units, this.name, this.queue, this.priority, this.emitThreadPoolMetrics);
        }

        public ThreadPoolExecutorBuilder numCoreThreads(int coreThreads) {
            this.coreThreads = coreThreads;
            return this;
        }

        public ThreadPoolExecutorBuilder numMaxThreads(int maxThreads) {
            this.maxThreads = maxThreads;
            return this;
        }

        public ThreadPoolExecutorBuilder withTimeOut(long timeOut, @NonNull TimeUnit units) {
            this.timeOut = timeOut;
            this.units = units;
            return this;
        }

        public ThreadPoolExecutorBuilder withQueue(@NonNull BlockingQueue<Runnable> queue) {
            this.queue = queue;
            return this;
        }

        public ThreadPoolExecutorBuilder atPriority(@NonNull OptionalInt priority) {
            this.priority = priority;
            return this;
        }

        public ThreadPoolExecutorBuilder enableThreadPoolMetrics() {
            return this.enableThreadPoolMetrics(true);
        }

        public ThreadPoolExecutorBuilder enableThreadPoolMetrics(boolean enable) {
            this.emitThreadPoolMetrics = enable;
            return this;
        }
    }
}

