/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.conf;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import org.apache.accumulo.core.conf.ConfigurationTypeHelper;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.conf.PropertyType;
import org.apache.accumulo.core.spi.crypto.CryptoServiceFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigCheckUtil {
    private static final Logger log = LoggerFactory.getLogger(ConfigCheckUtil.class);
    private static final String PREFIX = "BAD CONFIG ";

    public static void validate(Iterable<Map.Entry<String, String>> entries) {
        String instanceZkTimeoutValue = null;
        for (Map.Entry<String, String> entry : entries) {
            String key = entry.getKey();
            String value = entry.getValue();
            Property prop = Property.getPropertyByKey(entry.getKey());
            if (prop == null && Property.isValidPropertyKey(key)) continue;
            if (prop == null) {
                log.warn("BAD CONFIG unrecognized property key (" + key + ")");
            } else if (prop.getType() == PropertyType.PREFIX) {
                ConfigCheckUtil.fatal("BAD CONFIG incomplete property key (" + key + ")");
            } else if (!prop.getType().isValidFormat(value)) {
                ConfigCheckUtil.fatal("BAD CONFIG improperly formatted value for key (" + key + ", type=" + prop.getType() + ") : " + value);
            }
            if (key.equals(Property.INSTANCE_ZK_TIMEOUT.getKey())) {
                instanceZkTimeoutValue = value;
            }
            if (key.equals(Property.TABLE_FILE_COMPRESSED_BLOCK_SIZE_INDEX.getKey()) || key.equals(Property.TABLE_FILE_COMPRESSED_BLOCK_SIZE.getKey())) {
                long bsize = ConfigurationTypeHelper.getFixedMemoryAsBytes(value);
                Preconditions.checkArgument((bsize > 0L && bsize < Integer.MAX_VALUE ? 1 : 0) != 0, (Object)(key + " must be greater than 0 and less than 2147483647 but was: " + bsize));
            }
            if (!key.equals(Property.INSTANCE_CRYPTO_FACTORY.getKey())) continue;
            String cryptoStrategy = Objects.requireNonNull(value);
            ConfigCheckUtil.verifyValidClassName(key, cryptoStrategy, CryptoServiceFactory.class);
        }
        if (instanceZkTimeoutValue != null) {
            ConfigCheckUtil.checkTimeDuration(Property.INSTANCE_ZK_TIMEOUT, instanceZkTimeoutValue, new CheckTimeDurationBetween(1000L, 300000L));
        }
    }

    private static void checkTimeDuration(Property prop, String value, CheckTimeDuration chk) {
        ConfigCheckUtil.verifyPropertyTypes(PropertyType.TIMEDURATION, prop);
        if (!chk.check(ConfigurationTypeHelper.getTimeInMillis(value))) {
            ConfigCheckUtil.fatal(PREFIX + chk.getDescription(prop));
        }
    }

    private static void verifyPropertyTypes(PropertyType type, Property ... properties) {
        for (Property prop : properties) {
            if (prop.getType() == type) continue;
            ConfigCheckUtil.fatal("Unexpected property type (" + prop.getType() + " != " + type + ")");
        }
    }

    private static void fatal(String msg) {
        log.error("FATAL: {}", (Object)msg);
        throw new ConfigCheckException(msg);
    }

    private static void verifyValidClassName(String confOption, String className, Class<?> requiredBaseClass) {
        try {
            ConfigurationTypeHelper.getClassInstance(null, className, requiredBaseClass);
        }
        catch (IOException | ReflectiveOperationException e) {
            ConfigCheckUtil.fatal(confOption + " has an invalid class name: " + className);
        }
        catch (ClassCastException e) {
            ConfigCheckUtil.fatal(confOption + " must implement " + requiredBaseClass + ", but the configured class does not: " + className);
        }
    }

    public static class ConfigCheckException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public ConfigCheckException(String msg) {
            super(msg);
        }
    }

    private static class CheckTimeDurationBetween
    implements CheckTimeDuration {
        long min;
        long max;

        CheckTimeDurationBetween(long x, long y) {
            this.min = Math.min(x, y);
            this.max = Math.max(x, y);
        }

        @Override
        public boolean check(long propVal) {
            return propVal >= this.min && propVal <= this.max;
        }

        @Override
        public String getDescription(Property prop) {
            return "ensure " + this.min + " <= " + prop + " <= " + this.max;
        }
    }

    private static interface CheckTimeDuration {
        public boolean check(long var1);

        public String getDescription(Property var1);
    }
}

