/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.clientImpl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.security.SecureRandom;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import org.apache.accumulo.core.clientImpl.ClientContext;
import org.apache.accumulo.core.clientImpl.ThriftTransportKey;
import org.apache.accumulo.core.rpc.ThriftUtil;
import org.apache.accumulo.core.util.HostAndPort;
import org.apache.accumulo.core.util.Pair;
import org.apache.accumulo.core.util.threads.Threads;
import org.apache.thrift.TConfiguration;
import org.apache.thrift.transport.TTransport;
import org.apache.thrift.transport.TTransportException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ThriftTransportPool {
    private static final Logger log = LoggerFactory.getLogger(ThriftTransportPool.class);
    private static final SecureRandom random = new SecureRandom();
    private static final long ERROR_THRESHOLD = 20L;
    private static final long STUCK_THRESHOLD = TimeUnit.MINUTES.toMillis(2L);
    private final ConnectionPool connectionPool = new ConnectionPool();
    private final Map<ThriftTransportKey, Long> errorCount = new HashMap<ThriftTransportKey, Long>();
    private final Map<ThriftTransportKey, Long> errorTime = new HashMap<ThriftTransportKey, Long>();
    private final Set<ThriftTransportKey> serversWarnedAbout = new HashSet<ThriftTransportKey>();
    private final Thread checkThread;
    private final LongSupplier maxAgeMillis;

    private ThriftTransportPool(LongSupplier maxAgeMillis) {
        this.maxAgeMillis = maxAgeMillis;
        this.checkThread = Threads.createThread("Thrift Connection Pool Checker", () -> {
            try {
                long minNanos = TimeUnit.MILLISECONDS.toNanos(250L);
                long maxNanos = TimeUnit.MINUTES.toNanos(1L);
                long lastRun = System.nanoTime();
                while (!this.connectionPool.shutdown) {
                    long threshold = Math.min(maxNanos, Math.max(minNanos, TimeUnit.MILLISECONDS.toNanos(maxAgeMillis.getAsLong()) / 2L));
                    long currentNanos = System.nanoTime();
                    if (currentNanos - lastRun >= threshold) {
                        this.closeExpiredConnections();
                        lastRun = currentNanos;
                    }
                    Thread.sleep(250L);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (TransportPoolShutdownException e) {
                log.debug("Error closing expired connections", (Throwable)e);
            }
        });
    }

    static ThriftTransportPool startNew(LongSupplier maxAgeMillis) {
        ThriftTransportPool pool = new ThriftTransportPool(maxAgeMillis);
        log.debug("Set thrift transport pool idle time to {}ms", (Object)maxAgeMillis.getAsLong());
        pool.checkThread.start();
        return pool;
    }

    public TTransport getTransport(HostAndPort location, long milliseconds, ClientContext context) throws TTransportException {
        ThriftTransportKey cacheKey = new ThriftTransportKey(location, milliseconds, context);
        CachedConnection connection = this.connectionPool.reserveAny(cacheKey);
        if (connection != null) {
            log.trace("Using existing connection to {}", (Object)cacheKey.getServer());
            return connection.transport;
        }
        return this.createNewTransport(cacheKey);
    }

    @VisibleForTesting
    public Pair<String, TTransport> getAnyTransport(List<ThriftTransportKey> servers, boolean preferCachedConnection) throws TTransportException {
        servers = new ArrayList<ThriftTransportKey>(servers);
        if (preferCachedConnection) {
            HashSet<ThriftTransportKey> serversSet = new HashSet<ThriftTransportKey>(servers);
            serversSet.retainAll(this.connectionPool.getThriftTransportKeys());
            if (!serversSet.isEmpty()) {
                ArrayList<ThriftTransportKey> cachedServers = new ArrayList<ThriftTransportKey>(serversSet);
                Collections.shuffle(cachedServers, random);
                for (ThriftTransportKey ttk : cachedServers) {
                    CachedConnection connection = this.connectionPool.reserveAny(ttk);
                    if (connection == null) continue;
                    String serverAddr = ttk.getServer().toString();
                    log.trace("Using existing connection to {}", (Object)serverAddr);
                    return new Pair<String, TTransport>(serverAddr, connection.transport);
                }
            }
        }
        for (int retryCount = 0; !servers.isEmpty() && retryCount < 10; ++retryCount) {
            CachedConnection connection;
            int index = random.nextInt(servers.size());
            ThriftTransportKey ttk = servers.get(index);
            if (preferCachedConnection && (connection = this.connectionPool.reserveAnyIfPresent(ttk)) != null) {
                return new Pair<String, TTransport>(ttk.getServer().toString(), connection.transport);
            }
            try {
                return new Pair<String, TTransport>(ttk.getServer().toString(), this.createNewTransport(ttk));
            }
            catch (TTransportException tte) {
                log.debug("Failed to connect to {}", (Object)servers.get(index), (Object)tte);
                servers.remove(index);
                continue;
            }
        }
        throw new TTransportException("Failed to connect to a server");
    }

    private TTransport createNewTransport(ThriftTransportKey cacheKey) throws TTransportException {
        TTransport transport = ThriftUtil.createClientTransport(cacheKey.getServer(), (int)cacheKey.getTimeout(), cacheKey.getSslParams(), cacheKey.getSaslParams());
        log.trace("Creating new connection to connection to {}", (Object)cacheKey.getServer());
        CachedTTransport tsc = new CachedTTransport(transport, cacheKey);
        CachedConnection connection = new CachedConnection(tsc);
        connection.reserve();
        try {
            this.connectionPool.putReserved(cacheKey, connection);
        }
        catch (TransportPoolShutdownException e) {
            connection.transport.close();
            throw e;
        }
        return connection.transport;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void returnTransport(TTransport transport) {
        if (transport == null) {
            return;
        }
        CachedTTransport cachedTransport = (CachedTTransport)transport;
        ArrayList<CachedConnection> closeList = new ArrayList<CachedConnection>();
        boolean existInCache = this.connectionPool.returnTransport(cachedTransport, closeList);
        closeList.forEach(connection -> {
            try {
                connection.transport.close();
            }
            catch (Exception e) {
                log.debug("Failed to close connection w/ errors", (Throwable)e);
            }
        });
        if (cachedTransport.sawError) {
            long ecount;
            boolean shouldWarn = false;
            Map<ThriftTransportKey, Long> map = this.errorCount;
            synchronized (map) {
                ecount = this.errorCount.merge(cachedTransport.getCacheKey(), 1L, Long::sum);
                this.errorTime.computeIfAbsent(cachedTransport.getCacheKey(), k -> System.currentTimeMillis());
                if (ecount >= 20L && this.serversWarnedAbout.add(cachedTransport.getCacheKey())) {
                    shouldWarn = true;
                }
            }
            log.trace("Returned connection had error {}", (Object)cachedTransport.getCacheKey());
            if (shouldWarn) {
                log.warn("Server {} had {} failures in a short time period, will not complain anymore", (Object)cachedTransport.getCacheKey(), (Object)ecount);
            }
        }
        if (!existInCache) {
            log.warn("Returned tablet server connection to cache that did not come from cache");
            transport.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeExpiredConnections() {
        List<CachedConnection> expiredConnections = this.connectionPool.removeExpiredConnections(this.maxAgeMillis);
        Map<ThriftTransportKey, Long> map = this.errorCount;
        synchronized (map) {
            Iterator<Map.Entry<ThriftTransportKey, Long>> iter = this.errorTime.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry<ThriftTransportKey, Long> entry = iter.next();
                long delta = System.currentTimeMillis() - entry.getValue();
                if (delta < STUCK_THRESHOLD) continue;
                this.errorCount.remove(entry.getKey());
                iter.remove();
            }
        }
        expiredConnections.forEach(c -> c.transport.close());
    }

    void shutdown() {
        this.connectionPool.shutdown();
        try {
            this.checkThread.join();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    private static class CachedTTransport
    extends TTransport {
        private final ThriftTransportKey cacheKey;
        private final TTransport wrappedTransport;
        private boolean sawError = false;
        private volatile String ioThreadName = null;
        private volatile long ioStartTime = 0L;
        private volatile boolean reserved = false;
        private String stuckThreadName = null;
        int ioCount = 0;
        int lastIoCount = -1;

        private void sawError() {
            this.sawError = true;
        }

        final void setReserved(boolean reserved) {
            this.reserved = reserved;
            if (reserved) {
                this.ioThreadName = Thread.currentThread().getName();
                this.ioCount = 0;
                this.lastIoCount = -1;
            } else {
                if ((this.ioCount & 1) == 1) {
                    log.warn("Connection returned to thrift connection pool that may still be in use {} {}", new Object[]{this.ioThreadName, Thread.currentThread().getName(), new Exception()});
                }
                this.ioCount = 0;
                this.lastIoCount = -1;
                this.ioThreadName = null;
            }
            this.checkForStuckIO(STUCK_THRESHOLD);
        }

        final void checkForStuckIO(long threshold) {
            if ((this.ioCount & 1) == 1) {
                if (this.ioCount == this.lastIoCount) {
                    long delta = System.currentTimeMillis() - this.ioStartTime;
                    if (delta >= threshold && this.stuckThreadName == null) {
                        this.stuckThreadName = this.ioThreadName;
                        log.warn("Thread \"{}\" stuck on IO to {} for at least {} ms", new Object[]{this.ioThreadName, this.cacheKey, delta});
                    }
                } else {
                    this.lastIoCount = this.ioCount;
                    this.ioStartTime = System.currentTimeMillis();
                    if (this.stuckThreadName != null) {
                        log.info("Thread \"{}\" no longer stuck on IO to {} sawError = {}", new Object[]{this.stuckThreadName, this.cacheKey, this.sawError});
                        this.stuckThreadName = null;
                    }
                }
            } else if (this.stuckThreadName != null) {
                log.info("Thread \"{}\" no longer stuck on IO to {} sawError = {}", new Object[]{this.stuckThreadName, this.cacheKey, this.sawError});
                this.stuckThreadName = null;
            }
        }

        public CachedTTransport(TTransport transport, ThriftTransportKey cacheKey2) {
            this.wrappedTransport = transport;
            this.cacheKey = cacheKey2;
        }

        public boolean isOpen() {
            return this.wrappedTransport.isOpen();
        }

        public void open() throws TTransportException {
            try {
                ++this.ioCount;
                this.wrappedTransport.open();
            }
            catch (TTransportException tte) {
                this.sawError();
                throw tte;
            }
            finally {
                ++this.ioCount;
            }
        }

        public int read(byte[] arg0, int arg1, int arg2) throws TTransportException {
            try {
                ++this.ioCount;
                int n = this.wrappedTransport.read(arg0, arg1, arg2);
                return n;
            }
            catch (TTransportException tte) {
                this.sawError();
                throw tte;
            }
            finally {
                ++this.ioCount;
            }
        }

        public int readAll(byte[] arg0, int arg1, int arg2) throws TTransportException {
            try {
                ++this.ioCount;
                int n = this.wrappedTransport.readAll(arg0, arg1, arg2);
                return n;
            }
            catch (TTransportException tte) {
                this.sawError();
                throw tte;
            }
            finally {
                ++this.ioCount;
            }
        }

        public void write(byte[] arg0, int arg1, int arg2) throws TTransportException {
            try {
                ++this.ioCount;
                this.wrappedTransport.write(arg0, arg1, arg2);
            }
            catch (TTransportException tte) {
                this.sawError();
                throw tte;
            }
            finally {
                ++this.ioCount;
            }
        }

        public void write(byte[] arg0) throws TTransportException {
            try {
                ++this.ioCount;
                this.wrappedTransport.write(arg0);
            }
            catch (TTransportException tte) {
                this.sawError();
                throw tte;
            }
            finally {
                ++this.ioCount;
            }
        }

        public void close() {
            try {
                ++this.ioCount;
                if (this.wrappedTransport.isOpen()) {
                    this.wrappedTransport.close();
                }
            }
            finally {
                ++this.ioCount;
            }
        }

        public void flush() throws TTransportException {
            try {
                ++this.ioCount;
                this.wrappedTransport.flush();
            }
            catch (TTransportException tte) {
                this.sawError();
                throw tte;
            }
            finally {
                ++this.ioCount;
            }
        }

        public boolean peek() {
            try {
                ++this.ioCount;
                boolean bl = this.wrappedTransport.peek();
                return bl;
            }
            finally {
                ++this.ioCount;
            }
        }

        public byte[] getBuffer() {
            try {
                ++this.ioCount;
                byte[] byArray = this.wrappedTransport.getBuffer();
                return byArray;
            }
            finally {
                ++this.ioCount;
            }
        }

        public int getBufferPosition() {
            try {
                ++this.ioCount;
                int n = this.wrappedTransport.getBufferPosition();
                return n;
            }
            finally {
                ++this.ioCount;
            }
        }

        public int getBytesRemainingInBuffer() {
            try {
                ++this.ioCount;
                int n = this.wrappedTransport.getBytesRemainingInBuffer();
                return n;
            }
            finally {
                ++this.ioCount;
            }
        }

        public void consumeBuffer(int len) {
            try {
                ++this.ioCount;
                this.wrappedTransport.consumeBuffer(len);
            }
            finally {
                ++this.ioCount;
            }
        }

        public TConfiguration getConfiguration() {
            return this.wrappedTransport.getConfiguration();
        }

        public void updateKnownMessageSize(long size) throws TTransportException {
            try {
                ++this.ioCount;
                this.wrappedTransport.updateKnownMessageSize(size);
            }
            finally {
                ++this.ioCount;
            }
        }

        public void checkReadBytesAvailable(long numBytes) throws TTransportException {
            try {
                ++this.ioCount;
                this.wrappedTransport.checkReadBytesAvailable(numBytes);
            }
            finally {
                ++this.ioCount;
            }
        }

        public ThriftTransportKey getCacheKey() {
            return this.cacheKey;
        }
    }

    public static class TransportPoolShutdownException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public TransportPoolShutdownException(String msg) {
            super(msg);
        }
    }

    private static class CachedConnection {
        final CachedTTransport transport;
        long lastReturnTime;

        public CachedConnection(CachedTTransport t) {
            this.transport = t;
        }

        void reserve() {
            Preconditions.checkState((!this.transport.reserved ? 1 : 0) != 0);
            this.transport.setReserved(true);
        }

        void unreserve() {
            Preconditions.checkState((boolean)this.transport.reserved);
            this.transport.setReserved(false);
        }
    }

    private static class ConnectionPool {
        final Lock[] locks;
        final ConcurrentHashMap<ThriftTransportKey, CachedConnections> connections = new ConcurrentHashMap();
        private volatile boolean shutdown = false;

        ConnectionPool() {
            this.locks = new Lock[37];
            for (int i = 0; i < this.locks.length; ++i) {
                this.locks[i] = new ReentrantLock();
            }
        }

        Set<ThriftTransportKey> getThriftTransportKeys() {
            return this.connections.keySet();
        }

        CachedConnection reserveAny(ThriftTransportKey key) {
            CachedConnections connections = this.getOrCreateCachedConnections(key);
            return this.executeWithinLock(key, connections::reserveAny);
        }

        CachedConnection reserveAnyIfPresent(ThriftTransportKey key) {
            CachedConnections connections = this.getCachedConnections(key);
            return connections == null ? null : this.executeWithinLock(key, connections::reserveAny);
        }

        void putReserved(ThriftTransportKey key, CachedConnection connection) {
            CachedConnections connections = this.getOrCreateCachedConnections(key);
            this.executeWithinLock(key, () -> connections.reserved.put(connection.transport, connection));
        }

        boolean returnTransport(CachedTTransport transport, List<CachedConnection> toBeClosed) {
            CachedConnections connections = this.getOrCreateCachedConnections(transport.getCacheKey());
            return this.executeWithinLock(transport.getCacheKey(), () -> this.unreserveConnection(transport, connections, toBeClosed));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @SuppressFBWarnings(value={"UL_UNRELEASED_LOCK"}, justification="FindBugs doesn't recognize that all locks in ConnectionPool.locks are subsequently unlocked in the try-finally in ConnectionPool.shutdown()")
        void shutdown() {
            for (Lock lock : this.locks) {
                lock.lock();
            }
            try {
                if (this.shutdown) {
                    return;
                }
                this.shutdown = true;
                this.connections.values().forEach(CachedConnections::closeAllTransports);
            }
            finally {
                for (Lock lock : this.locks) {
                    lock.unlock();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        <T> T executeWithinLock(ThriftTransportKey key, Supplier<T> function) {
            Lock lock = this.getLock(key);
            try {
                T t = function.get();
                return t;
            }
            finally {
                lock.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void executeWithinLock(ThriftTransportKey key, Consumer<ThriftTransportKey> consumer) {
            Lock lock = this.getLock(key);
            try {
                consumer.accept(key);
            }
            finally {
                lock.unlock();
            }
        }

        Lock getLock(ThriftTransportKey key) {
            Lock lock = this.locks[(key.hashCode() & Integer.MAX_VALUE) % this.locks.length];
            lock.lock();
            if (this.shutdown) {
                lock.unlock();
                throw new TransportPoolShutdownException("The Accumulo singleton for connection pooling is disabled.  This is likely caused by all AccumuloClients being closed or garbage collected.");
            }
            return lock;
        }

        CachedConnections getCachedConnections(ThriftTransportKey key) {
            return this.connections.get(key);
        }

        CachedConnections getOrCreateCachedConnections(ThriftTransportKey key) {
            return this.connections.computeIfAbsent(key, k -> new CachedConnections());
        }

        boolean unreserveConnection(CachedTTransport transport, CachedConnections connections, List<CachedConnection> toBeClosed) {
            CachedConnection connection;
            if (connections != null && (connection = connections.removeReserved(transport)) != null) {
                if (transport.sawError) {
                    this.unreserveConnectionAndClearUnreserved(connections, connection, toBeClosed);
                } else {
                    this.returnConnectionToUnreserved(connections, connection);
                }
                return true;
            }
            return false;
        }

        void unreserveConnectionAndClearUnreserved(CachedConnections connections, CachedConnection connection, List<CachedConnection> toBeClosed) {
            toBeClosed.add(connection);
            connection.unreserve();
            toBeClosed.addAll(connections.unreserved);
            connections.unreserved.clear();
        }

        void returnConnectionToUnreserved(CachedConnections connections, CachedConnection connection) {
            log.trace("Returned connection {} ioCount: {}", (Object)connection.transport.getCacheKey(), (Object)connection.transport.ioCount);
            connection.lastReturnTime = System.currentTimeMillis();
            connection.unreserve();
            connections.unreserved.addFirst(connection);
        }

        List<CachedConnection> removeExpiredConnections(LongSupplier maxAgeMillis) {
            ArrayList<CachedConnection> expired = new ArrayList<CachedConnection>();
            for (Map.Entry<ThriftTransportKey, CachedConnections> entry : this.connections.entrySet()) {
                CachedConnections connections = entry.getValue();
                this.executeWithinLock(entry.getKey(), (ThriftTransportKey key) -> {
                    connections.removeExpiredConnections(expired, maxAgeMillis);
                    connections.checkReservedForStuckIO();
                });
            }
            return expired;
        }
    }

    private static class CachedConnections {
        Deque<CachedConnection> unreserved = new ArrayDeque<CachedConnection>();
        Map<CachedTTransport, CachedConnection> reserved = new HashMap<CachedTTransport, CachedConnection>();

        private CachedConnections() {
        }

        public CachedConnection reserveAny() {
            CachedConnection cachedConnection = this.unreserved.pollFirst();
            if (cachedConnection != null) {
                cachedConnection.reserve();
                this.reserved.put(cachedConnection.transport, cachedConnection);
                if (log.isTraceEnabled()) {
                    log.trace("Using existing connection to {}", (Object)cachedConnection.transport.cacheKey);
                }
            }
            return cachedConnection;
        }

        private void removeExpiredConnections(ArrayList<CachedConnection> expired, LongSupplier maxAgeMillis) {
            long currTime = System.currentTimeMillis();
            while (this.isLastUnreservedExpired(currTime, maxAgeMillis)) {
                expired.add(this.unreserved.removeLast());
            }
        }

        boolean isLastUnreservedExpired(long currTime, LongSupplier maxAgeMillis) {
            return !this.unreserved.isEmpty() && currTime - this.unreserved.peekLast().lastReturnTime > maxAgeMillis.getAsLong();
        }

        void checkReservedForStuckIO() {
            this.reserved.values().forEach(c -> c.transport.checkForStuckIO(STUCK_THRESHOLD));
        }

        void closeAllTransports() {
            this.closeTransports(this.unreserved);
            this.closeTransports(this.reserved.values());
        }

        void closeTransports(Iterable<CachedConnection> stream) {
            stream.forEach(connection -> {
                try {
                    connection.transport.close();
                }
                catch (Exception e) {
                    log.debug("Error closing transport during shutdown", (Throwable)e);
                }
            });
        }

        CachedConnection removeReserved(CachedTTransport transport) {
            return this.reserved.remove((Object)transport);
        }
    }
}

