/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.util;

import com.google.common.collect.Sets;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.server.util.Halt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemMonitor {
    private static final String PROC_MOUNTS = "/proc/mounts";
    private static final Logger log = LoggerFactory.getLogger(FileSystemMonitor.class);
    private static final String DEVICE_PREFIX = "/dev/";
    private static final Set<String> ACCEPTED_FILESYSTEMS = Sets.newHashSet((Object[])new String[]{"ext3", "ext4", "xfs"});
    private Map<String, Boolean> readWriteFilesystems = new HashMap<String, Boolean>();

    @SuppressFBWarnings(value={"PATH_TRAVERSAL_IN"}, justification="procFile path not from user input")
    static List<Mount> parse(String procFile) throws IOException {
        FileReader fr = new FileReader(procFile);
        try (BufferedReader br = new BufferedReader(fr);){
            List<Mount> list = FileSystemMonitor.getMountsFromFile(br);
            return list;
        }
    }

    static List<Mount> getMountsFromFile(BufferedReader br) throws IOException {
        String line;
        ArrayList<Mount> mounts = new ArrayList<Mount>();
        while ((line = br.readLine()) != null) {
            Mount mount = new Mount(line);
            if (mount.device.startsWith(DEVICE_PREFIX) && ACCEPTED_FILESYSTEMS.contains(mount.filesystemType)) {
                log.trace("Retaining mount to check: '{}'", (Object)mount);
                mounts.add(mount);
                continue;
            }
            log.trace("Ignoring mount to check: '{}'", (Object)mount);
        }
        return mounts;
    }

    public FileSystemMonitor(final String procFile, long period) throws IOException {
        List<Mount> mounts = FileSystemMonitor.parse(procFile);
        for (Mount mount : mounts) {
            if (mount.options.contains("rw")) {
                this.readWriteFilesystems.put(mount.mountPoint, true);
                continue;
            }
            if (mount.options.contains("ro")) {
                this.readWriteFilesystems.put(mount.mountPoint, false);
                continue;
            }
            throw new IOException("Filesystem " + mount + " does not have ro or rw option");
        }
        TimerTask tt = new TimerTask(){

            @Override
            public void run() {
                try {
                    FileSystemMonitor.this.checkMounts(procFile);
                }
                catch (Exception e) {
                    Halt.halt(-42, new Runnable(){

                        @Override
                        public void run() {
                            log.error("Exception while checking mount points, halting process", (Throwable)e);
                        }
                    });
                }
            }
        };
        Timer timer = new Timer("filesystem monitor timer", true);
        timer.schedule(tt, period, period);
    }

    protected void checkMounts(String procFile) throws Exception {
        List<Mount> mounts = FileSystemMonitor.parse(procFile);
        for (Mount mount : mounts) {
            if (!this.readWriteFilesystems.containsKey(mount.mountPoint)) {
                if (mount.options.contains("rw")) {
                    this.readWriteFilesystems.put(mount.mountPoint, true);
                    continue;
                }
                if (mount.options.contains("ro")) {
                    this.readWriteFilesystems.put(mount.mountPoint, false);
                    continue;
                }
                throw new Exception("Filesystem " + mount + " does not have ro or rw option");
            }
            if (!mount.options.contains("ro") || !this.readWriteFilesystems.get(mount.mountPoint).booleanValue()) continue;
            throw new Exception("Filesystem " + mount.mountPoint + " switched to read only");
        }
    }

    public static void start(AccumuloConfiguration conf, Property prop) {
        if (conf.getBoolean(prop)) {
            if (new File(PROC_MOUNTS).exists()) {
                try {
                    new FileSystemMonitor(PROC_MOUNTS, 60000L);
                    log.info("Filesystem monitor started");
                }
                catch (IOException e) {
                    log.error("Failed to initialize file system monitor", (Throwable)e);
                }
            } else {
                log.info("Not monitoring filesystems, /proc/mounts does not exists");
            }
        }
    }

    static class Mount {
        String device;
        String mountPoint;
        String filesystemType;
        Set<String> options;

        Mount(String line) {
            String[] tokens = line.split("\\s+");
            this.device = tokens[0].trim();
            this.mountPoint = tokens[1].trim();
            this.filesystemType = tokens[2].trim().toLowerCase();
            this.options = new HashSet<String>(Arrays.asList(tokens[3].split(",")));
        }

        public String toString() {
            StringBuilder sb = new StringBuilder(32);
            sb.append(this.device).append(" on ").append(this.mountPoint).append(" as ").append(this.filesystemType).append(" with options ").append(this.options);
            return sb.toString();
        }
    }
}

