/* operation.c - deal with operation subsystem */
/* $OpenLDAP: pkg/ldap/servers/slapd/back-monitor/operation.c,v 1.36 2004/11/08 17:52:27 ando Exp $ */
/* This work is part of OpenLDAP Software <http://www.openldap.org/>.
 *
 * Copyright 2001-2004 The OpenLDAP Foundation.
 * Portions Copyright 2001-2003 Pierangelo Masarati.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted only as authorized by the OpenLDAP
 * Public License.
 *
 * A copy of this license is available in file LICENSE in the
 * top-level directory of the distribution or, alternatively, at
 * <http://www.OpenLDAP.org/license.html>.
 */
/* ACKNOWLEDGEMENTS:
 * This work was initially developed by Pierangelo Masarati for inclusion
 * in OpenLDAP Software.
 */

#include "portable.h"

#include <stdio.h>
#include <ac/string.h>

#include "slap.h"
#include "back-monitor.h"
#include "lber_pvt.h"

struct monitor_ops_t {
	struct berval	rdn;
	struct berval	nrdn;
} monitor_op[] = {
	{ BER_BVC( "cn=Bind" ),		BER_BVNULL },
	{ BER_BVC( "cn=Unbind" ),	BER_BVNULL },
	{ BER_BVC( "cn=Add" ),		BER_BVNULL },
	{ BER_BVC( "cn=Delete" ),	BER_BVNULL },
	{ BER_BVC( "cn=Modrdn" ),	BER_BVNULL },
	{ BER_BVC( "cn=Modify" ),	BER_BVNULL },
	{ BER_BVC( "cn=Compare" ),	BER_BVNULL },
	{ BER_BVC( "cn=Search" ),	BER_BVNULL },
	{ BER_BVC( "cn=Abandon" ),	BER_BVNULL },
	{ BER_BVC( "cn=Extended" ),	BER_BVNULL },
	{ BER_BVNULL,			BER_BVNULL }
};

int
monitor_subsys_ops_init(
	BackendDB		*be,
	monitor_subsys_t	*ms
)
{
	monitor_info_t	*mi;
	
	Entry		*e_op, **ep;
	monitor_entry_t	*mp;
	char		buf[ BACKMONITOR_BUFSIZE ];
	int 		i;
	struct berval	bv_zero = BER_BVC( "0" );

	assert( be != NULL );

	mi = ( monitor_info_t * )be->be_private;

	if ( monitor_cache_get( mi,
			&ms->mss_ndn, &e_op ) )
	{
		Debug( LDAP_DEBUG_ANY,
			"monitor_subsys_ops_init: "
			"unable to get entry \"%s\"\n",
			ms->mss_ndn.bv_val, 
			0, 0 );
		return( -1 );
	}

	attr_merge_one( e_op, mi->mi_ad_monitorOpInitiated, &bv_zero, NULL );
	attr_merge_one( e_op, mi->mi_ad_monitorOpCompleted, &bv_zero, NULL );

	mp = ( monitor_entry_t * )e_op->e_private;
	mp->mp_children = NULL;
	ep = &mp->mp_children;

	for ( i = 0; i < SLAP_OP_LAST; i++ ) {
		struct berval	rdn;
		Entry		*e;

		/*
		 * Initiated ops
		 */
		snprintf( buf, sizeof( buf ),
				"dn: %s,%s\n"
				"objectClass: %s\n"
				"structuralObjectClass: %s\n"
				"cn: %s\n"
				"%s: 0\n"
				"%s: 0\n"
				"creatorsName: %s\n"
				"modifiersName: %s\n"
				"createTimestamp: %s\n"
				"modifyTimestamp: %s\n",
				monitor_op[ i ].rdn.bv_val,
				ms->mss_dn.bv_val,
				mi->mi_oc_monitorOperation->soc_cname.bv_val,
				mi->mi_oc_monitorOperation->soc_cname.bv_val,
				&monitor_op[ i ].rdn.bv_val[ STRLENOF( "cn=" ) ],
				mi->mi_ad_monitorOpInitiated->ad_cname.bv_val,
				mi->mi_ad_monitorOpCompleted->ad_cname.bv_val,
				mi->mi_creatorsName.bv_val,
				mi->mi_creatorsName.bv_val,
				mi->mi_startTime.bv_val,
				mi->mi_startTime.bv_val );

		e = str2entry( buf );
		if ( e == NULL ) {
			Debug( LDAP_DEBUG_ANY,
				"monitor_subsys_ops_init: "
				"unable to create entry \"%s,%s\"\n",
				monitor_op[ i ].rdn.bv_val,
				ms->mss_ndn.bv_val, 0 );
			return( -1 );
		}
	
		/* steal normalized RDN */
		dnRdn( &e->e_nname, &rdn );
		ber_dupbv( &monitor_op[ i ].nrdn, &rdn );
	
		mp = monitor_entrypriv_create();
		if ( mp == NULL ) {
			return -1;
		}
		e->e_private = ( void * )mp;
		mp->mp_info = ms;
		mp->mp_flags = ms->mss_flags \
			| MONITOR_F_SUB | MONITOR_F_PERSISTENT;

		if ( monitor_cache_add( mi, e ) ) {
			Debug( LDAP_DEBUG_ANY,
				"monitor_subsys_ops_init: "
				"unable to add entry \"%s,%s\"\n",
				monitor_op[ i ].rdn.bv_val,
				ms->mss_ndn.bv_val, 0 );
			return( -1 );
		}

		*ep = e;
		ep = &mp->mp_next;
	}

	monitor_cache_release( mi, e_op );

	return( 0 );
}

int
monitor_subsys_ops_update(
	Operation		*op,
	Entry                   *e
)
{
	monitor_info_t		*mi = ( monitor_info_t * )op->o_bd->be_private;

	ldap_pvt_mp_t		nInitiated,
				nCompleted;
	struct berval		rdn;
	int 			i;
	Attribute		*a;
	static struct berval	bv_ops = BER_BVC( "cn=operations" );

	assert( mi );
	assert( e );

	dnRdn( &e->e_nname, &rdn );

	if ( dn_match( &rdn, &bv_ops ) ) {
		ldap_pvt_mp_init( nInitiated );
		ldap_pvt_mp_init( nCompleted );

		ldap_pvt_thread_mutex_lock( &slap_counters.sc_ops_mutex );
		for ( i = 0; i < SLAP_OP_LAST; i++ ) {
			ldap_pvt_mp_add( nInitiated, slap_counters.sc_ops_initiated_[ i ] );
			ldap_pvt_mp_add( nCompleted, slap_counters.sc_ops_completed_[ i ] );
		}
		ldap_pvt_thread_mutex_unlock( &slap_counters.sc_ops_mutex );
		
	} else {
		for ( i = 0; i < SLAP_OP_LAST; i++ ) {
			if ( dn_match( &rdn, &monitor_op[ i ].nrdn ) )
			{
				ldap_pvt_thread_mutex_lock( &slap_counters.sc_ops_mutex );
				ldap_pvt_mp_init_set( nInitiated, slap_counters.sc_ops_initiated_[ i ] );
				ldap_pvt_mp_init_set( nCompleted, slap_counters.sc_ops_completed_[ i ] );
				ldap_pvt_thread_mutex_unlock( &slap_counters.sc_ops_mutex );
				break;
			}
		}

		if ( i == SLAP_OP_LAST ) {
			/* not found ... */
			return( 0 );
		}
	}

	a = attr_find( e->e_attrs, mi->mi_ad_monitorOpInitiated );
	assert ( a != NULL );

	/* NOTE: no minus sign is allowed in the counters... */
	UI2BV( &a->a_vals[ 0 ], nInitiated );
	ldap_pvt_mp_clear( nInitiated );
	
	a = attr_find( e->e_attrs, mi->mi_ad_monitorOpCompleted );
	assert ( a != NULL );

	/* NOTE: no minus sign is allowed in the counters... */
	UI2BV( &a->a_vals[ 0 ], nCompleted );
	ldap_pvt_mp_clear( nCompleted );

	return( 0 );
}

