/*
*  utils.c:  String and other utility functions.
*
*
*  Copyright (c) 2000 Thomas Esken
*
*  This software doesn't claim completeness, correctness or usability.
*  On principle I will not be liable for ANY damages or losses (implicit
*  or explicit), which result from using or handling my software.
*  If you use this software, you agree without any exception to this
*  agreement, which binds you LEGALLY !!
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the `GNU General Public License' as published by
*  the `Free Software Foundation'; either version 2, or (at your option)
*  any later version.
*
*  You should have received a copy of the `GNU General Public License'
*  along with this program; if not, write to the:
*
*    Free Software Foundation, Inc.
*    59 Temple Place - Suite 330
*    Boston, MA 02111-1307,  USA
*/



#ifdef RCSID
static char rcsid[]="$Id: utils.c 0.05 2000/06/06 00:00:05 tom Exp $";
#endif



/*
*  Include header files.
*/
#include "tailor.h"
#if HAVE_CTYPE_H
#  include <ctype.h>
#endif
#if HAVE_ERRNO_H
#  include <errno.h>
#endif
#include "common.h"
#include "dvichk.h"
#include "help.h"
#include "utils.h"
#include "globals.h"



/*
*  Function implementations.
*/
#if !HAVE_STRDUP
   PUBLIC char *
my_strdup (string)
   const char *string;
/*
   Returns a copy of STRING.
*/
{
   auto char  *str;


   str = (char *)my_malloc(strlen(string)+1, ERR_NO_MEMORY_AVAILABLE,
                           __FILE__, ((long)__LINE__)-1L);
   strcpy(str, string);

   return(str);
}
#endif



#if !HAVE_STRSTR
   PUBLIC char *
my_strstr (text, pattern)
   const char *text;
   const char *pattern;
/*
   Search the `pattern' needle in haystack `text'   =8^)
     (emulates the ANSI C strstr() function; not very well optimized).
*/
{
   register       int    i=(int)strlen(text);
   register       int    j=(int)strlen(pattern);
   register       int    k;
   auto     const char  *ptr_char=text;


   if (!j)
     return(ptr_char);
   if (i < j)
     return(NULL);
   for (i=0 ; *ptr_char ; i++,ptr_char++)
    {
      for (k=0,j=i ; pat[k] && (text[j]==pattern[k]) ; j++,k++)
        ;
      if (   k
          && !pattern[k])
        return(ptr_char);
    }

   return(NULL);
}
#endif /* !HAVE_STRSTR */



   PUBLIC char *
my_strrev (string)
   const char *string;
/*
   Returns a reverted copy of STRING.
*/
{
   register int    len=0;
   auto     char  *str;
   auto     char  *ptr_str;


   ptr_str=str = strdup (string);
   while (*string)
    {
      string++;
      len++;
    }
   string--;
   while (len--)
     *ptr_str++ = *string--;

   return(str);
}



   PUBLIC int
asc_sort (a, b)
   const char **a;
   const char **b;
/*
   The (q)sort compare function; ascending order.
*/
{
   return(strcmp(*a, *b));
}



   PUBLIC VOID_PTR
my_malloc (amount, exit_status, module_name, module_line)
   const int   amount;
   const int   exit_status;
   const char *module_name;
   const long  module_line;
/*
   Allocates AMOUNT bytes of memory dynamically, with error checking.
     Calls `my_error()' and terminates the program if any errors occur.
     AMOUNT is limited to `int' range instead of `size_t' range;
     this is wanted!
*/
{
   auto VOID_PTR  ptr_memblock;


   if ((Uint)amount > testval)
     /*
        Error, table size overflow!
     */
     (void)my_error (stderr, ERR_INTERNAL_TABLE_CRASH, module_name, module_line);
   ptr_memblock = (VOID_PTR)malloc((int)amount);
   if (ptr_memblock == (VOID_PTR)NULL)
     /*
        Error, `malloc()' function failed.
     */
     (void)my_error (stderr, exit_status, module_name, module_line);

   return(ptr_memblock);
}



   PUBLIC VOID_PTR
my_realloc (ptr_memblock, amount, exit_status, module_name, module_line)
         VOID_PTR  ptr_memblock;
   const int       amount;
   const int       exit_status;
   const char     *module_name;
   const long      module_line;
/*
   Changes the size of an allocated block of memory PTR_MEMBLOCK to AMOUNT
     bytes, with error checking.  Calls `my_error()' and terminates the program
     if any errors occur.  AMOUNT is limited to `int' range instead of `size_t'
     range; this is wanted!  If PTR_MEMBLOCK is NULL, `my_malloc()' is called
     instead.
*/
{
   if ((Uint)amount > testval)
     /*
        Error, table size overflow!
     */
     (void)my_error (stderr, ERR_INTERNAL_TABLE_CRASH, module_name, module_line);
   if (ptr_memblock == (VOID_PTR)NULL)
     return(my_malloc (amount, exit_status, module_name, module_line));
   ptr_memblock = (VOID_PTR)realloc(ptr_memblock, (int)amount);
   if (ptr_memblock == (VOID_PTR)NULL)
     /*
        Error, `realloc()' function failed.
     */
     (void)my_error (stderr, exit_status, module_name, module_line);

   return(ptr_memblock);
}



   PUBLIC int
my_error (stream, error, module_name, module_line)
         FILE *stream;
   const int   error;
   const char *module_name;
   const long  module_line;
/*
   Displays MSG on STREAM and terminates the program with ERROR status.
*/
{
   if (prgr_name == (char *)NULL)
     prgr_name = PACKAGE;
   switch (error)
    {
      case ERR_ZERO_FILE:
      case ERR_IRREGULAR_FILE:
      case ERR_CORRUPTED_FILE:
      case ERR_FILE_NOT_FOUND:
#if USE_DE
        fprintf(stream, "%s: Warnung, ", prgr_name);
#else /* !USE_DE */
        fprintf(stream, _("%s: warning, "), prgr_name);
#endif /* !USE_DE */
        break;
      default:
#if USE_DE
        fprintf(stream, "\n%s: Abbruch, ", prgr_name);
#else /* !USE_DE */
        fprintf(stream, _("\n%s: abort, "), prgr_name);
#endif /* !USE_DE */
    }
   switch (error)
    {
#if USE_DE
      case ERR_C_FUNCTION_FAILURE:
        fprintf(stream, "`%s' Zeile %ld: C-Funktion hat versagt",
                module_name, module_line);
        break;
      case ERR_NO_MEMORY_AVAILABLE:
        fprintf(stream, "`%s' Zeile %ld: virtueller Speicher ersch%spft",
                module_name, module_line, OE);
        break;
      case ERR_INTERNAL_TABLE_CRASH:
        fprintf(stream, "`%s' Zeile %ld: ung%sltiger Wert f%sr Tabellengr%s%se",
                module_name, module_line, UE, UE, OE, SZ);
        break;
      case ERR_ZERO_FILE:
      case ERR_IRREGULAR_FILE:
        fprintf(stream, "Datei `%s' ist keine regul%sre Datei", module_name, AE);
        break;
      case ERR_CORRUPTED_FILE:
        fprintf(stream, "kann Datei `%s' nicht verarbeiten", module_name);
        break;
      case ERR_CLOSE_FILE:
        fprintf(stream, "kann Datei `%s' nicht schlie%sen", module_name, SZ);
        break;
      case ERR_READ_FILE:
        fprintf(stream, "kann Datei `%s' nicht lesen", module_name);
        break;
      case ERR_FILE_NOT_FOUND:
        fprintf(stream, "Datei `%s' nicht gefunden", module_name);
        break;
      case ERR_NO_FILENAME_GIVEN:
        fputs("kein Dateiname angegeben\n\n", stream);
        my_usage (stream);
        break;
      case ERR_INVALID_OPTION:
        fprintf(stream, "ung%sltige Option in der Kommandozeile angegeben\n\n",
                UE);
        my_usage (stream);
        break;
      default:
        fprintf(stream, "`%s' Zeile %ld: unbehandelter Fehler (%d)",
                module_name, module_line, error);
#else /* !USE_DE */
      case ERR_C_FUNCTION_FAILURE:
        fprintf(stream, _("`%s' line %ld: C-function failure"),
                module_name, module_line);
        break;
      case ERR_NO_MEMORY_AVAILABLE:
        fprintf(stream, _("`%s' line %ld: virtual memory exhausted"),
                module_name, module_line);
        break;
      case ERR_INTERNAL_TABLE_CRASH:
        fprintf(stream, _("`%s' line %ld: invalid value for table size"),
                module_name, module_line);
        break;
      case ERR_ZERO_FILE:
      case ERR_IRREGULAR_FILE:
        fprintf(stream, _("file `%s' is no regular file"), module_name);
        break; 
      case ERR_CORRUPTED_FILE:
        fprintf(stream, _("file `%s' is corrupted"), module_name);
        break;
      case ERR_CLOSE_FILE:
        fprintf(stream, _("cannot close file `%s'"), module_name);
        break;
      case ERR_READ_FILE:
        fprintf(stream, _("cannot read file `%s'"), module_name);
        break;
      case ERR_FILE_NOT_FOUND:
        fprintf(stream, _("file `%s' not found"), module_name);
        break;
      case ERR_NO_FILENAME_GIVEN:
        fputs(_("no filename given\n\n"), stream);
        my_usage (stream);
        break;
      case ERR_INVALID_OPTION:
        fputs(_("invalid option specified in command line\n\n"), stream);
        my_usage (stream);
        break;
      default:
        fprintf(stream, _("`%s' line %ld: unmanaged error (%d)"),
                module_name, module_line, error);
#endif /* !USE_DE */
    }
   S_NEWLINE(stream);
   switch (error)
    {
      case ERR_ZERO_FILE:
      case ERR_IRREGULAR_FILE:
      case ERR_CORRUPTED_FILE:
      case ERR_FILE_NOT_FOUND:
        break;
      default:
        exit(error);
    }

   return(error);
}
