package RISCOS::Chunk;

use Carp;

use strict;
use vars qw (@ISA $VERSION);

$VERSION = 0.02;
@ISA = qw();

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self  = {};
    $self->{__ID} = shift;
    $self->{__DATA} = shift;
    
    my $length = shift;

    my $index = shift;
    $self->{__INDEX} = $index if defined $index;
    my $offset = shift;
    $self->{__OFFSET} = $offset if defined $offset;

    if (defined $length) {
	carp "Chunk size mismatch - expected $length, got "
	  . length $self->{__DATA} unless $length == length $self->{__DATA};
    }

    bless ($self, $class);
}

sub ID {
    my $self = shift;
    $self->{'__ID'};
}

sub Data {
    my $self = shift;
    $self->{'__DATA'};
}

sub Length {
    my $self = shift;
    length $self->{'__DATA'};
}

sub Index {
    my $self = shift;
    $self->{'__INDEX'};
}

sub Offset {
    my $self = shift;
    $self->{'__OFFSET'};
}

1;
__END__

=head1 NAME

RISCOS::Chunk -- manipulate Chunks in Acorn Chunkfiles

=head1 SYNOPSIS

    use RISCOS::Chunk;
    
    $chunk = new RISCOS::Chunk $chunkid, substr ($file, $offset, $size), $size,
			       $index, $offset;

=head1 DESCRIPTION

C<RISCOS::Chunk> provides a class to hold details about a chunk in a chunkfile.

=head2 Methods

=over 4

=item new <ID>, <data>, [<length>, [<index>, [<offset>]]]

Returns a new C<RISCOS::Chunk> object with a ChunkID I<ID>, and contents
I<data>. If <length> is defined it is checked against C<length I<data>>. The
I<index> of the chunk in the chunkfile header and the I<offset> of the chunk
data in the chunkfile are stored if supplied.

=item ID

Returns the chunk ID.

=item Data

Returns the chunk data.

=item Length

Returns the length of the chunk data - I<i.e.> literally C<length Data()>

=item Index

Returns the index of the chunk in the chunkfile header (if known, else returns
undefined);

=item Offset

Returns the offset of the chunk data in the chunkfile (if known, else returns
undefined);

=back

=head1 BUGS

None known.

=head1 AUTHOR

Nicholas Clark <F<nick@unfortu.net>>
