package Bencher::Scenario::StringFunctions::Trim;

use strict;
use warnings;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2022-03-27'; # DATE
our $DIST = 'Bencher-Scenarios-StringFunctions'; # DIST
our $VERSION = '0.006'; # VERSION

our $scenario = {
    summary => "Benchmark string trimming (removing whitespace at the start and end of string)",
    participants => [
        #{fcall_template=>'String::Trim::trim(<str>)'}, # currently disabled, see https://github.com/doherty/String-Trim/issues/7
        {fcall_template=>'String::Trim::More::trim(<str>)'},
        {fcall_template=>'String::Trim::NonRegex::trim(<str>)'},
        {fcall_template=>'String::Trim::Regex::trim(<str>)'},
        {fcall_template=>'String::Util::trim(<str>)'},
        {fcall_template=>'Text::Minify::XS::minify(<str>)'},
        {fcall_template=>'Text::Minify::XS::minify_ascii(<str>)'},
    ],
    datasets => [
        {name=>'empty'        , args=>{str=>''}},
        {name=>'len10ws1'     , args=>{str=>' '.('x' x   10).' '}},
        {name=>'len100ws1'    , args=>{str=>' '.('x' x  100).' '}},
        {name=>'len100ws10'   , args=>{str=>(' ' x   10).('x' x  100).(' ' x 10)}},
        {name=>'len100ws100'  , args=>{str=>(' ' x  100).('x' x  100).(' ' x 100)}},
        {name=>'len1000ws1'   , args=>{str=>' '.('x' x 1000).' '}},
        {name=>'len1000ws10'  , args=>{str=>(' ' x   10).('x' x 1000).(' ' x 10)}},
        {name=>'len1000ws100' , args=>{str=>(' ' x  100).('x' x 1000).(' ' x 100)}},
        {name=>'len1000ws1000', args=>{str=>(' ' x 1000).('x' x 1000).(' ' x 1000)}},
    ],
};

1;
# ABSTRACT: Benchmark string trimming (removing whitespace at the start and end of string)

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::StringFunctions::Trim - Benchmark string trimming (removing whitespace at the start and end of string)

=head1 VERSION

This document describes version 0.006 of Bencher::Scenario::StringFunctions::Trim (from Perl distribution Bencher-Scenarios-StringFunctions), released on 2022-03-27.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m StringFunctions::Trim

To run module startup overhead benchmark:

 % bencher --module-startup -m StringFunctions::Trim

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<String::Trim::More> 0.03

L<String::Trim::NonRegex> 0.002

L<String::Trim::Regex> 20210604

L<String::Util> 1.32

L<Text::Minify::XS> v0.6.1

=head1 BENCHMARK PARTICIPANTS

=over

=item * String::Trim::More::trim (perl_code)

Function call template:

 String::Trim::More::trim(<str>)



=item * String::Trim::NonRegex::trim (perl_code)

Function call template:

 String::Trim::NonRegex::trim(<str>)



=item * String::Trim::Regex::trim (perl_code)

Function call template:

 String::Trim::Regex::trim(<str>)



=item * String::Util::trim (perl_code)

Function call template:

 String::Util::trim(<str>)



=item * Text::Minify::XS::minify (perl_code)

Function call template:

 Text::Minify::XS::minify(<str>)



=item * Text::Minify::XS::minify_ascii (perl_code)

Function call template:

 Text::Minify::XS::minify_ascii(<str>)



=back

=head1 BENCHMARK DATASETS

=over

=item * empty

=item * len10ws1

=item * len100ws1

=item * len100ws10

=item * len100ws100

=item * len1000ws1

=item * len1000ws10

=item * len1000ws100

=item * len1000ws1000

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.34.0 >>, CPU: I<< Intel(R) Core(TM) i5-7200U CPU @ 2.50GHz (2 cores) >>, OS: I<< GNU/Linux Ubuntu version 20.04 >>, OS kernel: I<< Linux version 5.4.0-91-generic >>.

Benchmark with default options (C<< bencher -m StringFunctions::Trim >>):

 #table1#
 {dataset=>"empty"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Util::trim             |   1494940 |   668.924 |                 0.00% |               444.35% |   0     |      22 |
 | String::Trim::NonRegex::trim   |   1574290 |   635.205 |                 5.31% |               416.91% |   0     |      24 |
 | String::Trim::More::trim       |   3900000 |   260     |               157.92% |               111.05% | 4.4e-10 |      20 |
 | String::Trim::Regex::trim      |   5850660 |   170.921 |               291.36% |                39.09% |   0     |      20 |
 | Text::Minify::XS::minify       |   8120000 |   123     |               443.46% |                 0.16% | 4.6e-11 |      26 |
 | Text::Minify::XS::minify_ascii |   8100000 |   120     |               444.35% |                 0.00% | 2.1e-10 |      20 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate  SU:t  STN:t  STM:t  STR:t  TMX:m  TMX:m_a 
  SU:t     1494940/s    --    -5%   -61%   -74%   -81%     -82% 
  STN:t    1574290/s    5%     --   -59%   -73%   -80%     -81% 
  STM:t    3900000/s  157%   144%     --   -34%   -52%     -53% 
  STR:t    5850660/s  291%   271%    52%     --   -28%     -29% 
  TMX:m    8120000/s  443%   416%   111%    38%     --      -2% 
  TMX:m_a  8100000/s  457%   429%   116%    42%     2%       -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,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" />


 #table2#
 {dataset=>"len1000ws1"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim      |   18025.2 |  55.4778  |                 0.00% |              5589.45% |   0     |      20 |
 | String::Util::trim             |   23135.7 |  43.2232  |                28.35% |              4332.70% | 5.8e-12 |      20 |
 | Text::Minify::XS::minify       |  317000   |   3.15    |              1660.22% |               223.22% | 1.5e-09 |      24 |
 | String::Trim::More::trim       |  382825   |   2.61216 |              2023.83% |               167.89% |   0     |      21 |
 | Text::Minify::XS::minify_ascii |  405721   |   2.46475 |              2150.85% |               152.77% |   0     |      20 |
 | String::Trim::NonRegex::trim   | 1000000   |   0.98    |              5589.45% |                 0.00% | 1.2e-09 |      20 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate  STR:t   SU:t  TMX:m  STM:t  TMX:m_a  STN:t 
  STR:t    18025.2/s     --   -22%   -94%   -95%     -95%   -98% 
  SU:t     23135.7/s    28%     --   -92%   -93%     -94%   -97% 
  TMX:m     317000/s  1661%  1272%     --   -17%     -21%   -68% 
  STM:t     382825/s  2023%  1554%    20%     --      -5%   -62% 
  TMX:m_a   405721/s  2150%  1653%    27%     5%       --   -60% 
  STN:t    1000000/s  5561%  4310%   221%   166%     151%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,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" />


 #table3#
 {dataset=>"len1000ws10"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim      |     18100 |  55.4     |                 0.00% |              2117.58% | 2.4e-08 |      25 |
 | String::Util::trim             |     23100 |  43.2     |                28.18% |              1630.11% | 9.9e-09 |      36 |
 | String::Trim::NonRegex::trim   |    300724 |   3.32531 |              1566.06% |                33.10% |   0     |      20 |
 | Text::Minify::XS::minify       |    312634 |   3.19863 |              1632.04% |                28.03% |   0     |      20 |
 | String::Trim::More::trim       |    383685 |   2.60631 |              2025.68% |                 4.32% |   0     |      20 |
 | Text::Minify::XS::minify_ascii |    400273 |   2.49829 |              2117.58% |                 0.00% |   0     |      20 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

               Rate  STR:t   SU:t  STN:t  TMX:m  STM:t  TMX:m_a 
  STR:t     18100/s     --   -22%   -93%   -94%   -95%     -95% 
  SU:t      23100/s    28%     --   -92%   -92%   -93%     -94% 
  STN:t    300724/s  1566%  1199%     --    -3%   -21%     -24% 
  TMX:m    312634/s  1631%  1250%     3%     --   -18%     -21% 
  STM:t    383685/s  2025%  1557%    27%    22%     --      -4% 
  TMX:m_a  400273/s  2117%  1629%    33%    28%     4%       -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,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" />


 #table4#
 {dataset=>"len1000ws100"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim      |   18000   |  55.6     |                 0.00% |              1900.90% | 2.2e-08 |      30 |
 | String::Util::trim             |   23008.8 |  43.4616  |                28.04% |              1462.70% |   0     |      20 |
 | String::Trim::NonRegex::trim   |   34000   |  29       |                91.13% |               946.90% | 1.1e-07 |      22 |
 | Text::Minify::XS::minify       |  267998   |   3.73137 |              1391.38% |                34.17% |   0     |      20 |
 | Text::Minify::XS::minify_ascii |  351980   |   2.8411  |              1858.74% |                 2.15% | 5.7e-12 |      20 |
 | String::Trim::More::trim       |  359560   |   2.78118 |              1900.90% |                 0.00% |   0     |      20 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate  STR:t   SU:t  STN:t  TMX:m  TMX:m_a  STM:t 
  STR:t      18000/s     --   -21%   -47%   -93%     -94%   -94% 
  SU:t     23008.8/s    27%     --   -33%   -91%     -93%   -93% 
  STN:t      34000/s    91%    49%     --   -87%     -90%   -90% 
  TMX:m     267998/s  1390%  1064%   677%     --     -23%   -25% 
  TMX:m_a   351980/s  1856%  1429%   920%    31%       --    -2% 
  STM:t     359560/s  1899%  1462%   942%    34%       2%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,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" />


 #table5#
 {dataset=>"len1000ws1000"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::NonRegex::trim   |    4000   | 300       |                 0.00% |              6621.30% | 3.6e-06 |      20 |
 | String::Trim::Regex::trim      |   17614.9 |  56.7701  |               363.96% |              1348.68% |   0     |      25 |
 | String::Util::trim             |   22300   |  44.8     |               488.22% |              1042.65% | 1.2e-08 |      25 |
 | Text::Minify::XS::minify       |  122000   |   8.17    |              3124.73% |               108.43% | 3.1e-09 |      23 |
 | Text::Minify::XS::minify_ascii |  179361   |   5.57536 |              4624.19% |                42.27% |   0     |      20 |
 | String::Trim::More::trim       |  255184   |   3.91875 |              6621.30% |                 0.00% |   0     |      20 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate  STN:t  STR:t   SU:t  TMX:m  TMX:m_a  STM:t 
  STN:t       4000/s     --   -81%   -85%   -97%     -98%   -98% 
  STR:t    17614.9/s   428%     --   -21%   -85%     -90%   -93% 
  SU:t       22300/s   569%    26%     --   -81%     -87%   -91% 
  TMX:m     122000/s  3571%   594%   448%     --     -31%   -52% 
  TMX:m_a   179361/s  5280%   918%   703%    46%       --   -29% 
  STM:t     255184/s  7555%  1348%  1043%   108%      42%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,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" />


 #table6#
 {dataset=>"len100ws1"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim      |    162832 |  6.14131  |                 0.00% |              1567.19% |   0     |      20 |
 | String::Util::trim             |    195849 |  5.10598  |                20.28% |              1286.13% |   0     |      20 |
 | String::Trim::More::trim       |    900000 |  1        |               442.82% |               207.14% | 1.4e-08 |      20 |
 | String::Trim::NonRegex::trim   |   1000000 |  0.96     |               538.52% |               161.10% | 1.7e-09 |      20 |
 | Text::Minify::XS::minify       |   2310000 |  0.432    |              1320.87% |                17.34% |   9e-11 |      20 |
 | Text::Minify::XS::minify_ascii |   2714720 |  0.368362 |              1567.19% |                 0.00% |   0     |      20 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate  STR:t   SU:t  STM:t  STN:t  TMX:m  TMX:m_a 
  STR:t     162832/s     --   -16%   -83%   -84%   -92%     -94% 
  SU:t      195849/s    20%     --   -80%   -81%   -91%     -92% 
  STM:t     900000/s   514%   410%     --    -4%   -56%     -63% 
  STN:t    1000000/s   539%   431%     4%     --   -55%     -61% 
  TMX:m    2310000/s  1321%  1081%   131%   122%     --     -14% 
  TMX:m_a  2714720/s  1567%  1286%   171%   160%    17%       -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJmlDQ1BpY2MAAEiJlZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEWQRQEXXJUia0UUC4uCAhZ0gywCyrpxFVFBWXDfGZ33HT+8/5l7z2/+c+bec8/5cAEgiINlwct7YlK6wNvJjhkYFMwE3yiMn5bC8fR0A9/VuxEArcR7ut/P+a4IEZFp/OW4uLxy+SmCdACg7GXWzEpPWeGjy0wPj//CZ1dYsFzgMt9Y4eh/eexLzr8s+pLj681dfhUKABwp+hsO/4b/c++KVDiC9NioyGymT3JUelaYIJKZttIJHpfL9BQkR8UmRH5T8P+V/B2lR2anr0RucsomQWx0TDrzfw41MjA0BF9n8cbrS48hRv9/z2dFX73kegDYcwAg+7564ZUAdO4CQPrRV09tua+UfAA67vAzBJn/eqiVDQ0IgALoQAYoAlWgCXSBETADlsAWOAAX4AF8QRDYAPggBiQCAcgCuWAHKABFYB84CKpALWgATaAVnAad4Dy4Aq6D2+AuGAaPgRBMgpdABN6BBQiCsBAZokEykBKkDulARhAbsoYcIDfIGwqCQqFoKAnKgHKhnVARVApVQXVQE/QLdA66At2EBqGH0Dg0A/0NfYQRmATTYQVYA9aH2TAHdoV94fVwNJwK58D58F64Aq6HT8Id8BX4NjwMC+GX8BwCECLCQJQRXYSNcBEPJBiJQgTIVqQQKUfqkVakG+lD7iFCZBb5gMKgaCgmShdliXJG+aH4qFTUVlQxqgp1AtWB6kXdQ42jRKjPaDJaHq2DtkDz0IHoaHQWugBdjm5Et6OvoYfRk+h3GAyGgWFhzDDOmCBMHGYzphhzGNOGuYwZxExg5rBYrAxWB2uF9cCGYdOxBdhK7EnsJewQdhL7HkfEKeGMcI64YFwSLg9XjmvGXcQN4aZwC3hxvDreAu+Bj8BvwpfgG/Dd+Dv4SfwCQYLAIlgRfAlxhB2ECkIr4RphjPCGSCSqEM2JXsRY4nZiBfEU8QZxnPiBRCVpk7ikEFIGaS/pOOky6SHpDZlM1iDbkoPJ6eS95CbyVfJT8nsxmpieGE8sQmybWLVYh9iQ2CsKnqJO4VA2UHIo5ZQzlDuUWXG8uIY4VzxMfKt4tfg58VHxOQmahKGEh0SiRLFEs8RNiWkqlqpBdaBGUPOpx6hXqRM0hKZK49L4tJ20Bto12iQdQ2fRefQ4ehH9Z/oAXSRJlTSW9JfMlqyWvCApZCAMDQaPkcAoYZxmjDA+SilIcaQipfZItUoNSc1Ly0nbSkdKF0q3SQ9Lf5RhyjjIxMvsl+mUeSKLktWW9ZLNkj0ie012Vo4uZynHlyuUOy33SB6W15b3lt8sf0y+X35OQVHBSSFFoVLhqsKsIkPRVjFOsUzxouKMEk3JWilWqUzpktILpiSTw0xgVjB7mSJleWVn5QzlOuUB5QUVloqfSp5Km8oTVYIqWzVKtUy1R1WkpqTmrpar1qL2SB2vzlaPUT+k3qc+r8HSCNDYrdGpMc2SZvFYOawW1pgmWdNGM1WzXvO+FkaLrRWvdVjrrjasbaIdo12tfUcH1jHVidU5rDO4Cr3KfFXSqvpVo7okXY5upm6L7rgeQ89NL0+vU++Vvpp+sP5+/T79zwYmBgkGDQaPDamGLoZ5ht2GfxtpG/GNqo3uryavdly9bXXX6tfGOsaRxkeMH5jQTNxNdpv0mHwyNTMVmLaazpipmYWa1ZiNsulsT3Yx+4Y52tzOfJv5efMPFqYW6RanLf6y1LWMt2y2nF7DWhO5pmHNhJWKVZhVnZXQmmkdan3UWmijbBNmU2/zzFbVNsK20XaKo8WJ45zkvLIzsBPYtdvNcy24W7iX7RF7J/tC+wEHqoOfQ5XDU0cVx2jHFkeRk4nTZqfLzmhnV+f9zqM8BR6f18QTuZi5bHHpdSW5+rhWuT5z03YTuHW7w+4u7gfcx9aqr01a2+kBPHgeBzyeeLI8Uz1/9cJ4eXpVez33NvTO9e7zofls9Gn2eedr51vi+9hP0y/Dr8ef4h/i3+Q/H2AfUBogDNQP3BJ4O0g2KDaoKxgb7B/cGDy3zmHdwXWTISYhBSEj61nrs9ff3CC7IWHDhY2UjWEbz4SiQwNCm0MXwzzC6sPmwnnhNeEiPpd/iP8ywjaiLGIm0iqyNHIqyiqqNGo62ir6QPRMjE1MecxsLDe2KvZ1nHNcbdx8vEf88filhICEtkRcYmjiuSRqUnxSb7JicnbyYIpOSkGKMNUi9WCqSOAqaEyD0tandaXTlz/F/gzNjF0Z45nWmdWZ77P8s85kS2QnZfdv0t60Z9NUjmPOT5tRm/mbe3KVc3fkjm/hbKnbCm0N39qzTXVb/rbJ7U7bT+wg7Ijf8VueQV5p3tudATu78xXyt+dP7HLa1VIgViAoGN1tubv2B9QPsT8M7Fm9p3LP58KIwltFBkXlRYvF/OJbPxr+WPHj0t6ovQMlpiVH9mH2Je0b2W+z/0SpRGlO6cQB9wMdZcyywrK3BzcevFluXF57iHAo45Cwwq2iq1Ktcl/lYlVM1XC1XXVbjXzNnpr5wxGHh47YHmmtVagtqv14NPbogzqnuo56jfryY5hjmceeN/g39P3E/qmpUbaxqPHT8aTjwhPeJ3qbzJqamuWbS1rgloyWmZMhJ+/+bP9zV6tua10bo63oFDiVcerFL6G/jJx2Pd1zhn2m9az62Zp2WnthB9SxqUPUGdMp7ArqGjzncq6n27K7/Ve9X4+fVz5ffUHyQslFwsX8i0uXci7NXU65PHsl+spEz8aex1cDr97v9eoduOZ67cZ1x+tX+zh9l25Y3Th/0+LmuVvsW523TW939Jv0t/9m8lv7gOlAxx2zO113ze92D64ZvDhkM3Tlnv296/d5928Prx0eHPEbeTAaMip8EPFg+mHCw9ePMh8tPN4+hh4rfCL+pPyp/NP637V+bxOaCi+M24/3P/N59niCP/Hyj7Q/Fifzn5Ofl08pTTVNG02fn3Gcufti3YvJlykvF2YL/pT4s+aV5quzf9n+1S8KFE2+Frxe+rv4jcyb42+N3/bMec49fZf4bmG+8L3M+xMf2B/6PgZ8nFrIWsQuVnzS+tT92fXz2FLi0tI/QiyQvpNzTVQAAAAgY0hSTQAAeiYAAICEAAD6AAAAgOgAAHUwAADqYAAAOpgAABdwnLpRPAAAAKhQTFRF////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAlADVlQDVlQDWlADUlADUAAAAAAAAlADUlQDVAAAAlADUlADUlADUlADUlQDVAAAAlQDVlQDWmADaAAAAAAAAAAAAAAAAlADU////aywoKwAAADV0Uk5TABFEM2Yiqsy7mXeI3e5Vn2mp1crH0j/v/Pbs+fH0dXqnXN/H5O3W7PAiRDOI9U5pdSBbIDDIW5OrAAAAAWJLR0QAiAUdSAAAAAlwSFlzAAAASAAAAEgARslrPgAAAAd0SU1FB+YDGw8wOt5nUA0AABT3SURBVHja7dwNe+u2eYBhAPwWQG7L2mXpurTrkibLlq37QP//TxsASrKko2MbFmiDr5/numyJlg/s0LcoEJKiFBEREREREREREREREREREREREREREb1j2hyvGH3xxeajfy2ijNrufNX44xV/kq107/3Q5Y5J9GGNZ7z3QE+D1n3/0b8j0WtrhkM4ALfWmgi6S5cJtLG2UdqHCUdnP/qXJHptrZuMGp21cxtAz9PkbQJ9mO3k23DMbi4n1ES1F6ccYzhI2z6APih18F0A3cVDczsvfnRu5rSQ9tM6h26WYTzOocPh2ZvWmZD/m3i8tvNH/45Ery6Ctn6cxhPoOYK28xj72/gV/XSOSFR7AfQyxylHBK1Xvt4sTsV16WYFzZyDdtO4hBPDoDZNOaYA28VZhw7niPGqC7PqyX3070j06nrX6sGNbppbMwzrKWA4SLfzMISrzTxwUkh7Spsw3zBGq3iZrpy+nq6Gi4/+DYmIiIiIiIiISFCLG0ZeJ0ZS0k6r/vDRvwVRodrpo38DooLZ3vmJKQftKnPz9s31WdomOrZxysF7h2hHtbP3V+d9XXyxYzf49KKxgNmMH/0rEr26+ApHPTxNlbtliKDHXnfOqmZcURPtpPT+iviy9DZcHoxqxwg6vRnuMMQ34PcDc2jaWX0fjtSLWuZoNxJPzNOn5uJ1j3/3Tervf0NUpt8mUb/9h6KcRxdO/dTiujnhjY7bFfTNsfnbf/wu9u3vivZPZYfb36i//26LUb/7/R5G/eckyn9fFLRpXZwm9/M6WY6gDyvom+WP3/2m6M893Z+2GHRPo5pNTlLsJm8p2GbUwqDD0Tn6bX2bNq6nHJcBGtDVg7bxbxTp6tmmKXTa6OLBub19WyegAV09aBPXM+I7kker1v+fYDowh630cRWgAV09aDX5Mb4j+RBODLt5UafFjXlwX6zXARrQ9YNW3d13JN97n/I2oNstBt3TqN0mSMwm/z/qbUYtflL4yrYBTZ8+QJOoAE2iAjSJCtAkKkCTqABNogI0iQrQJCpAk6gATaICNIkK0CQqQJOoAE2iAjSJCtAkKkCTqABNogI0iQrQJCpAk6gATaICNIkK0CQqQJOoAE2iAjSJCtAkKkCTqABNogI0iQrQ9G794Y/P9C9lfgag6d3601+f6V/L/AxA07sFaBIVoElUgCZRAZpEBWgSFaBJVIAmUQGaRAVoEhWgSVSAJlEBmkQFaBIVoElUgCZRAZpEBWgSFaBJVIAmUQkDvbhh1KcNQH/CZIHWTqv+cNoC9CdMFuh2utwC9CdMFmjbOz8x5fjM7QR00339NrN+R3Rs45TDnm4A9CdsF6Ab571r1uvWh8aLGzsfPw3eh+mGDZjN+UZAf8J2AXqelJ7cer2fjDHN+aZuGSLosdeds6oZV9RrgP6E7QG08ToeiFfFY7t+LV4cjGrHCDrdeBiUmoZ+YA79mdsDaB1nycav02jfWhu29byoZdbphvVj/dSYp3/37Q821r3hR9Je2xZ0m0SVWOXohuOKnHd28m18CqWbE97ouF1B6+t/8+M3JqYzfxTtuW1BN0nU46C19ceZcWeDz8McrvXz+qUI+rCCvjkWM+X4hO1hyqGaYWwut7U38ah8nE1fTTkuA/QnbBeg3dMTgOlkMJ4D6tmmKXRy3MWDc+tu/hmgP2F7AL34NHMJZNvAt4lrGUqNVvV9vDUdmMNW+rgK0J+wPYBOz6X44HaMy8x+dK5RB6dVNy/qtLgxD264PfsD9CdsD6Cv64y581V956uA/oTtD/TrA/QnDNAkKkCTqABNogI0iQrQJCpAk6gATaICNIkK0CQqQJOoAE2iAjSJCtAkKkCTqABNogI0iQrQJCpAk6gATR/Un396pj+/dVRA0wf1LL0/bTIqoGm7AJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoA/YoWN4z6tAHoqgP0y2mnVX84bQG66gD9cu10uQXoqgP0y9ne+Ykpxz4C9LGmu94261ejYxunHPZ0A6CrDtCpxnnvmosvdD5+GrwP0w0bMJvxdAugqw7QqXlSenLnzW4ZIuix152zqhlX1GuArjpAx4zX8aDcmDZsHIxqxwg6fCFsDUpNQz8wh95HgI7pOGM2vtPzopZZp431Y/3UmKfv/fnnMda87SfRxu0RtE2iSq9ydMMUn0Lp5oQ3Om5X0Pr6+zhCV90eQa+VBa2tT7Pkfl4nyxH0YQV9s/wB6KoDdKoZjnOI1rfp8nrKcRmgqw7QKXd8MlDPNk2hk+MuHpxbd/OtgK46QMcWb2JKjVb1ffxKOjCHrfRxFaCrDtAx61Pq4LTq5kWdFjfmwQ0354SArjtAP5c25ouvAbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuurkg26asnsM0FUnHXQ7+9G4kqYBXXXCQTe+NaO2sy63xwBddcJB20mZUanBPPSjFjeM57sEoKtOOmhbALR2WvWH0xagq044aDM3AXT72JSjnS63AF11wkGrg3ezm9uHfpLtnZ+Ycuwj6aBV19rlq8fnZ2Yi601N/Kc2Tjns6QZAV51w0N3qsu3u3+pP16wPjbc3dYP3U5qIp5n4GqCrTjTozhwmE1rcvUNxtwxn0H38vub2prHXnbOqGVfUa4CuOtGg23FwY6zX9289gx7XWbaJFwdzvKnzgfhhUGoa+oE59D4SDTrMgZ89HTRn0L61NhzF9byoJS2JxJvSzelTc3GEB3TVCQd97Ctz6AvQzk6+jU+hdLM53dSuoG+O7j//nA76hV8hQoXaI2ibRL3ytRx9/N75/nLGGXRng9rDHK71sz3fdFhB39wZOEJX3R5Br73yiRU7jHaYvnKrv9zS3sSjcnu+6WnKcRmgq044aGvVMqXnru91xppOBuM5oJ6PL2Qy6aQwHJxbd/OPAF118kHHRbfxuSlH24YrTVzLCN9oVd+fbwpb6eMqQFedcNCt61Q4zrrnQI9xmdmPzjXqEA7l3bycbmrmwQ23B3dAV51w0FGrnd3w4vd15p55feergK464aDT5HhpC76+H9B1Jxz04eVjc26ArjrhoNVk44s5HnvDynWArjrhoI1fK7jHAF11wkFvEKCrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAJ0boKsO0LkBuuoAnRugqw7QuQG66gCdG6CrDtC5AbrqAP2KFjeM+rQB6KoD9Mtpp1V/OG0BuuoA/XLtdLkF6KoD9MvZ3vmJKcc+AvQpc2+ziY5tnHLY0w2ArjpAH+v8l5vd4H2YbtiA2YynWwBddYBOdcvgv9wce905q5pxRb0G6KoDdKodk2DThk8Hc9zsfBO2BqWmoR+YQ+8jQB8zEbSeF7XM+riZvpQ+NRcT7B+/MTH9th9D5/7tl2f697eOukfQTRK1BWi1uG42p812BX1D99sfbKwr+tM/Y398DslPbx11j6DbJGoT0Kqf7XnzsIK+ocuUo1CAvm4b0K1vz5tPU47LAF0oQF+3CWg92zSFTptdPDi37uYbAV0oQF+3CejRqr4/b4at9HEVoAsF6Ou2AH1wWnXzctps5sENt8sZgC4UoK97j9dyaGO++BqgCwXo63iB/84D9HWA3nmAvg7QOw/Q1wF65wH6OkDvPEBfB+idB+jrAP1uPfu6uF/eOiqgrwP0u/Ufz/05f33rqIC+DtDv1rOg30wP0NcB+t0CNKBFBWhAiwrQgBYVoAEtKkADWlSABrSoAA1oUQEa0KICNKBFBWhAiwrQgBYVoAEtKkADWlSABrSoAA1oUQEa0KICNKBFBWhAiwrQgBYVoAEtKkADWlSABrSoAA1oUQEa0KICNKBFBWhAiwrQgBYVoAEtKkADWlSABrSoAA1oUQEa0KICNKBFBWhAiwrQgBYVoAEtKkADWlSABrSoAA1oUQEa0KICNKBFBWhAiwrQgBYVoAEtKkADWlSABrSoAA1oUQF6b6Ab/brLGKABXSto60Oj6gbvp7D50uUaoAFdK+h+MsY0aux156x68XIN0ICuFfTYxs+db5Q6DC9eHgM0oGsF7VtrjTI+XA2fXro8BmhAVwva2cm37QpWv3R5/Eff/mBjXZn/il0E6G1Bt0nU46A7G5Ae5sMKtnvp8vivfvzGxPRbf+oOA/S2oJskqtCynfbfM+V4IUDvY8ph4jlh5/8zHnxbF649f3kM0ICuFXRcvpgGNcaluVd8rAEa0JWCVtaPzjWqmQc3hDnxS5drgAZ0raBVZ0y80OvFi5cpQAO6WtBvCdCABvS+AzSgRQVoQIsK0IAWFaABLSpAA1pUgAa0qAANaFEBGtCiAjSgRQVoQIsK0IAWFaABLSpAA1pUgAa0qAANaFEBGtCiAjSgRQVoQIsK0IAWFaABLSpAA1pUgAa0qAANaFEBGtCiAjSgRQVoQIsK0IAWFaABLSpAA1pUgAa0qAANaFEBGtCiAjSgRQVoQIsK0ID+oH759aev9+svbxwV0ID+oH55bsf/FdCbjAro7QI0oHMDNKABfaf/+ssz/fcbBwU0oHMrBfrX5/bRX944KKABnVsp0D8BGtCXAfpOgAZ0boAGNKDvBGhAXwXoOwEa0LkBGtCAvhOgAX0VoO8EaEDnBmhAA/pOgAb0VYC+E6ABnRugAQ3oOwEa0FcB+k6ABnRugAb07kE3+uk6oAG9c9Dd4P103gI0oHcOeux15+xpC9CA3jfozjdKHYbTJqABvW/Qxp8+pQAN6H2DblfQp/NCQAN636APK+juuPmzJ9qiD5py/M/3RFv0v+8EuosH59a9008j2rrRrh9EImrmwQ368XGI6kgbs+n43eNDvNOguxqVPqjDuJdBdzUqfVR6HqfHR3mPQXc1quqYJn5MZp5LPuZ2Wwy6za+65e+qRz83xUc9OM/ywFfS7RI/T53uhoI7KT0FpKf/KzqoaobDFqPGl8kUH3Vtst1UfCozDcsyb3tCtde0nb0PO1y75fJ5yAKNx1GLDrr4WZcfVfVb/K6hdp7nOJVpi466PiXheEriTosbTDhAhR2eJnp9X27oxi/rqCUH7fwwlR81DLvB76p0MyyH9HTYXHYarZ3Ww+QORQcVUT+nnXJ+uqabTYH53rgekCZXctD1Hqfmg29KjjqO6ZHbOl3+d53io146zyw+lenCQb+dOd28quuXQzoLStONNevd4zvpqEPPttygelxPrcb2NCMtMGrYA+GULZLWruDvqqxNJyVRXTj4m3i6WYbeKnhZVG/jbvbLowMKKkye+065cPwwzvshfSXsrhJnGifJh/hnjKM+OmiAp0YfJ0bBSmJSYtS0B8Jkw/pAuo137RKjxuIjXjxK2PgYZeMdsMjUfF0wacYwSwxTI+M0TwQ9tTgX/3TGt33g10WC6VjyWN00x5EOx3MrV2TU9a7X+UMc2ozpAaDA73rcAwFdNwXSY19kD6TSAoQ+Hvf1XOo4ui6YmDhLbGbvyi8G7rdunKf1j9f7KT6MtXHr4YfFZp7MEv9+w3pulQ5Oj456hBfkNeGEUPv1fvLwbOO0B9LDSSDtyuyBPj7Mne8XS3yMKgPvZsFEw/myKez2Mc1FO388g5sen+XpdIoZxzHrn7TAawOf7nphdD2MOhyYlgIz0qc9sC5BBNIPLxo0wyE8msxWPx3oSz31uNmCiaCOlG25PfT07gPVDzqcFhWFF6fkehrSMytF98Cwont80HWR/ODm40JMmJI3D09zN18w2XvHaa46LUac1yIKDB2AGDv48LCoh3n0Y1l46W85Pf6U7+0eMIVmzsdF8rhS4qdOlZmSb7ZgIqXzNPdMufXF9tDkvR/sYuMRqj08Osu7hRfkNQVmpF/ugQJPTV8tkhtnRj81RR5JNlowEdPTNFdFymnnFNxDjUlDF3lY/BJeeoZ6gz3QPboHbhbJ48tQA+kiz3dvtGAipotprjovRhTPFfhb3r3rFbjvbbIHrhfJ7Xq6/eCY2y2YSOpimqu2mI/Z+EDbl3ij2FZ3veJ7ID7rc71Ibou8um6rBRNhXU5ztxh+nqa5yMngZne9sntgfdbnepG8wJx8fZls4QUTmZWc5t6p7W2hNYPN7nol98DpWZ+rRXL7OL20AqjKLpjIrsQ0d+M2vusV2ANPz/qUXiQ/rQCWXDCRWrlp7vtU/q5XbA9cPOtTapH8ZgWw5IKJ1MpNc7duq7te0T1wetanzCL57QpgmQUT4ZWb5m7cZne9onvg9N6AAovkX75MtsyCCVXSLu56p2d9Hn965s7LZMu/v5bohdoy76y9+zLZAgsmRJmVeNZnq5fJEmVX4lmfLV8mS/QhFX6ZLNEHV/RlskQfXcmXyRJ9fAVfJktUQVu9Qp3oQ+Idg0REREREgvp/PIo/uDos8H4AAAAldEVYdGRhdGU6Y3JlYXRlADIwMjItMDMtMjdUMDg6NDg6NTgrMDc6MDC3itnIAAAAJXRFWHRkYXRlOm1vZGlmeQAyMDIyLTAzLTI3VDA4OjQ4OjU4KzA3OjAwxtdhdAAAACF0RVh0cHM6SGlSZXNCb3VuZGluZ0JveAA1MDR4NzIwKzUwKzUw1uIiwwAAABN0RVh0cHM6TGV2ZWwAQWRvYmUtMi4wCjmTdA0AAAAASUVORK5CYII=" />


 #table7#
 {dataset=>"len100ws10"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim      |    163000 |  6.14     |                 0.00% |              1458.96% | 1.4e-09 |      27 |
 | String::Util::trim             |    196000 |  5.11     |                20.08% |              1198.31% | 1.6e-09 |      21 |
 | String::Trim::NonRegex::trim   |    302000 |  3.31     |                85.47% |               740.53% | 1.7e-09 |      20 |
 | String::Trim::More::trim       |   1035120 |  0.966073 |               535.12% |               145.46% |   0     |      32 |
 | Text::Minify::XS::minify       |   2110000 |  0.474    |              1194.32% |                20.45% | 2.9e-10 |      26 |
 | Text::Minify::XS::minify_ascii |   2540780 |  0.39358  |              1458.96% |                 0.00% |   0     |      24 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate  STR:t   SU:t  STN:t  STM:t  TMX:m  TMX:m_a 
  STR:t     163000/s     --   -16%   -46%   -84%   -92%     -93% 
  SU:t      196000/s    20%     --   -35%   -81%   -90%     -92% 
  STN:t     302000/s    85%    54%     --   -70%   -85%     -88% 
  STM:t    1035120/s   535%   428%   242%     --   -50%     -59% 
  TMX:m    2110000/s  1195%   978%   598%   103%     --     -16% 
  TMX:m_a  2540780/s  1460%  1198%   740%   145%    20%       -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJmlDQ1BpY2MAAEiJlZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEWQRQEXXJUia0UUC4uCAhZ0gywCyrpxFVFBWXDfGZ33HT+8/5l7z2/+c+bec8/5cAEgiINlwct7YlK6wNvJjhkYFMwE3yiMn5bC8fR0A9/VuxEArcR7ut/P+a4IEZFp/OW4uLxy+SmCdACg7GXWzEpPWeGjy0wPj//CZ1dYsFzgMt9Y4eh/eexLzr8s+pLj681dfhUKABwp+hsO/4b/c++KVDiC9NioyGymT3JUelaYIJKZttIJHpfL9BQkR8UmRH5T8P+V/B2lR2anr0RucsomQWx0TDrzfw41MjA0BF9n8cbrS48hRv9/z2dFX73kegDYcwAg+7564ZUAdO4CQPrRV09tua+UfAA67vAzBJn/eqiVDQ0IgALoQAYoAlWgCXSBETADlsAWOAAX4AF8QRDYAPggBiQCAcgCuWAHKABFYB84CKpALWgATaAVnAad4Dy4Aq6D2+AuGAaPgRBMgpdABN6BBQiCsBAZokEykBKkDulARhAbsoYcIDfIGwqCQqFoKAnKgHKhnVARVApVQXVQE/QLdA66At2EBqGH0Dg0A/0NfYQRmATTYQVYA9aH2TAHdoV94fVwNJwK58D58F64Aq6HT8Id8BX4NjwMC+GX8BwCECLCQJQRXYSNcBEPJBiJQgTIVqQQKUfqkVakG+lD7iFCZBb5gMKgaCgmShdliXJG+aH4qFTUVlQxqgp1AtWB6kXdQ42jRKjPaDJaHq2DtkDz0IHoaHQWugBdjm5Et6OvoYfRk+h3GAyGgWFhzDDOmCBMHGYzphhzGNOGuYwZxExg5rBYrAxWB2uF9cCGYdOxBdhK7EnsJewQdhL7HkfEKeGMcI64YFwSLg9XjmvGXcQN4aZwC3hxvDreAu+Bj8BvwpfgG/Dd+Dv4SfwCQYLAIlgRfAlxhB2ECkIr4RphjPCGSCSqEM2JXsRY4nZiBfEU8QZxnPiBRCVpk7ikEFIGaS/pOOky6SHpDZlM1iDbkoPJ6eS95CbyVfJT8nsxmpieGE8sQmybWLVYh9iQ2CsKnqJO4VA2UHIo5ZQzlDuUWXG8uIY4VzxMfKt4tfg58VHxOQmahKGEh0SiRLFEs8RNiWkqlqpBdaBGUPOpx6hXqRM0hKZK49L4tJ20Bto12iQdQ2fRefQ4ehH9Z/oAXSRJlTSW9JfMlqyWvCApZCAMDQaPkcAoYZxmjDA+SilIcaQipfZItUoNSc1Ly0nbSkdKF0q3SQ9Lf5RhyjjIxMvsl+mUeSKLktWW9ZLNkj0ie012Vo4uZynHlyuUOy33SB6W15b3lt8sf0y+X35OQVHBSSFFoVLhqsKsIkPRVjFOsUzxouKMEk3JWilWqUzpktILpiSTw0xgVjB7mSJleWVn5QzlOuUB5QUVloqfSp5Km8oTVYIqWzVKtUy1R1WkpqTmrpar1qL2SB2vzlaPUT+k3qc+r8HSCNDYrdGpMc2SZvFYOawW1pgmWdNGM1WzXvO+FkaLrRWvdVjrrjasbaIdo12tfUcH1jHVidU5rDO4Cr3KfFXSqvpVo7okXY5upm6L7rgeQ89NL0+vU++Vvpp+sP5+/T79zwYmBgkGDQaPDamGLoZ5ht2GfxtpG/GNqo3uryavdly9bXXX6tfGOsaRxkeMH5jQTNxNdpv0mHwyNTMVmLaazpipmYWa1ZiNsulsT3Yx+4Y52tzOfJv5efMPFqYW6RanLf6y1LWMt2y2nF7DWhO5pmHNhJWKVZhVnZXQmmkdan3UWmijbBNmU2/zzFbVNsK20XaKo8WJ45zkvLIzsBPYtdvNcy24W7iX7RF7J/tC+wEHqoOfQ5XDU0cVx2jHFkeRk4nTZqfLzmhnV+f9zqM8BR6f18QTuZi5bHHpdSW5+rhWuT5z03YTuHW7w+4u7gfcx9aqr01a2+kBPHgeBzyeeLI8Uz1/9cJ4eXpVez33NvTO9e7zofls9Gn2eedr51vi+9hP0y/Dr8ef4h/i3+Q/H2AfUBogDNQP3BJ4O0g2KDaoKxgb7B/cGDy3zmHdwXWTISYhBSEj61nrs9ff3CC7IWHDhY2UjWEbz4SiQwNCm0MXwzzC6sPmwnnhNeEiPpd/iP8ywjaiLGIm0iqyNHIqyiqqNGo62ir6QPRMjE1MecxsLDe2KvZ1nHNcbdx8vEf88filhICEtkRcYmjiuSRqUnxSb7JicnbyYIpOSkGKMNUi9WCqSOAqaEyD0tandaXTlz/F/gzNjF0Z45nWmdWZ77P8s85kS2QnZfdv0t60Z9NUjmPOT5tRm/mbe3KVc3fkjm/hbKnbCm0N39qzTXVb/rbJ7U7bT+wg7Ijf8VueQV5p3tudATu78xXyt+dP7HLa1VIgViAoGN1tubv2B9QPsT8M7Fm9p3LP58KIwltFBkXlRYvF/OJbPxr+WPHj0t6ovQMlpiVH9mH2Je0b2W+z/0SpRGlO6cQB9wMdZcyywrK3BzcevFluXF57iHAo45Cwwq2iq1Ktcl/lYlVM1XC1XXVbjXzNnpr5wxGHh47YHmmtVagtqv14NPbogzqnuo56jfryY5hjmceeN/g39P3E/qmpUbaxqPHT8aTjwhPeJ3qbzJqamuWbS1rgloyWmZMhJ+/+bP9zV6tua10bo63oFDiVcerFL6G/jJx2Pd1zhn2m9az62Zp2WnthB9SxqUPUGdMp7ArqGjzncq6n27K7/Ve9X4+fVz5ffUHyQslFwsX8i0uXci7NXU65PHsl+spEz8aex1cDr97v9eoduOZ67cZ1x+tX+zh9l25Y3Th/0+LmuVvsW523TW939Jv0t/9m8lv7gOlAxx2zO113ze92D64ZvDhkM3Tlnv296/d5928Prx0eHPEbeTAaMip8EPFg+mHCw9ePMh8tPN4+hh4rfCL+pPyp/NP637V+bxOaCi+M24/3P/N59niCP/Hyj7Q/Fifzn5Ofl08pTTVNG02fn3Gcufti3YvJlykvF2YL/pT4s+aV5quzf9n+1S8KFE2+Frxe+rv4jcyb42+N3/bMec49fZf4bmG+8L3M+xMf2B/6PgZ8nFrIWsQuVnzS+tT92fXz2FLi0tI/QiyQvpNzTVQAAAAgY0hSTQAAeiYAAICEAAD6AAAAgOgAAHUwAADqYAAAOpgAABdwnLpRPAAAAJ9QTFRF////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAlADUlADUlADUlADUlADUAAAAAAAAAAAAlQDVlQDVAAAAlADUlADUlADUlQDVlQDVlQDWmADaAAAAAAAAAAAAAAAAlADU////cjmOKgAAADJ0Uk5TABFEZiK7d+4zzIhV3ZmqqdXKx9I/7/z27Pnx9HUzRCLfx+Tt8KfsTmbWiPVpdSBbIDA+iczSAAAAAWJLR0QAiAUdSAAAAAlwSFlzAAAASAAAAEgARslrPgAAAAd0SU1FB+YDGw8wOt5nUA0AABT1SURBVHja7dyLduM2YoBhAryDIJvuNs12m6ZpMk0v6faCff93KwBKsqDR2IYF2gD8f+fYMi0PzMi/aBCU0zQAAAAAAAAAAAAAAAAAAAAA3pGQpw+k+OhdAd6o7S4fSnP6wFzKNp6MHhX4IP1TrV8HLaQ1jBywUYppHuwhulVKuqA7f+uDlkpN+5fMy0fvJPBarV5l02ulxtYGPa6rUT7oYVSrad1XDNtH7yPwem7K0duDtNps0IPt13Q26M7Yw3M72vvF2D38PYB3s8+hp2XuT3Noe3g2stVu9uyqVutH7yEQwQWtTL/256BHF7Qae2eyB2iWOFASG/TiZhXKBS3sFMMfoRfd7OvSrf7oHQRi9IuPVvgph51eKO1mHcKeI/oPN2YcKMqmWzHrXq9jK+dZ63Hy0+h2nGf34dh+9A4CMYS08w0pReNu/Qfnz3MBHAAAAAAAAMkteu5Zd0IthBbNNnz0XgCJtFyyRU3Ups3KlANFkTevPd9f6Ti5jpWbcqiP3kHg9drRmOC8r3Ov5+1m4180ZmOW/UfvIvBq7hWOYn6aKnfL7ILuN9Fp1Uz9HjVQCP/3Fe5l6e4FjoNs2t4F7f8YbpibZp23mTk0CrNt9ki9NIv/X0e4xH3m/t109bdDf/Od97d/ANL4oy/qj3+XNOde21O/ZtHd/odvruN2D/rm2Pz93//gfP+npP4h7XDljfrnH44Y9Yc/lzDqP/qizI9Jg5atdtPkbdwnyy7oYQ/6ZvnjT39I+n3Pz6cjBi1pVHnISYo65M9yjxk1cdD26Oz6bff/I8rNlOMaQRN09kEr9zNy6YpR7f/3NelPCrs7f6pM0ASdfdD+f4Oy2nR75c4Nm9OB2W75twBBE3T2QTer6d3fKA/2xLAb3f9QcF/cGGf91XodQRN0/kE3nby3k+LOZ48J+pi/8C9o1O6QSOQh/zO9Y0ZNflL4SscEjU+PoFEVgkZVCBpVIWhUhaBRFYJGVQgaVSFoVIWgURWCRlUIGlUhaFSFoFEVgkZVCBpVIWhUhaBRFYJGVQgaVSFoVIWgURWCRlUIGlUhaFSFoFEVgkZVCBpVIWhUhaBRFYJGVQgaVSFoVIWgURWCRlUIGlUhaFSFoFEVgkZVCBpVIWhUhaBRFYJGVQgaVSFoVIWgURWCRlUIGlUhaFSFoFEVgkZVCBpVIWhUhaBRlXcMetFzL84bBI1DvF/QQotmG85bBI1DvF/Q7Xq9RdCf0D/99Ix/TvM93i9otWmzMuX4zH7+6zP+Jc33SBD01H37Prl/hetYuSmHOt9B0J9QEUFP2hg97R8rY/VXd3bGvZuNsdMNZWOWlzsJ+hMqIuhxbcSq94+3VUo5Xe7qltkF3W+i06qZ+j3qHUF/QiUELY1wB+K94r7dP+duBtm0vQva3znMTbPO28wc+jMrIWjhZsnS7NNo0yplt8W4NMso/B372/5ukk//7vtflNO94VuiVMcG3fqiUqxydPNpRc5otZrWXULpRh+v67jdgxbhv/n1O+mIyG+Fkh0b9OSLejxoocxpZtwp2+cw2o+2cf+UC3rYg745FjPl+IRKmHI009xP19vCSHdUPs2mgynHNYL+hIoIWj9dAPQng+4cUIzKT6F9x507OLf65p8R9CdUQtCL8TMXm2xr853cWkbT9KrZNnevPzDbLf8WIOhPqISg/bUUY7vt3TKz6bWemkGLphuX5ry4Mc56vj37I+hPqISgQ52Udz4r7nyWoD+h8oJ+PYL+hAgaVSFoVIWgURWCRlUIGlUhaFSFoFEVgkZVCBpVIWhUhaBRFYJGVQgaVSFoVIWgURWCRlUIGlUhaFSFoFEVgkZVCBpVIWhUhaBRFYJGVQgaVSFoVIWgURWCRlUIGlUhaFSFoFEVgkZVCBpVIWhUhaBRFYJGVQgaVSFoVIWgURWCRlUIGlUhaFSFoFEVgkZVCBpVIWhUhaDxQZ5N7+dDRiVoHIegYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYI+hUWPffivEHQWSPolwktmm04bxF01gj6Ze16vUXQWSPol6lNm5UpRxkI+mTqwm25f9Z1rNyUQ53vIOisEbQ3aWP0dPWJzrh3szF2uqFszLI/30PQWSNob1wbserLZrfMLuh+E51WzdTvUe8IOmsE7Ugj3EF5kq3dGGTT9i5o+wm7NTfNOm8zc+gyELQj3IxZmk6MS7OMwm/sb/u7ST597ZcvvTO97TvhYCUGrXxRqVc5unl1l1C60cfrOm73oEX4dRyhs1Zi0Lu0QQtl/Cx5G/fJsgt62IO+Wf4g6KwRtDfNpzlEa1p/G045rhF01gja06eLgWJUfgrtO+7cwbnVN19K0FkjaGcx0mmaXjXb5j7jD8x2y78FCDprBO0o4zWDFk03Ls15cWOc9XxzTkjQeSPo5wgpv/ocQWeNoGMRdNYIOhZBZ42gYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYIOhZBZ42gYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYIOhZBZ42gYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYIOhZBZ42gYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYIOhZBZ42gYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYIOhZBZ42gYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYIOhZBZ42gYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYIOhZBZ42gYxF01gg6FkFnjaBjEXTWCDoWQWeNoGMRdNYIOhZBZ63+oKcp7SNG0FmrPeh2NL3UKZsm6KxVHvRkWtkLNYp0jxhBZ63yoNXayL5pZvnQt1r03F+eEgSdtdqDVgmCFlo023DeIuisVR60HCcbdPvYlKNdr7cIOmuVB90MRo96bB/6TmrTZmXKUYbag266Vi3fPD4/MxPZ75rcP1VuyqHOdxB01ioPutu7bLv795rzR8pY/e1d3WzM6ififia+I+isVR10J4dVWou+dyjulvkS9Oa+brq9q99Ep1Uz9XvUO4LOWtVBt/2se2cT9++9BN3vs2zpbgZ5uqszNvFhbpp13mbm0GWoOmg7B372dFBegjatUvYoLsalWfySiLvL3+3fTVdHeILOWuVBn3xjDn0VtFarad0llG6U57vaPeibo/uXL/6gn/gVIkikxKCVL+qVr+XY3NeO95czLkF3ylY7jPajbVSXu4Y96JsnA0forJUY9O6VF1bU3Kt5/ca95npLGOmOyu3lrqcpxzWCzlrlQSvVLKu/dn3PJVZ/MujOAcV4eiGT9CeF9uDc6pt/RNBZqz9ot+jWPzflaFv7weTWMuwXqmbbLnfZLf8WIOisVR50q7vGHmf1c0H3bpnZ9FpPzWAP5d24nO+axlnPtwd3gs5a5UG7WtWo5xe/rpP3mhd3PkvQWas8aD85XtqEr+8n6LxVHvTw8rE5FkFnrfKgm1W5F3M89gcrIYLOWuVBS7NL+IgRdNYqD/oABJ01go5F0Fkj6FgEnTWCjkXQWSPoWASdNYKORdBZI+hYBJ01go5F0Fkj6FgEnTWCjkXQWSPoWASdNYKORdBZI+hYBJ01go5F0Fkj6FgEnTWCjkXQWSPoWASdNYKORdBZI+hYBJ01go5F0Fkj6FgEnTWCjkXQWSPoWASdNYKORdBZI+hYBJ01go5F0Fkj6FgEnTWCjkXQWSPoWASdNYKORdBZI+hYBJ01go5F0Fkj6FgEnTWCjkXQWSPoWASdNYKORdBZI+hYBJ01go5F0Fkj6FgEnTWCjkXQWSPoWASdNYKORdBZI+hYBJ01go5F0Fkj6FgEnTWCjkXQWSPoWASdNYKORdBZI+hYBJ01go5F0Fkj6FdY9NyL8wZBZ42gXya0aLbhvEXQWSPol7Xr9RZBZ42gX6Y2bVamHGUg6DN5b3NyHSs35VDnOwg6kX99LpLf3joqQZ905uvNbjbGTjeUjVn253sIOhGCDiUNultm8/Vmv4lOq2bq96h3BJ0IQYeSBt32vmDZ2neDPG12ZrJbc9Os8zYzh06NoEOJpxzSBS3GpVlGcdr0n/LvpqsJ9q/fSUe87dvggqDPJl/UEUE3i+5Ged5s96Bv0v3+F+V0Sb/7Z0TQZ60v6pCgm21Ul81hD/omXaYciRB06JigW9NeNp+mHNcIOhGCDh0StBiVn0L7zc4dnFt984UEnQhBhw4JulfNtl027ZZ/CxB0IgQdOiLoQYumG5fz5jTOer5dziDoRAg69B6v5RBSfvU5gk6EoEO8wL9wBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB0i6MIRdIigC0fQIYIuHEGHCLpwBB1KGfQkXnfrEHQiBB1KELQyVt90szGr3XzpdkfQiRB0KEHQ2yqlnJp+E51WzYu3O4JOhKBDCYLuW/e+M1PTDPOLtycEnQhBhxIEbVqlZCON/dC+e+n2hKATIehQiqC1Wk3b7sGKl25P/+j7X5TTpfmv+MQI+qz1RT0edKdspMM47MF2L92e/tWv30lHvPW74oSgzyZfVKJlO2F+ZMrxIQg69HjQ0p0Tdubf3MG31faj529PCDoRgg4lCNotX6xz07uluVe87Qg6EYIOJbmw0ms9NdM469nOiV+63RF0IgQdSjGH7qR0N2K/efHWI+hECDrEi5MKR9Ahgi4cQYcIunAEHSLowhF0iKALR9Ahgi4cQYcIunAEHSLowhF0iKALR9Ahgi4cQYcIunAEHSLowhF0iKALR9Ahgi4cQYcIunAEHSLowhF0iKALR9Ahgi4cQYcIunAEHSLowhF0iKALR9Ahgn43//7zM/7jraMSdIig381/HpIeQYcI+t0QNEFXhaAJuioETdBVIWiCrgpBE3RVCJqgq0LQBF0VgiboqhA0QVeFoAm6KgRN0FUhaIKuCkET9Af56fffvu33n944KkET9Af56bkH/q8EfcioBH0cgiboWARN0AR9x3/95Rn//cZBCZqgY6UK+vfnHqO/vHFQgiboWKmC/o2gCfoaQd9B0AQdi6AJmqDvIGiCDhD0HQRN0LEImqAJ+g6CJugAQd9B0AQdi6AJmqDvIGiCDhD0HQRN0LEImqAJ+g6CJugAQd9B0AQdi6AJuvigJ/H0MUETdOFBd7Mx62WLoAm68KD7TXRanbcImqDLDrozU9MM83mToAm67KClOb/zCJqgyw663YM+nxcSNEGXHfSwB92dNr8Y4AgfNOX4nx+BI/zvOwXduYNzq9/puwFH69X+BlRhGmc9i8fHAfIgpDx0/O7xId5p0KJGxQcZ+lIGLWpUfBQx9uvjo7zHoEWN2nRMEz+GHMeUv3O7IwY9aNRDHoCd6M04JR910IblgW8Q7eLer53o5oQPkr8EJNb/SzpoM7kXtKQfdR4OGHW3qm5NPpVZ52UZjz2hKpVQozH2ARd6ub4OmUB/GjXpoM12xKiLGcUB+9o07TiObirTJh11vyShuSRxx6JnaQ979gH3E71tSzf0ZJZ91JSD2p/lAaN2Zl4P2FcxzcvgL4eNaafRQgsxr3pIOmgVttE/KJfLNd0oE8z3+v2AtOqUg9ojvv8dq7RIOeo+2DiYKf2oq/ut588zk09lOnvQb0dONwPdtgz+LMhPN3bK6McfpFNzYlTJBrX7ak+uXNJCpxvVnrHtJ2x9e57npnkAlD8pcdXZo790p5tp0tsLXpZmU+5hNsujA1bETp63rtH2+CG1MbP/jH24UpxpnEse3I/RjfrooH5f7WRDGZt0656EKUa1T5KmN266ZQv08aUY1XG/8dxRQrnfUco9VZJMzfcFk6m3s0Q7NZJacCHoyaK1+9FJ0242v84l6I8lj+nW0Y00nM6tdJpRT/tq8+hWm3S/pRh1f0J3ZnA7LHv/ayXBvnp+AUKcfpuIMdVxdF8wkW6WOI1Gp18MLFfXj+v+w9vM6n6NtW7r4V+L07jKxf385v3cyh+cHp5tnPfVH/ht0jrFvp6eJLbjyZ4QCrM/+x5/BDb3a+7yvFjc76g04d0smAhyvrbah733s8bOnM7g1sdnecKfYrpx5P4jTfLawKd93RcLbNIPn94/PaHtPou5F/Zwtzw8z53mwR73RyWeDvSpLj0etmBSkVPKKt0j9PTXB802C3talGzo077Oex6Pj/r0JHETfbHO/srKo6Puy9mDHk9LJnZKPj08zT18waR0p2luc16MuKxFJBjaZifVbOyvRTGPvekfnm3c7KtMNMe97K3jClmTXEjel7PdSolZuybJ6cNxCya1uExzLym3JtkjtBpjZrUod4Rqh4dneV/va4KLyLdPEvssmR6f5wbL2VLL3qxTil8lRy2YVONpmtu4lP2Dk/ARmqQfOs2vxTv72j28r18/SfzV9Af3NFzOdi9DtUknud590IJJNa6muc1lMSI5neRnecS+3n1CP/6MDpez1X66/eCYxy2Y1ORqmtscMR9T7hftluYPxY7Y1wOeJO76TLicrZK8uu6oBZPKXE9zjxh+XNfx4ZPB4/Y1+ZNkvz4TLmcnmOnvL2hNvGBSp6TT3K+1m0q3EnHAviZ+kpyvzwTL2erx9PwKYJN2waRuaaa5Be5ryifJ0/WZlMvZfuTTCmDKBZNapZzmlruvSZ4kV9dn0i1nByuAKRdMapV0mlvgviZ+kpyvzyRZzv5qBTDNgknlkk5zy9vX1E+S818cPL6cfecFrWkWTFC1xE+S8/WZh5ez772gNf3f1wIvaNP8Ze3dF7QmWDABIqW4PnPMC1qBN0hxEfOYF7QCHyj9C1qBj5TyBa3Ah0v3glYgB+le0Ark4KhXqAMfgr8YBAAAACry/w3nsdcAIJgpAAAAJXRFWHRkYXRlOmNyZWF0ZQAyMDIyLTAzLTI3VDA4OjQ4OjU4KzA3OjAwt4rZyAAAACV0RVh0ZGF0ZTptb2RpZnkAMjAyMi0wMy0yN1QwODo0ODo1OCswNzowMMbXYXQAAAAhdEVYdHBzOkhpUmVzQm91bmRpbmdCb3gANTA0eDcyMCs1MCs1MNbiIsMAAAATdEVYdHBzOkxldmVsAEFkb2JlLTIuMAo5k3QNAAAAAElFTkSuQmCC" />


 #table8#
 {dataset=>"len100ws100"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::NonRegex::trim   |     30000 |  30       |                 0.00% |              4113.73% | 5.3e-07 |      20 |
 | String::Trim::Regex::trim      |    159440 |   6.27194 |               358.43% |               819.16% |   0     |      20 |
 | String::Util::trim             |    188000 |   5.31    |               441.53% |               678.11% | 1.6e-09 |      22 |
 | String::Trim::More::trim       |    870000 |   1.1     |              2409.75% |                67.89% | 1.7e-09 |      20 |
 | Text::Minify::XS::minify       |   1052000 |   0.9505  |              2924.88% |                39.30% | 1.2e-11 |      20 |
 | Text::Minify::XS::minify_ascii |   1470000 |   0.682   |              4113.73% |                 0.00% |   2e-10 |      21 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate  STN:t  STR:t  SU:t  STM:t  TMX:m  TMX:m_a 
  STN:t      30000/s     --   -79%  -82%   -96%   -96%     -97% 
  STR:t     159440/s   378%     --  -15%   -82%   -84%     -89% 
  SU:t      188000/s   464%    18%    --   -79%   -82%     -87% 
  STM:t     870000/s  2627%   470%  382%     --   -13%     -38% 
  TMX:m    1052000/s  3056%   559%  458%    15%     --     -28% 
  TMX:m_a  1470000/s  4298%   819%  678%    61%    39%       -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJmlDQ1BpY2MAAEiJlZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEWQRQEXXJUia0UUC4uCAhZ0gywCyrpxFVFBWXDfGZ33HT+8/5l7z2/+c+bec8/5cAEgiINlwct7YlK6wNvJjhkYFMwE3yiMn5bC8fR0A9/VuxEArcR7ut/P+a4IEZFp/OW4uLxy+SmCdACg7GXWzEpPWeGjy0wPj//CZ1dYsFzgMt9Y4eh/eexLzr8s+pLj681dfhUKABwp+hsO/4b/c++KVDiC9NioyGymT3JUelaYIJKZttIJHpfL9BQkR8UmRH5T8P+V/B2lR2anr0RucsomQWx0TDrzfw41MjA0BF9n8cbrS48hRv9/z2dFX73kegDYcwAg+7564ZUAdO4CQPrRV09tua+UfAA67vAzBJn/eqiVDQ0IgALoQAYoAlWgCXSBETADlsAWOAAX4AF8QRDYAPggBiQCAcgCuWAHKABFYB84CKpALWgATaAVnAad4Dy4Aq6D2+AuGAaPgRBMgpdABN6BBQiCsBAZokEykBKkDulARhAbsoYcIDfIGwqCQqFoKAnKgHKhnVARVApVQXVQE/QLdA66At2EBqGH0Dg0A/0NfYQRmATTYQVYA9aH2TAHdoV94fVwNJwK58D58F64Aq6HT8Id8BX4NjwMC+GX8BwCECLCQJQRXYSNcBEPJBiJQgTIVqQQKUfqkVakG+lD7iFCZBb5gMKgaCgmShdliXJG+aH4qFTUVlQxqgp1AtWB6kXdQ42jRKjPaDJaHq2DtkDz0IHoaHQWugBdjm5Et6OvoYfRk+h3GAyGgWFhzDDOmCBMHGYzphhzGNOGuYwZxExg5rBYrAxWB2uF9cCGYdOxBdhK7EnsJewQdhL7HkfEKeGMcI64YFwSLg9XjmvGXcQN4aZwC3hxvDreAu+Bj8BvwpfgG/Dd+Dv4SfwCQYLAIlgRfAlxhB2ECkIr4RphjPCGSCSqEM2JXsRY4nZiBfEU8QZxnPiBRCVpk7ikEFIGaS/pOOky6SHpDZlM1iDbkoPJ6eS95CbyVfJT8nsxmpieGE8sQmybWLVYh9iQ2CsKnqJO4VA2UHIo5ZQzlDuUWXG8uIY4VzxMfKt4tfg58VHxOQmahKGEh0SiRLFEs8RNiWkqlqpBdaBGUPOpx6hXqRM0hKZK49L4tJ20Bto12iQdQ2fRefQ4ehH9Z/oAXSRJlTSW9JfMlqyWvCApZCAMDQaPkcAoYZxmjDA+SilIcaQipfZItUoNSc1Ly0nbSkdKF0q3SQ9Lf5RhyjjIxMvsl+mUeSKLktWW9ZLNkj0ie012Vo4uZynHlyuUOy33SB6W15b3lt8sf0y+X35OQVHBSSFFoVLhqsKsIkPRVjFOsUzxouKMEk3JWilWqUzpktILpiSTw0xgVjB7mSJleWVn5QzlOuUB5QUVloqfSp5Km8oTVYIqWzVKtUy1R1WkpqTmrpar1qL2SB2vzlaPUT+k3qc+r8HSCNDYrdGpMc2SZvFYOawW1pgmWdNGM1WzXvO+FkaLrRWvdVjrrjasbaIdo12tfUcH1jHVidU5rDO4Cr3KfFXSqvpVo7okXY5upm6L7rgeQ89NL0+vU++Vvpp+sP5+/T79zwYmBgkGDQaPDamGLoZ5ht2GfxtpG/GNqo3uryavdly9bXXX6tfGOsaRxkeMH5jQTNxNdpv0mHwyNTMVmLaazpipmYWa1ZiNsulsT3Yx+4Y52tzOfJv5efMPFqYW6RanLf6y1LWMt2y2nF7DWhO5pmHNhJWKVZhVnZXQmmkdan3UWmijbBNmU2/zzFbVNsK20XaKo8WJ45zkvLIzsBPYtdvNcy24W7iX7RF7J/tC+wEHqoOfQ5XDU0cVx2jHFkeRk4nTZqfLzmhnV+f9zqM8BR6f18QTuZi5bHHpdSW5+rhWuT5z03YTuHW7w+4u7gfcx9aqr01a2+kBPHgeBzyeeLI8Uz1/9cJ4eXpVez33NvTO9e7zofls9Gn2eedr51vi+9hP0y/Dr8ef4h/i3+Q/H2AfUBogDNQP3BJ4O0g2KDaoKxgb7B/cGDy3zmHdwXWTISYhBSEj61nrs9ff3CC7IWHDhY2UjWEbz4SiQwNCm0MXwzzC6sPmwnnhNeEiPpd/iP8ywjaiLGIm0iqyNHIqyiqqNGo62ir6QPRMjE1MecxsLDe2KvZ1nHNcbdx8vEf88filhICEtkRcYmjiuSRqUnxSb7JicnbyYIpOSkGKMNUi9WCqSOAqaEyD0tandaXTlz/F/gzNjF0Z45nWmdWZ77P8s85kS2QnZfdv0t60Z9NUjmPOT5tRm/mbe3KVc3fkjm/hbKnbCm0N39qzTXVb/rbJ7U7bT+wg7Ijf8VueQV5p3tudATu78xXyt+dP7HLa1VIgViAoGN1tubv2B9QPsT8M7Fm9p3LP58KIwltFBkXlRYvF/OJbPxr+WPHj0t6ovQMlpiVH9mH2Je0b2W+z/0SpRGlO6cQB9wMdZcyywrK3BzcevFluXF57iHAo45Cwwq2iq1Ktcl/lYlVM1XC1XXVbjXzNnpr5wxGHh47YHmmtVagtqv14NPbogzqnuo56jfryY5hjmceeN/g39P3E/qmpUbaxqPHT8aTjwhPeJ3qbzJqamuWbS1rgloyWmZMhJ+/+bP9zV6tua10bo63oFDiVcerFL6G/jJx2Pd1zhn2m9az62Zp2WnthB9SxqUPUGdMp7ArqGjzncq6n27K7/Ve9X4+fVz5ffUHyQslFwsX8i0uXci7NXU65PHsl+spEz8aex1cDr97v9eoduOZ67cZ1x+tX+zh9l25Y3Th/0+LmuVvsW523TW939Jv0t/9m8lv7gOlAxx2zO113ze92D64ZvDhkM3Tlnv296/d5928Prx0eHPEbeTAaMip8EPFg+mHCw9ePMh8tPN4+hh4rfCL+pPyp/NP637V+bxOaCi+M24/3P/N59niCP/Hyj7Q/Fifzn5Ofl08pTTVNG02fn3Gcufti3YvJlykvF2YL/pT4s+aV5quzf9n+1S8KFE2+Frxe+rv4jcyb42+N3/bMec49fZf4bmG+8L3M+xMf2B/6PgZ8nFrIWsQuVnzS+tT92fXz2FLi0tI/QiyQvpNzTVQAAAAgY0hSTQAAeiYAAICEAAD6AAAAgOgAAHUwAADqYAAAOpgAABdwnLpRPAAAAKtQTFRF////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAlADUlADUlADUlADUlADUAAAAlADVlQDVAAAAAAAAlQDVAAAAAAAAlADUlADUlADUlQDWlQDVlgDXlADUAAAAAAAAAAAAAAAAlADU////KusarAAAADZ0Uk5TABFEMyJm3buIzJl37qpVQISp1crH0j/v/Pbs+fH0dTNEIt/H8Lfs7eSnXE5miBF1nzDxWyAw36+4YAAAAAFiS0dEAIgFHUgAAAAJcEhZcwAAAEgAAABIAEbJaz4AAAAHdElNRQfmAxsPMDreZ1ANAAAV3UlEQVR42u3dDX/jtmGAcYAEQfC1W9Y2S9e1aXdN23Rr94p+/282AKRkwedYogXKBPT8f7EpmWfY0T2iQIo6CQEAAAAAAAAAAAAAAAAAAADggWS1Xqjkyxdr9dm/FrBB8xJsZdcL9lS2qLW1rdw6JvBp2nO8bwWtjZBd/9m/I3CruhvcJroxpvJBq7AMQVfG1Msl0372LwncqtF9JVptzNi4oMe+tyZkPIymt40YByEmttDIh59ytG4jbSYXtMt3sMoFrazbPDejqEY9aubQyMcyh67nrl3n0G7zbKtGV479WTdVM3NoZMQHbWzbt6egRx+0GVvvH7T7Qm05codsuKDnUYVdv8q6yYUMW+jZlywr04nlK0Am2tntGLpqw5TDTS6M9rMO6fYR3cXaT6XN+Nm/I3CzSTey063ux6bqOq3H5WBdM3adu9jYLnwFyISs3HyjqqTwy3Dh9PVwUVXMNwAAAAAAAJDOrDvOTEcxpJZiGj77twASaThFDCUxk7Y9Uw4c38VTsbJ+a1XtOzZ+ymE++3cFrlGnV3IKOVnbqderVGfDSWMu5ooXw+Hg1Nydg+47Kafp9ap2kkobUbdL1MCRNe05aOnP1VVGVI27MlTrqvBiuKHzuU8dc2gc3vkfj3AXan+GoxxnMY9yXRVWh0/1xWT7H78J/unnQBq/CEX94pfpgp5tu5yOPms1VqdVzRL0q23zt//8nfftr5L6l7TD5Tfqr7/bY9Tvfp3DqP8airK/SRe08f+CRHh50DSa86phCfrVyzp/9fN0DxEv9tnnzGjUapedFLPLSwr2GTVl0OFCeAFnY5vzV16mHJcImqAPH3S9BF27SbQJU+iwSvmNs3/VZ4SgCfrYQTdum6wHIXqXbmvEcvQurHLXwkeEoAn62EG37i+oHsMrkgcthRrn0yr/1a+O1xE0QR816AvyzVckv/XVfYJu9hg0p1HVLpFUu/xDTfuMmjbo2+0TNJ4eQaMoBI2iEDSKQtAoCkGjKASNohA0ikLQKApBoygEjaIQNIpC0CgKQaMoBI2iEDSKQtAoCkGjKASNohA0ikLQKApBoygEjaIQNIpC0CgKQaMoBI2H+e337/hdmp9B0HiY3//9Hf+W5mcQNB6GoFEUgkZRCBpFIWgUhaBRFIJGUQgaRSks6Fl37fnNsAj6CZUVtNRSTMPpGkE/obKCbvrLawT9hMoK2kza9kw5nlkuQUdv9li/sar2HRs/5Ti/0SlBP6FMglb24oppv1qlOmvddMO4mKvzWoJ+QlkErebuIujKtl+taieptBF1u0S9IOgnlEXQTXsRtBwnl23l38Z6qNZVyrpZyNAJ0XdTxxz6mWURtN8qny9Oxk855DiLeZTrqrA6fKovJtsE/YSyC7rpljn0rNVYnVY1S9Ay/qYvX1qv3vKDkLl9gzahqJRBK63WncJpNOdVwxK0ir+JLfQTym0LbTo349BG+a1yc171MuW4RNBPKLegK7MGLUcTptBhlfIb50a/+iaCfkJZBd0s2+Qw5WiNmKbzKnctfEQI+gllFXS7HID2QQ9aCjXOp1X12Onu1T4hQT+jTIK+SlbVV18j6CdUStBvIegnRNAoCkGjKASNohA0ikLQKApBoygEjaIQNIpC0CgKQaMoBI2iEDSKQtAoCkGjKASNohA0ikLQKApBoygEjaIQNIpC0CgKQaMoBI2iEDSKQtAoCkGjKASNohA0ikLQKApBoygEjaIQNIpC0CgKQaMoBI2iEDSKUljQs+7a87u7EfQTKitoqaWYhtM1gn5CZQXd9JfXCPoJlRW0mbTtmXI8s1yCvnif2Fq9tar2HRs/5Ti/5TdBP6FMglb2dKnW1ur69SrVWeumG8bFXLWnNQT9hLIIWs3dOeixF7LXr1e1k1TaiLpdol4Q9BPKIuimPQddWek3ynXVuCtDta5yX3DXOiH6buqYQz+zLIL2Ha8XZBWuKTnOYh7luiqsDp/qi8n2ly+tV2/+acjXvkGbUFTKoD3V9f4pFDVWp1XNErSMv4kt9BPKbQvtttHGhlnyNJrzqmEJ+tXhD4J+QtkFXXfrHKKxzXnVy5TjEkE/oeyC1uuTgXI0YQodVim/cW70q28i6CeUVdBNI2ZbeUK0RkzTeZW7Fj4iBP2Esgq6bYWxgRi0FGqcT6vqsdPdq31Cgn5GmQR9layqr75G0E+olKDfQtBPiKBRFIJGUQgaRSFoFIWgURSCRlEIGkUhaBSFoFEUgkZRCBpFIWgUhaBRFIJGUQgaRSFoFIWgURSCRlEIGkUhaBSFoFEUgkZRCBpFIWgUhaBRFIJGUQgaRSFoFIWgURSCRlEIGp/kDz+84w8fHZWg8UneTe/3u4xK0NgPQW9F0IdG0FsR9KER9A1m3bW8eX0eCPo6qaWYhtM1gj40gr6u6S+vEfShEfR1ZtK2Z8qRB4I+qd66WvuOjZ9ynN8hmaAPjaBXyn59VXXWuumGcTFX7WkNQR8aQQdq7uzXV9tJKm1E3S5RLwj60Ag6aNpQcNW4T0O1XlW2dtc6Ifpu6phD54GgV5UPWo6zmEe5Xg1fCp/qiwn2H7+pPPmxH4Od5Rh0HYraI2gxazVWp6vNEvSrdL/9k/FU0p+OVHIMuglF7RK0mEZzvjosQb9KlynHoeUY9GKfoBvbnK++TDkuEfShEfRqCVeOJkyhw1XlN86NfvUHCfrQCHq1BN0aMU3nq+5a+IgQ9KER9CoUPGgp1DifrtZjp7vXhzMI+tAI+j2yqr76GkEfGkFvRdCHRtBbEfShEfRWBH1oBL0VQR8aQW9F0IdG0FsR9KER9FYEfWgEvRVBHxpBb0XQh0bQWxH0oRH0VgR9aAS9FUEfGkFvRdCHRtBbEfShEfRWBH1oBL0VQR8aQW9F0IdG0FsR9KER9FYEfWgEvRVBHxpBb0XQh0bQWxH0oRH0VgR9aAS9FUEfGkFvRdCHRtBbEfShEfRWBJ3I775/x28/OipBb0XQifz5vUh++OioBL0VQSdC0DGCzhxBxwg6cwQdI+jMEXSMoDNH0LFbg67rND/vhKATIejYbUE3o20rnbJpgk6EoGM3BV3bpmrl+lZtaRB0IgQduylo04uqFaKrbvizNyLoRAg6dlvQhqCPiqBjNwVdjbULumHKcUAEHbttp3CwetRj81Or6+Wd6es1+GtLj6ATIejYjYftVGPmn9o+19raVgrVWdv7P3pluSDoRAg6dlPQapk8N+rNtdoI2fWinaTS/m29rywXBJ0IQcduCFpVQ185s357p9C6L5tW2dpNTTpxbbki6EQIOnZD0E3b6dab3p50jIMQUx/ep959urZcEXQiBB277YmV5r21ldtf1LJZgr26XL+JoBMh6NiWk5PenkPLbqrmrh+WYNW15fpdX76EjX7iM0SeEEGfmFDUjedyTP7Pjm/OoRst/JPjPzLl+BQEHbvxiRXTtabr31xp/I6edEGrELe6slwRdCIEHbv1qe+5F1K/uVNY+8MXZhStPzR3w8eCoBMh6NitQdety/Htw3aN7fRYi3rsdOeSv7ZcEHQiBB27KehGK+FmDD9xHFqoKqyQy+LqMiDoRAg6dttOYdu6SYXubvmjNyLoRAg6dttOoT8OPTcJT7Yj6FQIOnZT0EPKbfOCoBMh6NhtU47e+JM5Ep7fT9CpEHTstimHXaT5kQFBJ0LQMf5djswRdIygM0fQMYLOHEHHCDpzBB0j6MwRdIygM0fQMYLOHEHHCDpzBB0j6MwRdIygM0fQMYLOHEHHCDpzBB0j6MwRdIygM0fQMYLOHEHHCDpzBB0j6MwRdIygM0fQMYLOHEHHCDpzBB0j6MwRdIygM0fQMYLOHEHHCDpzBB0j6MwRdIygM0fQMYLOHEHHCDpzBB0j6MwRdIygM0fQMYLOHEHHCDpzBB0j6MwRdIygM0fQMYLOHEHHkgQt67Co1/d9u7b0CDoRgo4lCFpO1nZKqM5a//b215YLgk6EoGMJgu47KadJtJNU2r9B/ZXlgqATIejY/UFL6yYcyii/GDpxbbki6EQIOnZ/0JUVdSX9Ily+tlwRdCIEHbs/6Nm2Wo91swQrry3X7yLoRAg6dn/QxrqJsRmHJVh1bbl+15cvrVen+b94YgR9YkJRSaYcfiL9G6Ycn4KgY/cHXS9B/8VvfBvtdv7eX64IOhGCjiU4bKcHIXotWn9o7oaPBUEnQtCxBEHXY+d2CsOik+LqckHQiRB0LMVT37KqLhZXlwFBJ0LQMU5OyhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxwg6cwQdI+jMEXSMoDNH0DGCzhxBxxIFXYdP6xsfX1t6BJ0IQcfSBG1aIVRnbS+uLxcEnQhBx5IEXVkXdDtJpc315YKgEyHoWJI3rx+nVijrph1Dd3W5IuhECDqWIujJuClHZYXfVF9drgg6EYKOJQi66fwculmCldeW6zf98ZvKkx/9oVgR9Ekdiro/aKWVD3pYglXXlut3ffsn46mP/lSsCPqkCUXdH7Tp3IxDmx+ZcnwKgo7dH3RlQtC/9BvfRrudv/eXK4JOhKBj6Y5Dt+a2jwVBJ0LQsXRB12OnO3l9uSDoRAg6lvBcDllVNy0Dgk6EoGOcnJQ5go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjBJ05go4RdOYIOkbQmSPoGEFnjqBjSYKu1bJY3yf22tIj6EQIOpYg6Fpbq2uhOmt7d/XackHQiRB0LEHQYy9kr0U7SaX9G9RfWS4IOhGCjt0fdGXdTELZv9haiKFzl95frgg6EYKO3R+09G9JX9kfbVj4/95drgg6EYKOpTnKobq+WYKV15brdxB0IgQdSxG0NNaIYQlWXVuu3/PlS+vVaf4vnhhBn5hQVIqjHJ3v8tpUgynHPgg6liBoHQ7GKb/xbfTV5YqgEyHo2P1Bz7byROsPzd3wsXjCoP/9h3d8OBKCjt0ftLGBqMdOd26f79py8YRB/8cu6RF0LOG5HLKqbloGBE3QRw96E4ImaILOG0ETdFEImqCLQtAE/Un++rf3/PWDoxI0QX+S79+74f/+/QdHJWiC/iQETdBbETRBE/SjEDRBb0XQBE3Qj0LQBL0VQRM0QT8KQRP0VgRN0AT9KARN0FsRNEET9KMQNEFvRdAETdCPQtAEvRVBEzRBPwpBE/RWBE3QBP0oBE3QWxE0QRP0oxA0QW9F0ARN0I9C0AS9FUETNEE/CkET9FYETdAE/SgETdBbETRBE/SjEDRBb0XQBE3Qj0LQBL0VQRM0QT8KQRP0VgRN0AT9hv/88zv+64ODEjRBb5Uq6P9+7zb62wcHJWiCvkUtXy6nCvoHgiboS48LWnXW9udrBE3QmQfdTlLp5G9eT9AEHXlY0MrWQgzd6SpBE3TeQVf29CkgaILOO+hmCfq0X0jQBJ130MMStFqvfrHAHj5pyvE/vwH28L8PClr5jXOjH/TTgL21ZvkAilCPne7k/eMAxyCratfx1f1DPGjQrEbFJxnaXAbNalR8Fjm2/f2jPGLQrEYVimnig62PtNU4pn/MTT2o2u9X3WdU2dqxTj7qoC2HB35K7U8Skb2Sqkt3I8lm9p/7/0s56PrEUvJR627YYdRFb1SffCrTd/M87rtDlbPJ3eBSz5fPQ95JmtHaddRkgwbtHqPOdpQ7/K5CNOM4+qlMk3TU5SkJzVMSP0VZF3OY6E1TkgFn3VVuw98soyYadFGffteUoyrb9Tv8rrLu5iE8HTamnUZLLWXX6yHpoEVo2/C4ZfR6g6uxSjDfm8ZwU5+eBEozqGiXzVyvk466/I+Pg63Tj9r7R72wn5l8KqPcRr8Z2d2MqGl2Oyw+aXl+3YCx+s4byY06hH2rMIlJNGgYZRlEjibdqLJddtja5jTPTfK7GhN2Snx1butf+d3NNOktBc+zmIy/Qex874AFcdPcSbnJhrEu6cYnKI27ue7c0wijCu22SpW2thPLqEl2X04lDz6OFKO6u55orZ8YuQJDfKl+V//Y5O/Pxj+aGH9XSTI1Xw6Y1K2bz7mpUaUlTwS9mLX2f3XuJle9S7qd1m1JklEr20wuP+UTTDCq6kc/0rDusekkoy53PWUHP3TVhgeABL9rEA5AyPWRT46ptqPLAZPKz+fq0er0BwPzpdqxD395YbPnktb+2r0Pi+dRxWR7P1iTYlT3l9dXs6+iW/bYwibv3lHXu57ruHY7hNIu95P7f9fJP8yd7xezfzRJE96rAyaSnC/17mZvw6xx2QF3SSfYZX4ZVdl1D66/f+4owy6mH6daQklwxuHLXc+NLrtWus3dfPc8t+4Gt90fjXzZ0Kd66nG3AyYFWaPrlps81c20jmrS3e4vr2kQUyfdzlaCoV/uen5KLvsuPLNy76jL4exBj+shEzclr++e5u5+wCR364RUnA4bVEnmja9HPR+LSDC0u4tUprPuwVZ2Y2vbRHeV06OIL6RP8kTycjjbHymxvRJJdh/2O2BSivOE9Bxdiidmvx61sclu995a25nZ+O1eM9w7d3x913P36Pr+eW50OLvSVWv7OsnD3k4HTIrxMiEVPrqwHbn/Fnpj1JS3e12FoZM82H591wvP/N/5/x8fzvanobqkkzzfvdMBk2JcTEjF+bDBQUd9TSco5M273v33vfhwtll2jO8cc78DJiW5mJCKhE9g7TLqC+MfvqcULz/b4a7nn5+JD2ebJGfX7XXApDCXE9Kjj/oy/Nj3Y5KdweR3veX5mfhwdoK9kuWE1sQHTMqUckK696hnzWQSnf2b+K53en4mOpxt7k8vHAEUaQ+YlC3FhPRRoyaV8q738vxMysPZYeT1CGDKAyalSjch3X/U/SS56108P5PucHZ0BDDlAZNSpZuQ7j/qHhLf9U7PzyQ5nP3VEcA0B0wKl25Cuv+oO0h91zu9OuL+w9lvnNCa5oAJipb4rnd6fubuw9lvndCa/vW1wBVNmlfWvnlCa4IDJsBGKZ6f2eeEVuADUjw1us8JrcAnSn9CK/CZUp7QCny6dCe0AkeQ7oRW4Aj2Opcc+BS8YhAAAAAoyP8DXzgkj1Ps6yIAAAAldEVYdGRhdGU6Y3JlYXRlADIwMjItMDMtMjdUMDg6NDg6NTgrMDc6MDC3itnIAAAAJXRFWHRkYXRlOm1vZGlmeQAyMDIyLTAzLTI3VDA4OjQ4OjU4KzA3OjAwxtdhdAAAACF0RVh0cHM6SGlSZXNCb3VuZGluZ0JveAA1MDR4NzIwKzUwKzUw1uIiwwAAABN0RVh0cHM6TGV2ZWwAQWRvYmUtMi4wCjmTdA0AAAAASUVORK5CYII=" />


 #table9#
 {dataset=>"len10ws1"}
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Util::trim             |    850421 |  1175.89  |                 0.00% |               687.42% |   0     |      20 |
 | String::Trim::Regex::trim      |    860740 |  1161.8   |                 1.21% |               677.97% | 5.8e-12 |      25 |
 | String::Trim::More::trim       |    960000 |  1000     |                12.71% |               598.63% | 1.7e-09 |      20 |
 | String::Trim::NonRegex::trim   |   1000000 |   960     |                22.77% |               541.36% | 1.7e-09 |      20 |
 | Text::Minify::XS::minify       |   6200000 |   160     |               624.51% |                 8.68% | 3.5e-10 |      29 |
 | Text::Minify::XS::minify_ascii |   6696350 |   149.335 |               687.42% |                 0.00% |   0     |      20 |
 +--------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate  SU:t  STR:t  STM:t  STN:t  TMX:m  TMX:m_a 
  SU:t      850421/s    --    -1%   -14%   -18%   -86%     -87% 
  STR:t     860740/s    1%     --   -13%   -17%   -86%     -87% 
  STM:t     960000/s   17%    16%     --    -4%   -84%     -85% 
  STN:t    1000000/s   22%    21%     4%     --   -83%     -84% 
  TMX:m    6200000/s  634%   626%   525%   500%     --      -6% 
  TMX:m_a  6696350/s  687%   677%   569%   542%     7%       -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify
   TMX:m_a: participant=Text::Minify::XS::minify_ascii

=for html <img src="data:image/png;base64,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" />


Benchmark module startup overhead (C<< bencher -m StringFunctions::Trim --module-startup >>):

 #table10#
 +------------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+
 | participant            | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+
 | String::Util           |     14.3  |              8.14 |                 0.00% |               132.46% | 1.2e-05 |      20 |
 | String::Trim::Regex    |     12.2  |              6.04 |                17.49% |                97.85% |   1e-05 |      21 |
 | Text::Minify::XS       |      9.36 |              3.2  |                53.00% |                51.94% | 6.4e-06 |      21 |
 | String::Trim::More     |      9.07 |              2.91 |                58.00% |                47.13% |   9e-06 |      20 |
 | String::Trim::NonRegex |      9    |              2.84 |                59.03% |                46.18% |   1e-05 |      20 |
 | perl -e1 (baseline)    |      6.16 |              0    |               132.46% |                 0.00% | 5.4e-06 |      23 |
 +------------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+


Formatted as L<Benchmark.pm|Benchmark> result:

                          Rate   S:U  ST:R  TM:X  ST:M  ST:N  perl -e1 (baseline) 
  S:U                   69.9/s    --  -14%  -34%  -36%  -37%                 -56% 
  ST:R                  82.0/s   17%    --  -23%  -25%  -26%                 -49% 
  TM:X                 106.8/s   52%   30%    --   -3%   -3%                 -34% 
  ST:M                 110.3/s   57%   34%    3%    --    0%                 -32% 
  ST:N                 111.1/s   58%   35%    4%    0%    --                 -31% 
  perl -e1 (baseline)  162.3/s  132%   98%   51%   47%   46%                   -- 
 
 Legends:
   S:U: mod_overhead_time=8.14 participant=String::Util
   ST:M: mod_overhead_time=2.91 participant=String::Trim::More
   ST:N: mod_overhead_time=2.84 participant=String::Trim::NonRegex
   ST:R: mod_overhead_time=6.04 participant=String::Trim::Regex
   TM:X: mod_overhead_time=3.2 participant=Text::Minify::XS
   perl -e1 (baseline): mod_overhead_time=0 participant=perl -e1 (baseline)

=for html <img src="data:image/png;base64,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" />


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-StringFunctions>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-StringFunctions>.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla plugin and/or Pod::Weaver::Plugin. Any additional steps required
beyond that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022, 2021, 2018 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-StringFunctions>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
