#! /usr/bin/env perl

use strict;
use Modern::Perl;

use App::HPGL2Cadsoft;
use Getopt::Long;
use Pod::Usage;

my ( $infile, $outfile, $scaling, $help, $man);

GetOptions(
    'input=s' =>   \$infile,
    'output=s'  => \$outfile,
    'scaling=s' => \$scaling,
    'help|?|h' =>  \$help,
    'man'      =>  \$man,
) or pod2usage(2);
pod2usage(1)
  if ( $help || !defined($infile) );
pod2usage( -exitstatus => 0, -verbose => 2 ) if ($man);

my $params;
$params->{input_file} = $infile;

# Auto-generate output file if none is specified.
if (!defined $outfile) {
    # Strip extension from inputfile name
    $infile =~ s{\.[^.]+$}{};
    $outfile = $infile . ".scr";
}

$params->{output_file} = $outfile;

$params->{scaling_factor} = $scaling if (defined $scaling);

my $convertor = App::HPGL2Cadsoft->new( $params );
$convertor->run();

# PODNAME: hpgl2cadsoft.pl
# ABSTRACT: program to convert a HPGL graphical file into a Cadsoft Eagle script for importing into your PCB

__END__

=pod

=encoding UTF-8

=head1 NAME

hpgl2cadsoft.pl - program to convert a HPGL graphical file into a Cadsoft Eagle script for importing into your PCB

=head1 VERSION

version 0.01

=head1 SYNOPSIS

Converts a HPGL file into a Cadsoft Eagle script. Usage:

    hpgl2cadsoft.pl --input <inputfile.hpgl> --output <outputfile.scr> --scaling <scalingfactor>

=head1 DESCRIPTION

This program converts a graphical file in HPGL format into a script file that you can run
in the Cadsoft Eagle board editor to import the graphic into your board file.

It allows you to scale the output by using a different scaling factor. 

Only the input file is required, all other parameters are optional. If the output file is not specified, it 
will be generated by stripping the extension from the inputfile and appending the 'scr' extension required
by Cadsoft Eagle.

If scaling is not defined it will default to 342.3151, the scaling factor required for 1-1 mappings from
Inkscape DXF import to millimeters in Eagle.

=head1 AUTHOR

Lieven Hollevoet <hollie@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Lieven Hollevoet.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
