#!/usr/local/bin/perl
#
#Base class for sacctmgr entities

package Slurm::Sacctmgr::EntityBase;
use strict;
use warnings;
use base qw(Class::Accessor);
use Carp qw(carp croak);

#-------------------------------------------------------------------
#	Globals
#-------------------------------------------------------------------

#-------------------------------------------------------------------
#	Accessors
#-------------------------------------------------------------------

sub _ro_fields($)
{	return [];
}

sub _rw_fields($)
{	return [];
}

sub _required_fields($)
{	return [];
}

sub _special_fields($)
#This should be overloaded to return any fields with special data types
#for this class of entity
{	my $class = shift;
	return {};
}

#-------------------------------------------------------------------
#	Constructors, etc
#-------------------------------------------------------------------

sub new($;@)
{	my $class = shift;
	my @args = @_;
	$class = ref($class) if ref($class);

	my $obj = {};
	bless $obj, $class;

	$obj->_parse_args(@args);
	$obj->_set_defaults;
	$obj->_init;

	return $obj;
}

sub _parse_args($@)
{	my $obj = shift;
	my %args = @_;

	my $accessors = $obj->_rw_fields;
	my ($arg, $meth, $val);
	RWARG: foreach $arg (@$accessors)
	{	next RWARG unless exists $args{$arg};
		$val = delete $args{$arg};
		next RWARG unless defined $val;
		$meth = $arg;
		$obj->$meth($val);
	}

	$accessors = $obj->_ro_fields;
	ROARG: foreach $arg (@$accessors)
	{	next ROARG unless exists $args{$arg};
		$val = delete $args{$arg};
		next ROARG unless defined $val;
		$meth = $arg;
		$obj->set($meth,$val);
	}


	#Warn about unknown arguments
	if ( scalar(keys %args) )
	{	my $tmp = join ", ", (keys %args);
		croak "Unrecognized arguments [ $tmp ] to constructor at ";
	};
}

sub _set_defaults($)
{	my $obj = shift;

	return;
}

sub _init($)
{	my $obj = shift;

	my ($fld, $meth, $val);
	my $req_parms = $obj->_required_fields;
	foreach $fld (@$req_parms)
	{	$meth = $fld;
		$val = $obj->$meth;
		unless ( defined $val )
		{	croak "Missing required argument $fld";
		}
	}

}

	
#-------------------------------------------------------------------
#	Special constructor to generate from a sacctmgr list entry
#-------------------------------------------------------------------

sub _sacctmgr_fields_in_order($)
#Should return a list ref of field names in order sacctmgr will return them
{	my $class = shift;
	$class = ref($class) if ref($class);
	die "Class $class forgot to overload _sacctmgr_fields_in_order"
}

#-------------------------------------------------------------------
#	Lookup entity with sacctmgr
#-------------------------------------------------------------------

sub _sacctmgr_entity_name($)
{	my $class = shift;
	$class = ref($class) if ref($class);

	my $base = $class;
	$base =~ s/^.*://;
	$base = lc $base;
	return $base;
}


sub _sacctmgr_name_field($)
{	my $class = shift;
	$class = ref($class) if ref($class);
	die "Class $class did not overload _sacctmgr_name_field ";
}

sub _my_sacctmgr_where_clause($)
#This might need to be overloaded.
#Returns a where clause hash that should return the current entity.
{	my $obj = shift;
	croak "Must be called as an instance method at "
		unless $obj && ref($obj);
	my $namefld = $obj->_sacctmgr_name_field;
	my $meth = $namefld;
	my $val = $obj->$meth;
	$val = '' unless defined $val;
	return { name => $val };
}

1;
__END__

=head1 NAME

Slurm::Sacctmgr::EntityBase

=head1 SYNOPSIS

  package Slurm::Sacctmgr::Account;
  use base qw(Slurm::Sacctmgr::EntityBase);

  sub _ro_fields($)
  {	my $class = shift;
	return qw( account description organization coordinators);
  }

  sub _special_fields($)
  {	my $class = shift;
	return { 'coordinators' => 'csv' };
  }

  ...


=head1 DESCRIPTION

This is the base class for entities managed by sacctmgr.
It provides common constructors, etc.

Child classes should overload the following methods to customize
behavior:

=over 4

=item B<_ro_fields> 

Should return a list ref of the names of the
read-only fields for this entity.  Default is empty.

=item B<_rw_fields>

Should return a list ref of the names of the
read-write fields for this entity.  Default is empty.

=item B<_special_fields>

Should return a hash ref, keyed on field name, with value 
a string indicating the type of the field, for any and all
fields of this entity class which are non-trivial.  See
below for list of known data types.  Default is empty.

=item B<_required_fields>

A list of required fields for the constructor to make the object.
Default is empty.

=item B<_sacctmgr_fields_in_order>

A list ref of field names in order sacctmgr will return them
Must be overridden.

=item B<_sacctmgr_entity_name>

The name of sacctmgr entity associated with this Perl class.
Default is just derived from the class name.

=item B<_my_sacctmgr_where_clause>

The "where" clause to give to the sacctmgr list command to list
just the record corresponding to this instance in the database.
The default is "name=>$NameValue" where NameValue is the value of
the field named B<_sacctmgr_name_field>.  Should be a hash ref
of key => value pairs.

=item B<_sacctmgr_name_field>

Used to default B<_my_sacctmgr_where_clause> above.


=back

The data types for the B<_special_fields> hash are:

=over 4

=item B<csv>: 

The should be converted to a list ref from a comma separated string

=back

Most useful methods are in subclasses depending on which commands
can be issues on entities of that type.  E.g, 

=over 4

=item B<EntityBaseListable> for entities supporting the "list" command

=item B<EntityBaseAddDel> for entities with the "add" and "delete" command

=item B<EntityBaseModifiable> for entities supporting the "modify" command

=item B<EntityBaseRW> for entities with the "list", "add", "delete" and "modify" commands

=back

=head2 EXPORT

None.  Pure OO interface.

=head1 SEE ALSO

B<EntityBaseListable>
B<EntityBaseAddDel>
B<EntityBaseModifiable>
B<EntityBaseRW>

=head1 AUTHOR

Tom Payerle, payerle@umd.edu

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2014-2016 by the University of Maryland.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

