package DBIx::QuickORM::Builder::DB;
use strict;
use warnings;

use DBIx::QuickORM::BuilderState;

use Carp qw/croak confess/;
use Sub::Util qw/set_subname/;

use Importer Importer => 'import';

$Carp::Internal{(__PACKAGE__)}++;

my @EXPORT = qw{
    db
    db_attributes
    db_class
    db_connect
    db_dsn
    db_host
    db_name
    db_password
    db_port
    db_socket
    db_user
};

# sub db {
build_top_builder db => sub {
    my %params = @_;

    my $args      = $params{args};
    my $state     = $params{state};
    my $caller    = $params{caller};
    my $wantarray = $params{wantarray};

    require DBIx::QuickORM::DB;
    if (@$args == 1 && !ref($args->[0])) {
        croak 'useless use of db($name) in void context' unless defined $wantarray;
        return _get('db', $caller->[0], $args->[0]);
    }

    my ($name, $cb);
    for my $arg (@$args) {
        $name = $arg and next unless ref($arg);
        $cb = $arg and next if ref($arg) eq 'CODE';
        croak "Not sure what to do with argument '$arg'";
    }

    croak "A codeblock is required to build a database" unless $cb;

    my $orm = $state->{+ORM_STATE};
    if ($orm) {
        croak "Quick ORM '$orm->{name}' already has a database" if $orm->{db};
    }
    elsif (!$name) {
        croak "useless use of db(sub { ... }) in void context. Either provide a name, or assign the result";
    }

    my %db = _new_db_params($name => $caller);

    $state->{+DB} = \%db;

    update_subname($name ? "db builder $name" : "db builder", $cb)->(\%db) if $cb;

    my $db = _build_db(\%db);

    if ($orm) {
        croak "Quick ORM instance already has a db" if $orm->{db};
        $orm->{db} = $db;
    }

    _set('db', $caller->[0], $name, $db) if $name;

    return $db;
};

sub _get_db {
    my $state = build_state or return;

    return $state->{+DB} if $state->{+DB};
    return unless $state->{+ORM_STATE};

    croak "Attempt to use db builder tools outside of a db builder in an orm that already has a db defined"
        if $state->{+ORM_STATE}->{db};

    my %params = _new_db_params(undef, [caller(1)]);

    $state->{+DB} = \%params;
}

sub db_attributes {
    my %attrs = @_ == 1 ? (%{$_[0]}) : (@_);

    my $db = _get_db() or croak "attributes() must be called inside of a db or orm builer";
    %{$db->{attributes} //= {}} = (%{$db->{attributes} // {}}, %attrs);

    return $db->{attributes};
}

sub db_connect {
    my ($in) = @_;

    my $db = _get_db() or croak "connect() must be called inside of a db or ORM builer";

    if (ref($in) eq 'CODE') {
        return $db->{connect} = $in;
    }

    my $code = do { no strict 'refs'; \&{$in} };
    croak "'$in' does not appear to be a defined subroutine" unless defined(&$code);
    return $db->{connect} = $code;
}

for my $db_field (qw/db_class db_name db_dsn db_host db_socket db_port db_user db_password/) {
    my $name = $db_field;
    my $attr = $name;
    $attr =~ s/^db_// unless $attr eq 'db_name';
    my $sub  = sub {
        my $db = _get_db() or croak "$name() must be called inside of a db builer";
        $db->{$attr} = $_[0];
    };

    no strict 'refs';
    *{$name} = set_subname $name => $sub;
}

1;
