package ore;

use 5.014;

use strict;
use warnings;
use routines;

use base 'Exporter';

use Data::Dump;
use Data::Object::Space;

no strict 'refs';

our @EXPORT=('dd');

our $VERSION = '0.01'; # VERSION

sub import {
  new_vars();
  use_vars();

  ore->export_to_level(1, @_);
}

fun new_vars() {
  new_vars_pump([map new_vars_form($_), grep /^New_/, keys %ENV])
}

fun new_vars_args($spec) {
  [map { /^\$(\w+)$/ ? ${"ore::$1"} : $_ } @$spec[2..$#$spec]]
}

fun new_vars_form($name) {
  new_vars_make(new_vars_spec($name))
}

fun new_vars_make($spec) {
  [$spec->[1], Data::Object::Space->new($spec->[0])->build(@{new_vars_args($spec)})]
}

fun new_vars_name($name) {
  $name =~ s/^New_//gr =~ s/_/\//gr
}

fun new_vars_spec($name) {
  [new_vars_name($name), split /;\s*/, $ENV{$name}]
}

fun new_vars_pump($conf) {
  push @EXPORT, map { ${"ore::$$_[0]"} = $$_[1]; '$'.$$_[0] } @$conf
}

fun use_vars() {
  use_vars_pump([map use_vars_form($_), grep /^Use_/, keys %ENV])
}

fun use_vars_args($spec) {
  [map { /^\$(\w+)$/ ? ${"ore::$1"} : $_ } @$spec[2..$#$spec]]
}

fun use_vars_form($name) {
  use_vars_make(use_vars_spec($name))
}

fun use_vars_make($spec) {
  [$spec->[1], Data::Object::Space->new($spec->[0])]
}

fun use_vars_name($name) {
  $name =~ s/^Use_//gr =~ s/_/\//gr
}

fun use_vars_spec($name) {
  [use_vars_name($name), split /;\s*/, $ENV{$name}]
}

fun use_vars_pump($conf) {
  push @EXPORT, map { ${"ore::$$_[0]"} = $$_[1]; '$'.$$_[0] } @$conf
}

1;

=encoding utf8

=head1 NAME

ore

=cut

=head1 ABSTRACT

Sugar for Perl 5 one-liners

=cut

=head1 SYNOPSIS

  BEGIN {
    $ENV{New_File_Temp} = 'ft';
  }

  use ore;

  $ft

=cut

=head1 DESCRIPTION

This package provides automatic object instantiation based on environment
variables. This package exists because I was bored and inspired by L<new>.

Simple command-line example with object instantiation:

  $ New_File_Temp=ft perl -More -e 'dd $ft'

  # "File::Temp"

Another simple command-line example returning a L<Data::Object::Space> object:

  $ Use_DBI=dbi perl -More -e 'dd $dbi->children'

  # [
  #   ...,
  #   bless(do{\(my $o = "DBI/DBD")}, "Data::Object::Space"),
  #   bless(do{\(my $o = "DBI/Profile")}, "Data::Object::Space"),
  #   bless(do{\(my $o = "DBI/ProfileData")}, "Data::Object::Space"),
  #   bless(do{\(my $o = "DBI/ProfileDumper")}, "Data::Object::Space"),
  #   ...,
  # ]

=cut

=head1 AUTHOR

Al Newkirk, C<awncorp@cpan.org>

=head1 LICENSE

Copyright (C) 2011-2019, Al Newkirk, et al.

This is free software; you can redistribute it and/or modify it under the terms
of the The Apache License, Version 2.0, as elucidated in the L<"license
file"|https://github.com/iamalnewkirk/ore/blob/master/LICENSE>.

=head1 PROJECT

L<Wiki|https://github.com/iamalnewkirk/ore/wiki>

L<Project|https://github.com/iamalnewkirk/ore>

L<Initiatives|https://github.com/iamalnewkirk/ore/projects>

L<Milestones|https://github.com/iamalnewkirk/ore/milestones>

L<Contributing|https://github.com/iamalnewkirk/ore/blob/master/CONTRIBUTE.md>

L<Issues|https://github.com/iamalnewkirk/ore/issues>

=cut
