#!/usr/bin/perl
#
# Script by Philippe "BooK" Bruhat
# Table by Toby Nelson
#

use strict;
use warnings;
use Games::Mastermind;

# the strategy table is initialised in the BEGIN block at the end
our $strategy;

my $game  = Games::Mastermind->new();    # standard game
my $holes = $game->holes();
my @pegs  = @{ $game->pegs() };
my $here  = $strategy;

my $result = [ 0, 0 ];
while ( defined $result && $result->[0] != $holes ) {
    my @guess = map { $pegs[$_] } @{ $here->{guess} };
    $result = $game->play(@guess);

    print "@guess | @$result\n";
    my $key = join ",", @$result;
    $here = $here->{$key};
}

BEGIN {

    # this strategy table is based on the one available here:
    # http://www.tnelson.demon.co.uk/mastermind/table.html
    $strategy = {
        guess => [ '0', '0', '1', '1' ],
        '0,0' => {
            guess => [ '2', '2', '3', '4' ],
            '0,0' => { guess => [ '5', '5', '5', '5' ] },
            '0,1' => {
                guess => [ '5', '5', '3', '5' ],
                '0,2' => { guess => [ '3', '3', '5', '3' ] },
                '1,0' => { guess => [ '4', '4', '4', '5' ] },
                '1,1' => { guess => [ '4', '4', '5', '5' ] },
                '1,2' => {
                    guess => [ '0', '3', '0', '5' ],
                    '0,2' => { guess => [ '3', '5', '5', '3' ] },
                    '1,1' => { guess => [ '5', '3', '5', '3' ] },
                    '2,0' => { guess => [ '3', '3', '5', '5' ] }
                },
                '2,0' => {
                    guess => [ '0', '4', '0', '0' ],
                    '0,1' => { guess => [ '4', '5', '4', '5' ] },
                    '1,0' => { guess => [ '5', '4', '4', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '4', '0', '0' ],
                    '0,0' => { guess => [ '5', '5', '5', '2' ] },
                    '0,1' => { guess => [ '4', '5', '5', '5' ] },
                    '1,0' => { guess => [ '5', '4', '5', '5' ] }
                },
                '2,2' => {
                    guess => [ '0', '3', '0', '5' ],
                    '0,2' => { guess => [ '5', '5', '5', '3' ] },
                    '1,1' => { guess => [ '3', '5', '5', '5' ] },
                    '2,0' => { guess => [ '5', '3', '5', '5' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,0' => { guess => [ '5', '5', '4', '5' ] },
                    '0,1' => { guess => [ '5', '5', '2', '5' ] }
                }
            },
            '0,2' => {
                guess => [ '5', '5', '2', '3' ],
                '0,1' => { guess => [ '4', '4', '4', '2' ] },
                '0,2' => {
                    guess => [ '0', '3', '0', '3' ],
                    '0,0' => { guess => [ '4', '4', '5', '2' ] },
                    '0,1' => { guess => [ '3', '4', '4', '5' ] },
                    '1,0' => { guess => [ '4', '3', '4', '5' ] },
                    '1,1' => { guess => [ '3', '3', '4', '5' ] }
                },
                '0,3' => {
                    guess => [ '0', '3', '0', '2' ],
                    '0,1' => { guess => [ '3', '4', '5', '5' ] },
                    '1,0' => { guess => [ '4', '3', '5', '5' ] },
                    '2,0' => { guess => [ '3', '3', '5', '2' ] }
                },
                '1,0' => {
                    guess => [ '0', '3', '3', '3' ],
                    '1,0' => { guess => [ '4', '4', '4', '3' ] },
                    '1,1' => { guess => [ '3', '4', '4', '3' ] },
                    '2,0' => { guess => [ '4', '3', '4', '3' ] },
                    '2,1' => { guess => [ '3', '3', '4', '3' ] }
                },
                '1,1' => {
                    guess => [ '0', '4', '2', '4' ],
                    '0,1' => { guess => [ '4', '3', '5', '3' ] },
                    '0,3' => { guess => [ '4', '5', '4', '2' ] },
                    '1,0' => { guess => [ '3', '4', '5', '3' ] },
                    '1,1' => { guess => [ '4', '4', '5', '3' ] },
                    '1,2' => { guess => [ '5', '4', '4', '2' ] },
                    '2,1' => { guess => [ '4', '4', '2', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '5', '0', '2' ],
                    '0,1' => { guess => [ '5', '3', '4', '5' ] },
                    '0,2' => { guess => [ '3', '3', '2', '5' ] },
                    '1,0' => { guess => [ '3', '5', '4', '5' ] },
                    '1,1' => { guess => [ '5', '4', '5', '2' ] },
                    '2,0' => { guess => [ '4', '5', '5', '2' ] }
                },
                '1,3' => {
                    guess => [ '0', '3', '0', '0' ],
                    '0,1' => { guess => [ '3', '5', '5', '2' ] },
                    '1,0' => { guess => [ '5', '3', '5', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '5', '4', '4' ],
                    '0,0' => { guess => [ '3', '3', '2', '3' ] },
                    '1,1' => { guess => [ '5', '3', '4', '3' ] },
                    '1,2' => { guess => [ '5', '4', '4', '3' ] },
                    '2,0' => { guess => [ '3', '5', '4', '3' ] },
                    '2,1' => { guess => [ '4', '5', '4', '3' ] }
                },
                '2,1' => {
                    guess => [ '0', '4', '2', '0' ],
                    '0,1' => { guess => [ '4', '5', '5', '3' ] },
                    '0,2' => { guess => [ '5', '5', '4', '2' ] },
                    '1,0' => { guess => [ '5', '4', '5', '3' ] },
                    '1,1' => { guess => [ '4', '5', '2', '5' ] },
                    '2,0' => { guess => [ '5', '4', '2', '5' ] }
                },
                '2,2' => {
                    guess => [ '0', '3', '0', '0' ],
                    '0,1' => { guess => [ '3', '5', '2', '5' ] },
                    '1,0' => { guess => [ '5', '3', '2', '5' ] }
                },
                '3,0' => {
                    guess => [ '0', '3', '0', '2' ],
                    '0,1' => { guess => [ '5', '5', '4', '3' ] },
                    '0,2' => { guess => [ '3', '5', '2', '3' ] },
                    '1,0' => { guess => [ '5', '5', '2', '2' ] },
                    '1,1' => { guess => [ '5', '3', '2', '3' ] }
                }
            },
            '0,3' => {
                guess => [ '3', '5', '4', '2' ],
                '0,3' => {
                    guess => [ '0', '0', '3', '3' ],
                    '1,0' => { guess => [ '4', '4', '2', '3' ] },
                    '1,1' => { guess => [ '4', '3', '2', '3' ] }
                },
                '0,4' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,1' => { guess => [ '4', '3', '2', '5' ] },
                    '1,0' => { guess => [ '5', '4', '2', '3' ] }
                },
                '1,1' => { guess => [ '4', '4', '2', '2' ] },
                '1,2' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,0' => { guess => [ '5', '4', '2', '2' ] },
                    '0,1' => { guess => [ '5', '3', '2', '2' ] },
                    '1,0' => { guess => [ '3', '4', '2', '3' ] }
                },
                '1,3' => {
                    guess => [ '0', '0', '2', '3' ],
                    '0,2' => { guess => [ '4', '3', '5', '2' ] },
                    '1,1' => { guess => [ '3', '4', '2', '5' ] },
                    '2,0' => { guess => [ '4', '5', '2', '3' ] }
                },
                '2,0' => { guess => [ '3', '3', '2', '2' ] },
                '2,1' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,0' => { guess => [ '4', '5', '2', '2' ] },
                    '0,1' => { guess => [ '4', '3', '4', '2' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '4', '0' ],
                    '0,1' => { guess => [ '3', '4', '5', '2' ] },
                    '1,0' => { guess => [ '5', '3', '4', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '4', '4' ],
                    '0,0' => { guess => [ '3', '5', '2', '2' ] },
                    '1,0' => { guess => [ '3', '3', '4', '2' ] },
                    '1,1' => { guess => [ '3', '4', '4', '2' ] }
                }
            },
            '0,4' => {
                guess => [ '0', '3', '0', '0' ],
                '0,1' => { guess => [ '3', '4', '2', '2' ] },
                '1,0' => { guess => [ '4', '3', '2', '2' ] }
            },
            '1,0' => {
                guess => [ '2', '5', '4', '5' ],
                '0,0' => { guess => [ '3', '3', '3', '3' ] },
                '0,1' => { guess => [ '5', '3', '3', '3' ] },
                '0,2' => { guess => [ '4', '4', '5', '4' ] },
                '0,3' => { guess => [ '5', '4', '5', '4' ] },
                '1,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,0' => { guess => [ '4', '4', '4', '4' ] },
                    '0,1' => { guess => [ '3', '3', '3', '5' ] },
                    '1,0' => { guess => [ '3', '5', '3', '3' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,0' => { guess => [ '5', '4', '4', '4' ] },
                    '0,1' => { guess => [ '5', '3', '3', '5' ] },
                    '1,0' => { guess => [ '5', '5', '3', '3' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '4', '4' ],
                    '0,0' => { guess => [ '5', '2', '5', '5' ] },
                    '1,0' => { guess => [ '5', '5', '5', '4' ] },
                    '1,1' => { guess => [ '4', '5', '5', '4' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '3', '3' ],
                    '0,0' => { guess => [ '4', '5', '4', '4' ] },
                    '1,0' => { guess => [ '5', '5', '3', '5' ] },
                    '1,1' => { guess => [ '3', '5', '3', '5' ] }
                },
                '2,1' => { guess => [ '5', '5', '4', '4' ] },
                '3,0' => { guess => [ '2', '5', '5', '5' ] }
            },
            '1,1' => {
                guess => [ '2', '2', '5', '5' ],
                '0,0' => {
                    guess => [ '0', '3', '3', '3' ],
                    '0,1' => { guess => [ '3', '4', '4', '4' ] },
                    '1,0' => { guess => [ '4', '3', '4', '4' ] },
                    '1,1' => { guess => [ '3', '3', '4', '4' ] },
                    '2,0' => { guess => [ '4', '4', '3', '3' ] },
                    '2,1' => { guess => [ '3', '4', '3', '3' ] },
                    '3,0' => { guess => [ '4', '3', '3', '3' ] }
                },
                '0,1' => {
                    guess => [ '2', '5', '3', '3' ],
                    '0,1' => { guess => [ '4', '4', '2', '4' ] },
                    '0,2' => { guess => [ '5', '3', '4', '4' ] },
                    '1,1' => { guess => [ '3', '5', '4', '4' ] },
                    '1,2' => { guess => [ '3', '3', '3', '2' ] },
                    '2,1' => { guess => [ '5', '4', '3', '3' ] },
                    '3,0' => { guess => [ '4', '5', '3', '3' ] }
                },
                '0,2' => {
                    guess => [ '0', '5', '0', '3' ],
                    '0,1' => { guess => [ '5', '4', '2', '4' ] },
                    '0,2' => { guess => [ '5', '3', '3', '2' ] },
                    '1,0' => { guess => [ '4', '5', '2', '4' ] },
                    '1,1' => { guess => [ '3', '5', '3', '2' ] }
                },
                '0,3' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '5', '5', '2', '4' ] },
                    '1,0' => { guess => [ '5', '5', '3', '2' ] }
                },
                '1,0' => {
                    guess => [ '0', '3', '3', '5' ],
                    '0,2' => { guess => [ '3', '4', '5', '4' ] },
                    '1,1' => { guess => [ '4', '3', '5', '4' ] },
                    '1,2' => { guess => [ '3', '3', '5', '4' ] },
                    '2,0' => { guess => [ '4', '4', '3', '5' ] },
                    '2,1' => { guess => [ '3', '4', '3', '5' ] },
                    '3,0' => { guess => [ '4', '3', '3', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '5', '3', '5' ],
                    '0,3' => { guess => [ '5', '3', '5', '4' ] },
                    '1,2' => { guess => [ '3', '5', '5', '4' ] },
                    '2,1' => { guess => [ '5', '4', '3', '5' ] },
                    '3,0' => { guess => [ '4', '5', '3', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '2', '0', '0' ],
                    '0,1' => { guess => [ '2', '5', '2', '2' ] },
                    '1,0' => { guess => [ '5', '2', '2', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '2', '0', '3' ],
                    '0,1' => { guess => [ '2', '4', '4', '5' ] },
                    '1,0' => { guess => [ '4', '2', '4', '5' ] },
                    '1,1' => { guess => [ '2', '3', '5', '3' ] },
                    '2,0' => { guess => [ '3', '2', '5', '3' ] }
                },
                '2,1' => {
                    guess => [ '0', '2', '0', '3' ],
                    '0,1' => { guess => [ '2', '5', '4', '5' ] },
                    '1,0' => { guess => [ '5', '2', '4', '5' ] },
                    '1,1' => { guess => [ '2', '5', '5', '3' ] },
                    '2,0' => { guess => [ '5', '2', '5', '3' ] }
                },
                '2,2' => {
                    guess => [ '0', '2', '2', '5' ],
                    '0,3' => { guess => [ '2', '5', '5', '2' ] },
                    '1,2' => { guess => [ '5', '2', '5', '2' ] },
                    '2,1' => { guess => [ '2', '5', '2', '5' ] },
                    '3,0' => { guess => [ '5', '2', '2', '5' ] }
                },
                '3,0' => {
                    guess => [ '0', '2', '0', '3' ],
                    '0,1' => { guess => [ '2', '4', '5', '5' ] },
                    '0,2' => { guess => [ '2', '3', '5', '5' ] },
                    '1,0' => { guess => [ '4', '2', '5', '5' ] },
                    '1,1' => { guess => [ '3', '2', '5', '5' ] }
                }
            },
            '1,2' => {
                guess => [ '2', '3', '4', '3' ],
                '0,2' => {
                    guess => [ '0', '3', '2', '5' ],
                    '0,2' => { guess => [ '4', '2', '5', '2' ] },
                    '0,3' => { guess => [ '3', '2', '5', '2' ] },
                    '1,0' => { guess => [ '4', '2', '2', '2' ] },
                    '1,1' => { guess => [ '3', '2', '2', '2' ] },
                    '2,0' => { guess => [ '4', '2', '2', '5' ] },
                    '2,1' => { guess => [ '3', '2', '2', '5' ] }
                },
                '0,3' => {
                    guess => [ '0', '4', '2', '4' ],
                    '0,2' => { guess => [ '4', '5', '3', '2' ] },
                    '1,1' => { guess => [ '5', '4', '3', '2' ] },
                    '1,2' => { guess => [ '4', '4', '3', '2' ] },
                    '2,0' => { guess => [ '3', '5', '2', '4' ] },
                    '3,0' => { guess => [ '3', '4', '2', '4' ] }
                },
                '0,4' => { guess => [ '3', '4', '3', '2' ] },
                '1,1' => {
                    guess => [ '0', '2', '4', '5' ],
                    '0,2' => { guess => [ '2', '4', '2', '2' ] },
                    '0,3' => { guess => [ '2', '4', '5', '2' ] },
                    '1,1' => { guess => [ '5', '2', '2', '3' ] },
                    '1,2' => { guess => [ '2', '4', '2', '5' ] },
                    '2,0' => { guess => [ '4', '2', '4', '2' ] },
                    '2,1' => { guess => [ '5', '2', '4', '2' ] }
                },
                '1,2' => {
                    guess => [ '0', '2', '0', '5' ],
                    '0,1' => { guess => [ '4', '3', '2', '4' ] },
                    '0,2' => { guess => [ '5', '3', '2', '4' ] },
                    '1,0' => { guess => [ '3', '2', '2', '3' ] },
                    '1,1' => { guess => [ '4', '2', '5', '3' ] },
                    '2,0' => { guess => [ '3', '2', '4', '5' ] }
                },
                '1,3' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '3', '3', '2', '4' ] },
                    '1,0' => { guess => [ '4', '3', '3', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '3', '0', '5' ],
                    '0,0' => { guess => [ '2', '4', '4', '2' ] },
                    '0,1' => { guess => [ '2', '5', '4', '2' ] },
                    '0,2' => { guess => [ '2', '5', '2', '3' ] },
                    '1,0' => { guess => [ '2', '3', '2', '2' ] },
                    '1,1' => { guess => [ '2', '3', '5', '2' ] },
                    '2,0' => { guess => [ '2', '3', '2', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '4', '4' ],
                    '0,1' => { guess => [ '2', '4', '5', '3' ] },
                    '1,0' => { guess => [ '5', '2', '4', '3' ] },
                    '1,1' => { guess => [ '4', '2', '4', '3' ] }
                },
                '2,2' => { guess => [ '3', '2', '4', '3' ] },
                '3,0' => {
                    guess => [ '0', '0', '2', '5' ],
                    '0,1' => { guess => [ '2', '4', '4', '3' ] },
                    '0,2' => { guess => [ '2', '5', '4', '3' ] },
                    '1,0' => { guess => [ '2', '3', '2', '3' ] },
                    '1,1' => { guess => [ '2', '3', '4', '5' ] }
                }
            },
            '1,3' => {
                guess => [ '0', '2', '2', '3' ],
                '0,3' => { guess => [ '2', '3', '4', '2' ] },
                '1,2' => { guess => [ '3', '2', '4', '2' ] },
                '2,1' => { guess => [ '2', '4', '2', '3' ] },
                '3,0' => { guess => [ '4', '2', '2', '3' ] }
            },
            '2,0' => {
                guess => [ '2', '5', '2', '5' ],
                '0,0' => {
                    guess => [ '0', '3', '3', '3' ],
                    '1,0' => { guess => [ '4', '4', '3', '4' ] },
                    '1,1' => { guess => [ '3', '4', '3', '4' ] },
                    '2,0' => { guess => [ '4', '3', '3', '4' ] },
                    '2,1' => { guess => [ '3', '3', '3', '4' ] }
                },
                '0,1' => {
                    guess => [ '0', '0', '3', '3' ],
                    '0,0' => { guess => [ '4', '2', '4', '4' ] },
                    '1,0' => { guess => [ '5', '4', '3', '4' ] },
                    '1,1' => { guess => [ '5', '3', '3', '4' ] },
                    '2,0' => { guess => [ '3', '2', '3', '3' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '4', '0' ],
                    '0,0' => { guess => [ '5', '2', '3', '3' ] },
                    '0,1' => { guess => [ '4', '2', '5', '4' ] },
                    '1,0' => { guess => [ '5', '2', '4', '4' ] }
                },
                '0,3' => { guess => [ '5', '2', '5', '4' ] },
                '1,0' => {
                    guess => [ '0', '0', '3', '3' ],
                    '0,0' => { guess => [ '2', '4', '4', '4' ] },
                    '1,0' => { guess => [ '4', '5', '3', '4' ] },
                    '1,1' => { guess => [ '3', '5', '3', '4' ] },
                    '2,0' => { guess => [ '2', '3', '3', '3' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '2', '4' ],
                    '0,1' => { guess => [ '3', '2', '3', '5' ] },
                    '1,0' => { guess => [ '5', '5', '3', '4' ] },
                    '1,1' => { guess => [ '2', '4', '5', '4' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '5', '2', '3', '5' ] },
                    '1,0' => { guess => [ '2', '2', '5', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '2', '3' ],
                    '0,1' => { guess => [ '2', '5', '4', '4' ] },
                    '0,2' => { guess => [ '2', '3', '3', '5' ] },
                    '1,0' => { guess => [ '2', '2', '2', '2' ] },
                    '1,1' => { guess => [ '2', '5', '3', '3' ] }
                },
                '2,1' => { guess => [ '2', '5', '5', '4' ] },
                '2,2' => { guess => [ '2', '2', '5', '5' ] },
                '3,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,0' => { guess => [ '2', '2', '2', '5' ] },
                    '0,1' => { guess => [ '2', '5', '3', '5' ] }
                }
            },
            '2,1' => {
                guess => [ '0', '2', '3', '3' ],
                '0,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '2', '4', '2', '4' ] },
                    '0,1' => { guess => [ '2', '5', '2', '4' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '2', '3', '4', '4' ] },
                    '0,1' => { guess => [ '2', '3', '5', '4' ] }
                },
                '1,0' => {
                    guess => [ '0', '0', '4', '5' ],
                    '0,1' => { guess => [ '4', '2', '2', '4' ] },
                    '0,2' => { guess => [ '5', '2', '2', '4' ] },
                    '1,0' => { guess => [ '2', '2', '4', '2' ] },
                    '2,0' => { guess => [ '2', '2', '4', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '3', '5' ],
                    '0,1' => { guess => [ '3', '2', '4', '4' ] },
                    '0,2' => { guess => [ '3', '2', '5', '4' ] },
                    '1,1' => { guess => [ '2', '5', '3', '2' ] },
                    '2,0' => { guess => [ '2', '4', '3', '5' ] }
                },
                '1,2' => { guess => [ '2', '3', '3', '2' ] },
                '2,0' => {
                    guess => [ '0', '0', '3', '5' ],
                    '0,1' => { guess => [ '2', '2', '2', '3' ] },
                    '0,2' => { guess => [ '2', '2', '5', '3' ] },
                    '1,1' => { guess => [ '5', '2', '3', '2' ] },
                    '2,0' => { guess => [ '4', '2', '3', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '2', '4', '3', '3' ] },
                    '1,0' => { guess => [ '3', '2', '3', '2' ] }
                },
                '3,0' => { guess => [ '4', '2', '3', '3' ] }
            },
            '2,2' => {
                guess => [ '0', '2', '0', '3' ],
                '0,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '2', '3', '2', '4' ] },
                    '1,0' => { guess => [ '2', '4', '3', '2' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '3', '2', '2', '4' ] },
                    '1,0' => { guess => [ '4', '2', '3', '2' ] }
                },
                '2,0' => { guess => [ '2', '2', '4', '3' ] }
            },
            '3,0' => {
                guess => [ '2', '3', '3', '5' ],
                '1,0' => {
                    guess => [ '0', '0', '2', '0' ],
                    '0,1' => { guess => [ '2', '2', '4', '4' ] },
                    '1,0' => { guess => [ '2', '2', '2', '4' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,0' => { guess => [ '2', '2', '5', '4' ] },
                    '0,1' => { guess => [ '4', '2', '3', '4' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '3', '2', '3', '4' ] },
                    '0,1' => { guess => [ '5', '2', '3', '4' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '2', '4', '3', '4' ] },
                    '1,0' => { guess => [ '2', '2', '3', '2' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,1' => { guess => [ '2', '5', '3', '4' ] },
                    '1,0' => { guess => [ '2', '2', '3', '3' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '0,0' => { guess => [ '2', '2', '3', '5' ] },
                    '1,0' => { guess => [ '2', '3', '3', '4' ] }
                }
            }
        },
        '0,1' => {
            guess => [ '1', '2', '3', '3' ],
            '0,0' => {
                guess => [ '0', '4', '0', '4' ],
                '0,1' => { guess => [ '5', '5', '5', '0' ] },
                '0,2' => {
                    guess => [ '0', '0', '4', '0' ],
                    '1,1' => { guess => [ '4', '5', '5', '0' ] },
                    '2,0' => { guess => [ '5', '5', '4', '0' ] }
                },
                '0,3' => { guess => [ '4', '5', '4', '0' ] },
                '1,0' => { guess => [ '5', '5', '0', '5' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '5', '4', '5', '0' ] },
                    '1,0' => { guess => [ '4', '5', '0', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '4', '5' ],
                    '0,3' => { guess => [ '4', '4', '5', '0' ] },
                    '1,1' => { guess => [ '4', '4', '4', '0' ] },
                    '1,2' => { guess => [ '5', '4', '4', '0' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '1,1' => { guess => [ '5', '4', '0', '5' ] },
                    '2,0' => { guess => [ '5', '5', '0', '4' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '4' ],
                    '1,1' => { guess => [ '4', '4', '0', '5' ] },
                    '2,0' => { guess => [ '4', '5', '0', '4' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '4', '4', '0', '4' ] },
                    '1,1' => { guess => [ '5', '4', '0', '4' ] }
                }
            },
            '0,1' => {
                guess => [ '1', '4', '0', '4' ],
                '0,1' => {
                    guess => [ '0', '3', '5', '5' ],
                    '0,2' => { guess => [ '2', '5', '2', '0' ] },
                    '0,3' => { guess => [ '5', '5', '2', '0' ] },
                    '1,2' => { guess => [ '2', '5', '5', '0' ] },
                    '1,3' => { guess => [ '3', '5', '5', '0' ] },
                    '2,0' => { guess => [ '5', '1', '5', '5' ] },
                    '2,2' => { guess => [ '5', '3', '5', '0' ] }
                },
                '0,2' => {
                    guess => [ '0', '5', '4', '2' ],
                    '0,2' => { guess => [ '4', '1', '5', '5' ] },
                    '0,3' => { guess => [ '4', '3', '5', '0' ] },
                    '1,1' => { guess => [ '5', '1', '4', '5' ] },
                    '1,2' => { guess => [ '5', '3', '4', '0' ] },
                    '1,3' => { guess => [ '4', '5', '2', '0' ] },
                    '2,1' => { guess => [ '3', '5', '4', '0' ] },
                    '2,2' => { guess => [ '2', '5', '4', '0' ] }
                },
                '0,3' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '4', '1', '4', '5' ] },
                    '1,0' => { guess => [ '4', '3', '4', '0' ] }
                },
                '1,0' => {
                    guess => [ '0', '3', '5', '5' ],
                    '0,2' => { guess => [ '2', '5', '0', '2' ] },
                    '0,3' => { guess => [ '5', '5', '0', '2' ] },
                    '1,2' => { guess => [ '2', '5', '0', '5' ] },
                    '1,3' => { guess => [ '3', '5', '0', '5' ] },
                    '2,2' => { guess => [ '5', '3', '0', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '2', '2', '0' ],
                    '0,0' => { guess => [ '5', '1', '5', '4' ] },
                    '0,1' => { guess => [ '4', '3', '0', '5' ] },
                    '0,2' => { guess => [ '4', '5', '0', '2' ] },
                    '1,0' => { guess => [ '3', '4', '5', '0' ] },
                    '1,1' => { guess => [ '2', '4', '5', '0' ] },
                    '2,0' => { guess => [ '5', '4', '2', '0' ] },
                    '2,1' => { guess => [ '2', '4', '2', '0' ] }
                },
                '1,2' => {
                    guess => [ '2', '0', '5', '4' ],
                    '0,2' => { guess => [ '3', '4', '4', '0' ] },
                    '0,3' => { guess => [ '4', '4', '2', '0' ] },
                    '1,0' => { guess => [ '4', '1', '4', '4' ] },
                    '1,1' => { guess => [ '5', '1', '4', '4' ] },
                    '1,2' => { guess => [ '2', '4', '4', '0' ] },
                    '2,0' => { guess => [ '4', '1', '5', '4' ] }
                },
                '2,0' => {
                    guess => [ '3', '4', '0', '5' ],
                    '1,2' => { guess => [ '2', '5', '0', '4' ] },
                    '1,3' => { guess => [ '5', '3', '0', '4' ] },
                    '2,0' => { guess => [ '2', '4', '0', '2' ] },
                    '2,1' => { guess => [ '5', '4', '0', '2' ] },
                    '2,2' => { guess => [ '3', '5', '0', '4' ] },
                    '3,0' => { guess => [ '2', '4', '0', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,0' => { guess => [ '4', '3', '0', '4' ] },
                    '2,0' => { guess => [ '4', '4', '0', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,0' => { guess => [ '3', '4', '0', '4' ] },
                    '1,1' => { guess => [ '2', '4', '0', '4' ] }
                }
            },
            '0,2' => {
                guess => [ '2', '1', '2', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '3', '3', '5', '0' ] },
                    '1,0' => { guess => [ '3', '3', '0', '5' ] }
                },
                '0,1' => {
                    guess => [ '0', '3', '0', '2' ],
                    '1,1' => { guess => [ '3', '3', '4', '0' ] },
                    '2,1' => { guess => [ '3', '5', '0', '2' ] },
                    '3,0' => { guess => [ '5', '3', '0', '2' ] }
                },
                '0,2' => {
                    guess => [ '0', '3', '0', '0' ],
                    '1,1' => { guess => [ '3', '4', '0', '2' ] },
                    '2,0' => { guess => [ '4', '3', '0', '2' ] }
                },
                '1,0' => {
                    guess => [ '1', '5', '2', '0' ],
                    '0,1' => { guess => [ '3', '3', '0', '4' ] },
                    '0,2' => { guess => [ '3', '1', '5', '5' ] },
                    '0,3' => { guess => [ '2', '3', '0', '5' ] },
                    '1,2' => { guess => [ '2', '3', '5', '0' ] },
                    '2,1' => { guess => [ '5', '3', '2', '0' ] },
                    '3,0' => { guess => [ '3', '5', '2', '0' ] }
                },
                '1,1' => {
                    guess => [ '0', '3', '4', '2' ],
                    '0,4' => { guess => [ '3', '4', '2', '0' ] },
                    '1,0' => { guess => [ '5', '1', '5', '2' ] },
                    '1,1' => { guess => [ '3', '1', '4', '5' ] },
                    '1,3' => { guess => [ '4', '3', '2', '0' ] },
                    '2,1' => { guess => [ '2', '3', '0', '2' ] },
                    '2,2' => { guess => [ '2', '3', '4', '0' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '4', '4' ],
                    '0,1' => { guess => [ '4', '1', '5', '2' ] },
                    '1,0' => { guess => [ '5', '1', '4', '2' ] },
                    '1,1' => { guess => [ '4', '1', '4', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '2', '2', '5' ],
                    '0,0' => { guess => [ '3', '1', '4', '4' ] },
                    '0,1' => { guess => [ '3', '1', '5', '4' ] },
                    '0,2' => { guess => [ '2', '3', '0', '4' ] },
                    '1,1' => { guess => [ '2', '1', '5', '5' ] },
                    '1,2' => { guess => [ '2', '3', '2', '0' ] },
                    '2,0' => { guess => [ '5', '1', '2', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '2', '2' ],
                    '0,1' => { guess => [ '2', '1', '4', '5' ] },
                    '1,0' => { guess => [ '4', '1', '2', '5' ] },
                    '1,1' => { guess => [ '2', '1', '5', '2' ] },
                    '2,0' => { guess => [ '5', '1', '2', '2' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '2', '0' ],
                    '0,1' => { guess => [ '2', '1', '4', '2' ] },
                    '1,0' => { guess => [ '4', '1', '2', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '4', '2', '5' ],
                    '0,2' => { guess => [ '2', '1', '4', '4' ] },
                    '0,3' => { guess => [ '2', '1', '5', '4' ] },
                    '1,0' => { guess => [ '2', '1', '2', '2' ] },
                    '1,1' => { guess => [ '4', '1', '2', '4' ] },
                    '1,2' => { guess => [ '5', '1', '2', '4' ] },
                    '2,0' => { guess => [ '2', '1', '2', '5' ] }
                }
            },
            '0,3' => {
                guess => [ '1', '2', '2', '4' ],
                '0,1' => { guess => [ '3', '3', '0', '2' ] },
                '0,2' => { guess => [ '3', '1', '5', '2' ] },
                '0,3' => { guess => [ '3', '1', '4', '2' ] },
                '1,0' => { guess => [ '3', '3', '2', '0' ] },
                '1,1' => { guess => [ '3', '1', '2', '5' ] },
                '1,2' => { guess => [ '3', '1', '2', '2' ] },
                '2,1' => { guess => [ '3', '1', '2', '4' ] }
            },
            '1,0' => {
                guess => [ '0', '4', '0', '2' ],
                '0,0' => { guess => [ '1', '5', '5', '5' ] },
                '0,1' => {
                    guess => [ '0', '1', '4', '5' ],
                    '0,2' => { guess => [ '5', '5', '3', '0' ] },
                    '0,3' => { guess => [ '1', '5', '5', '4' ] },
                    '1,2' => { guess => [ '1', '5', '4', '4' ] },
                    '2,1' => { guess => [ '1', '5', '4', '5' ] }
                },
                '0,2' => {
                    guess => [ '0', '2', '2', '2' ],
                    '0,1' => { guess => [ '4', '5', '3', '0' ] },
                    '1,1' => { guess => [ '5', '2', '5', '0' ] },
                    '1,2' => { guess => [ '2', '2', '5', '0' ] },
                    '2,1' => { guess => [ '5', '2', '2', '0' ] },
                    '2,2' => { guess => [ '2', '2', '2', '0' ] }
                },
                '0,3' => {
                    guess => [ '2', '0', '4', '5' ],
                    '0,3' => { guess => [ '4', '2', '2', '0' ] },
                    '0,4' => { guess => [ '4', '2', '5', '0' ] },
                    '1,2' => { guess => [ '4', '2', '4', '0' ] },
                    '1,3' => { guess => [ '5', '2', '4', '0' ] },
                    '2,1' => { guess => [ '2', '2', '4', '0' ] }
                },
                '1,0' => {
                    guess => [ '0', '1', '4', '5' ],
                    '0,2' => { guess => [ '5', '5', '0', '3' ] },
                    '0,3' => { guess => [ '1', '4', '5', '4' ] },
                    '1,1' => { guess => [ '1', '4', '4', '4' ] },
                    '1,2' => { guess => [ '1', '4', '5', '5' ] },
                    '2,1' => { guess => [ '1', '4', '4', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '5', '2', '2' ],
                    '0,1' => { guess => [ '4', '4', '3', '0' ] },
                    '0,2' => { guess => [ '5', '4', '3', '0' ] },
                    '0,3' => { guess => [ '5', '2', '0', '5' ] },
                    '0,4' => { guess => [ '2', '2', '0', '5' ] },
                    '1,1' => { guess => [ '4', '5', '0', '3' ] }
                },
                '1,2' => {
                    guess => [ '2', '0', '0', '5' ],
                    '1,1' => { guess => [ '4', '2', '0', '4' ] },
                    '1,2' => { guess => [ '5', '2', '0', '4' ] },
                    '2,0' => { guess => [ '2', '2', '0', '4' ] },
                    '2,1' => { guess => [ '4', '2', '0', '5' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '5', '2' ],
                    '0,1' => { guess => [ '4', '4', '0', '3' ] },
                    '0,2' => { guess => [ '5', '4', '0', '3' ] },
                    '1,1' => { guess => [ '2', '2', '0', '2' ] },
                    '1,2' => { guess => [ '5', '2', '0', '2' ] }
                },
                '2,1' => { guess => [ '4', '2', '0', '2' ] }
            },
            '1,1' => {
                guess => [ '3', '3', '0', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '2', '5' ],
                    '0,2' => { guess => [ '1', '5', '5', '2' ] },
                    '1,1' => { guess => [ '1', '5', '2', '2' ] },
                    '2,0' => { guess => [ '1', '5', '2', '5' ] }
                },
                '0,1' => {
                    guess => [ '0', '4', '2', '5' ],
                    '0,1' => { guess => [ '5', '1', '5', '3' ] },
                    '0,3' => { guess => [ '1', '5', '4', '2' ] },
                    '1,0' => { guess => [ '5', '1', '3', '5' ] },
                    '1,1' => { guess => [ '1', '4', '4', '2' ] },
                    '1,2' => { guess => [ '1', '4', '5', '2' ] },
                    '2,0' => { guess => [ '1', '4', '2', '2' ] },
                    '3,0' => { guess => [ '1', '4', '2', '5' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '5', '3' ],
                    '0,2' => { guess => [ '4', '1', '3', '5' ] },
                    '0,3' => { guess => [ '2', '5', '3', '0' ] },
                    '1,0' => { guess => [ '4', '1', '4', '3' ] },
                    '1,1' => { guess => [ '5', '1', '4', '3' ] },
                    '2,0' => { guess => [ '4', '1', '5', '3' ] }
                },
                '0,3' => { guess => [ '2', '4', '3', '0' ] },
                '1,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '4', '2', '4' ] },
                    '0,1' => { guess => [ '1', '5', '2', '4' ] },
                    '1,0' => { guess => [ '1', '3', '5', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '1', '5', '4' ],
                    '0,1' => { guess => [ '3', '2', '2', '0' ] },
                    '0,2' => { guess => [ '2', '5', '0', '3' ] },
                    '0,3' => { guess => [ '1', '3', '4', '5' ] },
                    '1,1' => { guess => [ '3', '2', '5', '0' ] },
                    '2,0' => { guess => [ '4', '1', '3', '4' ] },
                    '2,1' => { guess => [ '5', '1', '3', '4' ] }
                },
                '1,2' => {
                    guess => [ '0', '3', '0', '2' ],
                    '0,2' => { guess => [ '3', '5', '3', '0' ] },
                    '0,3' => { guess => [ '3', '2', '4', '0' ] },
                    '1,1' => { guess => [ '5', '3', '3', '0' ] },
                    '1,2' => { guess => [ '2', '4', '0', '3' ] }
                },
                '1,3' => {
                    guess => [ '0', '3', '0', '0' ],
                    '1,1' => { guess => [ '3', '4', '3', '0' ] },
                    '2,0' => { guess => [ '4', '3', '3', '0' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '3', '4', '4' ] },
                    '0,1' => { guess => [ '1', '3', '5', '4' ] },
                    '1,0' => { guess => [ '3', '2', '0', '2' ] },
                    '2,0' => { guess => [ '3', '2', '0', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '3', '0', '3' ],
                    '1,2' => { guess => [ '3', '3', '3', '0' ] },
                    '2,1' => { guess => [ '3', '5', '0', '3' ] },
                    '3,0' => { guess => [ '5', '3', '0', '3' ] }
                },
                '2,2' => {
                    guess => [ '0', '3', '0', '0' ],
                    '1,1' => { guess => [ '3', '4', '0', '3' ] },
                    '2,0' => { guess => [ '4', '3', '0', '3' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,0' => { guess => [ '3', '3', '0', '3' ] },
                    '1,1' => { guess => [ '3', '2', '0', '4' ] }
                }
            },
            '1,2' => {
                guess => [ '2', '1', '3', '4' ],
                '0,3' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '3', '2', '2' ] },
                    '0,1' => { guess => [ '1', '3', '5', '2' ] },
                    '1,0' => { guess => [ '1', '3', '2', '5' ] }
                },
                '0,4' => { guess => [ '1', '3', '4', '2' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '3', '1', '5', '3' ] },
                    '1,0' => { guess => [ '2', '3', '0', '3' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,0' => { guess => [ '3', '1', '4', '3' ] },
                    '0,1' => { guess => [ '5', '1', '2', '3' ] }
                },
                '1,3' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,1' => { guess => [ '1', '3', '2', '4' ] },
                    '1,0' => { guess => [ '4', '1', '2', '3' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '3', '1', '3', '5' ] },
                    '1,0' => { guess => [ '2', '3', '3', '0' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '2', '2' ],
                    '0,1' => { guess => [ '2', '1', '5', '3' ] },
                    '1,0' => { guess => [ '5', '1', '3', '2' ] },
                    '1,1' => { guess => [ '2', '1', '2', '3' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '2', '1', '4', '3' ] },
                    '1,0' => { guess => [ '4', '1', '3', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,0' => { guess => [ '3', '1', '3', '4' ] },
                    '0,1' => { guess => [ '2', '1', '3', '5' ] },
                    '1,0' => { guess => [ '2', '1', '3', '2' ] }
                }
            },
            '1,3' => {
                guess => [ '0', '0', '0', '2' ],
                '0,1' => { guess => [ '3', '1', '2', '3' ] },
                '1,0' => { guess => [ '3', '1', '3', '2' ] }
            },
            '2,0' => {
                guess => [ '0', '4', '3', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '2', '5' ],
                    '0,2' => { guess => [ '1', '2', '5', '2' ] },
                    '1,0' => { guess => [ '1', '2', '2', '2' ] },
                    '1,1' => { guess => [ '1', '2', '5', '5' ] },
                    '2,0' => { guess => [ '1', '2', '2', '5' ] }
                },
                '0,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,0' => { guess => [ '1', '5', '5', '3' ] },
                    '0,1' => { guess => [ '1', '2', '4', '5' ] },
                    '1,0' => { guess => [ '1', '2', '4', '2' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,1' => { guess => [ '1', '5', '4', '3' ] },
                    '1,0' => { guess => [ '2', '2', '0', '3' ] },
                    '1,1' => { guess => [ '5', '2', '0', '3' ] }
                },
                '0,3' => { guess => [ '4', '2', '0', '3' ] },
                '1,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '2', '2', '4' ] },
                    '0,1' => { guess => [ '1', '2', '5', '4' ] },
                    '1,0' => { guess => [ '1', '5', '3', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '2', '2' ],
                    '0,0' => { guess => [ '1', '4', '5', '3' ] },
                    '0,1' => { guess => [ '1', '2', '4', '4' ] },
                    '0,2' => { guess => [ '5', '2', '3', '0' ] },
                    '0,3' => { guess => [ '2', '2', '3', '0' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '1', '4', '4', '3' ] },
                    '1,0' => { guess => [ '4', '2', '3', '0' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '0,1' => { guess => [ '1', '4', '3', '5' ] },
                    '1,0' => { guess => [ '1', '5', '3', '4' ] }
                },
                '3,0' => { guess => [ '1', '4', '3', '4' ] }
            },
            '2,1' => {
                guess => [ '1', '3', '1', '4' ],
                '0,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '3', '2', '3', '0' ] },
                    '1,0' => { guess => [ '3', '2', '0', '3' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '3', '1', '3', '3' ] },
                    '0,1' => { guess => [ '5', '1', '3', '3' ] }
                },
                '0,3' => { guess => [ '4', '1', '3', '3' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '1', '5', '2', '3' ] },
                    '1,0' => { guess => [ '1', '5', '3', '2' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '1', '4', '2', '3' ] },
                    '1,0' => { guess => [ '1', '4', '3', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,1' => { guess => [ '1', '3', '3', '5' ] },
                    '1,0' => { guess => [ '1', '3', '5', '3' ] }
                },
                '2,1' => { guess => [ '1', '3', '4', '3' ] },
                '3,0' => { guess => [ '1', '3', '3', '4' ] }
            },
            '2,2' => {
                guess => [ '0', '0', '2', '3' ],
                '0,2' => { guess => [ '1', '3', '3', '2' ] },
                '1,1' => { guess => [ '2', '1', '3', '3' ] },
                '2,0' => { guess => [ '1', '3', '2', '3' ] }
            },
            '3,0' => {
                guess => [ '0', '2', '2', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '3', '3', '3' ] },
                    '0,1' => { guess => [ '1', '5', '3', '3' ] }
                },
                '0,1' => { guess => [ '1', '4', '3', '3' ] },
                '1,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,1' => { guess => [ '1', '2', '3', '5' ] },
                    '1,0' => { guess => [ '1', '2', '5', '3' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '1', '2', '4', '3' ] },
                    '1,0' => { guess => [ '1', '2', '3', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,1' => { guess => [ '1', '2', '3', '4' ] },
                    '1,0' => { guess => [ '1', '2', '2', '3' ] }
                }
            }
        },
        '0,2' => {
            guess => [ '1', '2', '3', '3' ],
            '0,0' => {
                guess => [ '0', '4', '0', '4' ],
                '1,1' => { guess => [ '5', '5', '0', '0' ] },
                '1,2' => { guess => [ '4', '5', '0', '0' ] },
                '2,1' => { guess => [ '5', '4', '0', '0' ] },
                '2,2' => { guess => [ '4', '4', '0', '0' ] }
            },
            '0,1' => {
                guess => [ '0', '1', '4', '4' ],
                '0,1' => {
                    guess => [ '0', '3', '0', '0' ],
                    '2,0' => { guess => [ '2', '5', '0', '0' ] },
                    '2,1' => { guess => [ '3', '5', '0', '0' ] },
                    '3,0' => { guess => [ '5', '3', '0', '0' ] }
                },
                '0,2' => {
                    guess => [ '0', '3', '0', '0' ],
                    '2,0' => { guess => [ '2', '4', '0', '0' ] },
                    '2,1' => { guess => [ '3', '4', '0', '0' ] },
                    '3,0' => { guess => [ '4', '3', '0', '0' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '5', '1', '5', '0' ] },
                    '1,1' => { guess => [ '5', '1', '0', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '4', '1', '5', '0' ] },
                    '1,1' => { guess => [ '4', '1', '0', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '5', '1', '4', '0' ] },
                    '1,1' => { guess => [ '5', '1', '0', '4' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '4', '1', '4', '0' ] },
                    '1,1' => { guess => [ '4', '1', '0', '4' ] }
                }
            },
            '0,2' => {
                guess => [ '0', '3', '0', '4' ],
                '0,1' => {
                    guess => [ '0', '0', '2', '2' ],
                    '0,2' => { guess => [ '2', '1', '5', '0' ] },
                    '1,1' => { guess => [ '5', '1', '2', '0' ] },
                    '1,2' => { guess => [ '2', '1', '2', '0' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '2', '0' ],
                    '1,0' => { guess => [ '3', '1', '5', '0' ] },
                    '1,1' => { guess => [ '2', '1', '4', '0' ] },
                    '2,0' => { guess => [ '4', '1', '2', '0' ] }
                },
                '0,3' => { guess => [ '3', '1', '4', '0' ] },
                '1,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '2', '1', '0', '2' ] },
                    '1,1' => { guess => [ '5', '1', '0', '2' ] },
                    '2,0' => { guess => [ '2', '1', '0', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,0' => { guess => [ '3', '1', '0', '5' ] },
                    '2,0' => { guess => [ '4', '1', '0', '2' ] }
                },
                '2,0' => { guess => [ '2', '1', '0', '4' ] },
                '2,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,0' => { guess => [ '3', '1', '0', '4' ] },
                    '1,1' => { guess => [ '3', '3', '0', '0' ] },
                    '1,2' => { guess => [ '2', '3', '0', '0' ] }
                }
            },
            '0,3' => {
                guess => [ '0', '0', '0', '1' ],
                '0,2' => { guess => [ '3', '1', '2', '0' ] },
                '1,1' => { guess => [ '3', '1', '0', '2' ] }
            },
            '1,0' => {
                guess => [ '1', '0', '4', '4' ],
                '0,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,1' => { guess => [ '2', '2', '0', '0' ] },
                    '1,2' => { guess => [ '5', '2', '0', '0' ] }
                },
                '0,2' => { guess => [ '4', '2', '0', '0' ] },
                '1,0' => { guess => [ '1', '1', '5', '5' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '1', '5', '5', '0' ] },
                    '1,1' => { guess => [ '1', '5', '0', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '1', '4', '5', '0' ] },
                    '1,1' => { guess => [ '1', '4', '0', '5' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '0,1' => { guess => [ '1', '1', '4', '5' ] },
                    '1,0' => { guess => [ '1', '1', '5', '4' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '1', '5', '4', '0' ] },
                    '1,1' => { guess => [ '1', '5', '0', '4' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '1', '4', '4', '0' ] },
                    '1,1' => { guess => [ '1', '4', '0', '4' ] }
                },
                '3,0' => { guess => [ '1', '1', '4', '4' ] }
            },
            '1,1' => {
                guess => [ '0', '3', '0', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '1', '2', '2' ] },
                    '0,1' => { guess => [ '1', '1', '5', '2' ] },
                    '1,0' => { guess => [ '1', '1', '2', '5' ] }
                },
                '0,1' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '1', '1', '4', '2' ] },
                    '1,0' => { guess => [ '1', '5', '2', '0' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '5', '1', '3', '0' ] },
                    '0,2' => { guess => [ '1', '4', '2', '0' ] }
                },
                '0,3' => { guess => [ '4', '1', '3', '0' ] },
                '1,0' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '1', '1', '2', '4' ] },
                    '1,0' => { guess => [ '1', '5', '0', '2' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '1', '3', '5', '0' ] },
                    '1,0' => { guess => [ '5', '1', '0', '3' ] },
                    '2,0' => { guess => [ '1', '4', '0', '2' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,1' => { guess => [ '1', '3', '4', '0' ] },
                    '1,0' => { guess => [ '4', '1', '0', '3' ] },
                    '1,2' => { guess => [ '3', '2', '0', '0' ] }
                },
                '2,0' => { guess => [ '1', '3', '0', '5' ] },
                '3,0' => { guess => [ '1', '3', '0', '4' ] }
            },
            '1,2' => {
                guess => [ '1', '3', '0', '2' ],
                '0,3' => { guess => [ '3', '1', '3', '0' ] },
                '0,4' => { guess => [ '2', '1', '3', '0' ] },
                '1,2' => { guess => [ '3', '1', '0', '3' ] },
                '1,3' => { guess => [ '2', '1', '0', '3' ] },
                '2,2' => { guess => [ '1', '3', '2', '0' ] }
            },
            '2,0' => {
                guess => [ '2', '2', '0', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,1' => { guess => [ '1', '1', '3', '5' ] },
                    '1,0' => { guess => [ '1', '1', '5', '3' ] }
                },
                '0,1' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '1', '1', '4', '3' ] },
                    '1,0' => { guess => [ '1', '5', '3', '0' ] }
                },
                '0,2' => { guess => [ '1', '4', '3', '0' ] },
                '1,0' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '1', '1', '3', '4' ] },
                    '1,0' => { guess => [ '1', '5', '0', '3' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,2' => { guess => [ '1', '2', '5', '0' ] },
                    '1,1' => { guess => [ '1', '4', '0', '3' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '4' ],
                    '0,1' => { guess => [ '1', '2', '2', '0' ] },
                    '0,2' => { guess => [ '1', '2', '4', '0' ] }
                },
                '2,0' => { guess => [ '1', '2', '0', '5' ] },
                '2,1' => { guess => [ '1', '2', '0', '2' ] },
                '3,0' => { guess => [ '1', '2', '0', '4' ] }
            },
            '2,1' => {
                guess => [ '0', '0', '0', '3' ],
                '0,1' => { guess => [ '1', '1', '3', '2' ] },
                '0,2' => { guess => [ '1', '3', '3', '0' ] },
                '1,0' => { guess => [ '1', '1', '2', '3' ] },
                '2,0' => { guess => [ '1', '3', '0', '3' ] }
            },
            '3,0' => {
                guess => [ '0', '0', '0', '1' ],
                '0,1' => { guess => [ '1', '1', '3', '3' ] },
                '0,2' => { guess => [ '1', '2', '3', '0' ] },
                '1,1' => { guess => [ '1', '2', '0', '3' ] }
            }
        },
        '0,3' => {
            guess => [ '0', '1', '0', '2' ],
            '1,1' => {
                guess => [ '0', '0', '3', '4' ],
                '0,1' => { guess => [ '1', '1', '5', '0' ] },
                '0,2' => { guess => [ '1', '1', '4', '0' ] },
                '1,1' => { guess => [ '1', '1', '3', '0' ] }
            },
            '1,2' => {
                guess => [ '0', '3', '0', '4' ],
                '0,1' => { guess => [ '1', '1', '2', '0' ] },
                '1,1' => { guess => [ '1', '5', '0', '0' ] },
                '1,2' => { guess => [ '1', '4', '0', '0' ] },
                '2,1' => { guess => [ '1', '3', '0', '0' ] }
            },
            '1,3' => { guess => [ '1', '2', '0', '0' ] },
            '2,0' => {
                guess => [ '0', '0', '3', '4' ],
                '0,1' => { guess => [ '1', '1', '0', '5' ] },
                '0,2' => { guess => [ '1', '1', '0', '3' ] },
                '1,1' => { guess => [ '1', '1', '0', '4' ] }
            },
            '2,1' => {
                guess => [ '3', '0', '0', '4' ],
                '1,1' => { guess => [ '5', '1', '0', '0' ] },
                '1,2' => { guess => [ '4', '1', '0', '0' ] },
                '2,1' => { guess => [ '3', '1', '0', '0' ] }
            },
            '2,2' => { guess => [ '2', '1', '0', '0' ] },
            '3,0' => { guess => [ '1', '1', '0', '2' ] }
        },
        '0,4' => { guess => [ '1', '1', '0', '0' ] },
        '1,0' => {
            guess => [ '0', '2', '3', '3' ],
            '0,0' => {
                guess => [ '0', '4', '1', '4' ],
                '0,1' => { guess => [ '5', '5', '5', '1' ] },
                '0,2' => {
                    guess => [ '0', '0', '4', '0' ],
                    '0,1' => { guess => [ '4', '5', '5', '1' ] },
                    '1,0' => { guess => [ '5', '5', '4', '1' ] }
                },
                '0,3' => { guess => [ '4', '5', '4', '1' ] },
                '1,0' => { guess => [ '5', '5', '1', '5' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '4', '5', '1', '5' ] },
                    '1,0' => { guess => [ '5', '4', '5', '1' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '4', '5' ],
                    '0,2' => { guess => [ '4', '4', '5', '1' ] },
                    '1,0' => { guess => [ '4', '4', '4', '1' ] },
                    '1,1' => { guess => [ '5', '4', '4', '1' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '0,1' => { guess => [ '5', '4', '1', '5' ] },
                    '1,0' => { guess => [ '5', '5', '1', '4' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '4' ],
                    '0,1' => { guess => [ '4', '4', '1', '5' ] },
                    '1,0' => { guess => [ '4', '5', '1', '4' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '4', '4', '1', '4' ] },
                    '0,1' => { guess => [ '5', '4', '1', '4' ] }
                }
            },
            '0,1' => {
                guess => [ '0', '4', '1', '4' ],
                '0,1' => {
                    guess => [ '0', '5', '2', '2' ],
                    '0,1' => { guess => [ '5', '3', '5', '1' ] },
                    '0,2' => { guess => [ '5', '0', '5', '5' ] },
                    '1,0' => { guess => [ '3', '5', '5', '1' ] },
                    '1,1' => { guess => [ '2', '5', '5', '1' ] },
                    '2,0' => { guess => [ '5', '5', '2', '1' ] },
                    '2,1' => { guess => [ '2', '5', '2', '1' ] }
                },
                '0,2' => {
                    guess => [ '1', '5', '4', '2' ],
                    '0,2' => { guess => [ '4', '0', '5', '5' ] },
                    '0,3' => { guess => [ '4', '3', '5', '1' ] },
                    '1,1' => { guess => [ '5', '0', '4', '5' ] },
                    '1,2' => { guess => [ '5', '3', '4', '1' ] },
                    '1,3' => { guess => [ '4', '5', '2', '1' ] },
                    '2,1' => { guess => [ '3', '5', '4', '1' ] },
                    '2,2' => { guess => [ '2', '5', '4', '1' ] }
                },
                '0,3' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '4', '3', '4', '1' ] },
                    '1,0' => { guess => [ '4', '0', '4', '5' ] }
                },
                '1,0' => {
                    guess => [ '0', '3', '5', '5' ],
                    '0,1' => { guess => [ '2', '5', '1', '2' ] },
                    '0,2' => { guess => [ '5', '5', '1', '2' ] },
                    '1,1' => { guess => [ '2', '5', '1', '5' ] },
                    '1,2' => { guess => [ '3', '5', '1', '5' ] },
                    '2,1' => { guess => [ '5', '3', '1', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '4', '5', '3' ],
                    '0,2' => { guess => [ '4', '5', '1', '2' ] },
                    '0,3' => { guess => [ '4', '3', '1', '5' ] },
                    '1,0' => { guess => [ '2', '4', '2', '1' ] },
                    '1,1' => { guess => [ '5', '4', '2', '1' ] },
                    '1,2' => { guess => [ '5', '0', '5', '4' ] },
                    '2,0' => { guess => [ '2', '4', '5', '1' ] },
                    '2,1' => { guess => [ '3', '4', '5', '1' ] }
                },
                '1,2' => {
                    guess => [ '2', '0', '5', '4' ],
                    '0,1' => { guess => [ '3', '4', '4', '1' ] },
                    '0,2' => { guess => [ '4', '4', '2', '1' ] },
                    '1,1' => { guess => [ '2', '4', '4', '1' ] },
                    '2,0' => { guess => [ '4', '0', '4', '4' ] },
                    '2,1' => { guess => [ '5', '0', '4', '4' ] },
                    '3,0' => { guess => [ '4', '0', '5', '4' ] }
                },
                '2,0' => {
                    guess => [ '3', '4', '0', '5' ],
                    '0,2' => { guess => [ '2', '5', '1', '4' ] },
                    '0,3' => { guess => [ '5', '3', '1', '4' ] },
                    '1,0' => { guess => [ '2', '4', '1', '2' ] },
                    '1,1' => { guess => [ '5', '4', '1', '2' ] },
                    '1,2' => { guess => [ '3', '5', '1', '4' ] },
                    '2,0' => { guess => [ '2', '4', '1', '5' ] },
                    '3,0' => { guess => [ '3', '4', '1', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,0' => { guess => [ '4', '3', '1', '4' ] },
                    '1,0' => { guess => [ '4', '4', '1', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,0' => { guess => [ '3', '4', '1', '4' ] },
                    '0,1' => { guess => [ '2', '4', '1', '4' ] }
                }
            },
            '0,2' => {
                guess => [ '2', '0', '2', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '3', '3', '1', '5' ] },
                    '1,0' => { guess => [ '3', '3', '5', '1' ] }
                },
                '0,1' => {
                    guess => [ '0', '3', '0', '1' ],
                    '0,2' => { guess => [ '3', '5', '1', '2' ] },
                    '1,1' => { guess => [ '5', '3', '1', '2' ] },
                    '2,0' => { guess => [ '3', '3', '4', '1' ] }
                },
                '0,2' => {
                    guess => [ '0', '3', '0', '0' ],
                    '0,1' => { guess => [ '3', '4', '1', '2' ] },
                    '1,0' => { guess => [ '4', '3', '1', '2' ] }
                },
                '1,0' => {
                    guess => [ '0', '5', '2', '1' ],
                    '0,1' => { guess => [ '3', '3', '1', '4' ] },
                    '0,2' => { guess => [ '3', '0', '5', '5' ] },
                    '0,3' => { guess => [ '2', '3', '1', '5' ] },
                    '1,2' => { guess => [ '2', '3', '5', '1' ] },
                    '2,1' => { guess => [ '5', '3', '2', '1' ] },
                    '3,0' => { guess => [ '3', '5', '2', '1' ] }
                },
                '1,1' => {
                    guess => [ '0', '3', '4', '0' ],
                    '0,1' => { guess => [ '5', '0', '5', '2' ] },
                    '0,2' => { guess => [ '3', '4', '2', '1' ] },
                    '1,0' => { guess => [ '2', '3', '1', '2' ] },
                    '1,1' => { guess => [ '4', '3', '2', '1' ] },
                    '1,2' => { guess => [ '3', '0', '4', '5' ] },
                    '2,0' => { guess => [ '2', '3', '4', '1' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '4', '4' ],
                    '1,1' => { guess => [ '4', '0', '5', '2' ] },
                    '2,0' => { guess => [ '5', '0', '4', '2' ] },
                    '2,1' => { guess => [ '4', '0', '4', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '3', '2', '5' ],
                    '0,2' => { guess => [ '3', '0', '4', '4' ] },
                    '0,3' => { guess => [ '3', '0', '5', '4' ] },
                    '1,1' => { guess => [ '2', '3', '1', '4' ] },
                    '1,2' => { guess => [ '2', '0', '5', '5' ] },
                    '2,0' => { guess => [ '2', '3', '2', '1' ] },
                    '2,1' => { guess => [ '5', '0', '2', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '2', '2' ],
                    '1,1' => { guess => [ '2', '0', '4', '5' ] },
                    '2,0' => { guess => [ '4', '0', '2', '5' ] },
                    '2,1' => { guess => [ '2', '0', '5', '2' ] },
                    '3,0' => { guess => [ '5', '0', '2', '2' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '2', '0' ],
                    '1,1' => { guess => [ '2', '0', '4', '2' ] },
                    '2,0' => { guess => [ '4', '0', '2', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '4', '2', '5' ],
                    '0,3' => { guess => [ '2', '0', '4', '4' ] },
                    '0,4' => { guess => [ '2', '0', '5', '4' ] },
                    '1,1' => { guess => [ '2', '0', '2', '2' ] },
                    '1,2' => { guess => [ '4', '0', '2', '4' ] },
                    '1,3' => { guess => [ '5', '0', '2', '4' ] },
                    '2,1' => { guess => [ '2', '0', '2', '5' ] }
                }
            },
            '0,3' => {
                guess => [ '0', '2', '2', '4' ],
                '0,1' => { guess => [ '3', '3', '1', '2' ] },
                '0,2' => { guess => [ '3', '0', '5', '2' ] },
                '0,3' => { guess => [ '3', '0', '4', '2' ] },
                '1,0' => { guess => [ '3', '3', '2', '1' ] },
                '1,1' => { guess => [ '3', '0', '2', '5' ] },
                '1,2' => { guess => [ '3', '0', '2', '2' ] },
                '2,1' => { guess => [ '3', '0', '2', '4' ] }
            },
            '1,0' => {
                guess => [ '1', '4', '1', '2' ],
                '0,0' => { guess => [ '0', '5', '5', '5' ] },
                '0,1' => {
                    guess => [ '0', '0', '4', '5' ],
                    '0,1' => { guess => [ '5', '5', '3', '1' ] },
                    '1,2' => { guess => [ '0', '5', '5', '4' ] },
                    '2,1' => { guess => [ '0', '5', '4', '4' ] },
                    '3,0' => { guess => [ '0', '5', '4', '5' ] }
                },
                '0,2' => {
                    guess => [ '0', '2', '2', '2' ],
                    '0,0' => { guess => [ '4', '5', '3', '1' ] },
                    '1,0' => { guess => [ '5', '2', '5', '1' ] },
                    '1,1' => { guess => [ '2', '2', '5', '1' ] },
                    '2,0' => { guess => [ '5', '2', '2', '1' ] },
                    '2,1' => { guess => [ '2', '2', '2', '1' ] }
                },
                '0,3' => {
                    guess => [ '2', '0', '4', '5' ],
                    '0,2' => { guess => [ '4', '2', '2', '1' ] },
                    '0,3' => { guess => [ '4', '2', '5', '1' ] },
                    '1,1' => { guess => [ '4', '2', '4', '1' ] },
                    '1,2' => { guess => [ '5', '2', '4', '1' ] },
                    '2,0' => { guess => [ '2', '2', '4', '1' ] }
                },
                '1,0' => {
                    guess => [ '0', '0', '4', '5' ],
                    '0,1' => { guess => [ '5', '5', '1', '3' ] },
                    '1,2' => { guess => [ '0', '4', '5', '4' ] },
                    '2,0' => { guess => [ '0', '4', '4', '4' ] },
                    '2,1' => { guess => [ '0', '4', '5', '5' ] },
                    '3,0' => { guess => [ '0', '4', '4', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '5', '2', '2' ],
                    '0,0' => { guess => [ '4', '4', '3', '1' ] },
                    '0,1' => { guess => [ '5', '4', '3', '1' ] },
                    '0,2' => { guess => [ '5', '2', '1', '5' ] },
                    '0,3' => { guess => [ '2', '2', '1', '5' ] },
                    '1,0' => { guess => [ '4', '5', '1', '3' ] }
                },
                '1,2' => {
                    guess => [ '2', '0', '0', '5' ],
                    '0,1' => { guess => [ '4', '2', '1', '4' ] },
                    '0,2' => { guess => [ '5', '2', '1', '4' ] },
                    '1,0' => { guess => [ '2', '2', '1', '4' ] },
                    '1,1' => { guess => [ '4', '2', '1', '5' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '5', '2' ],
                    '0,0' => { guess => [ '4', '4', '1', '3' ] },
                    '0,1' => { guess => [ '5', '4', '1', '3' ] },
                    '1,0' => { guess => [ '2', '2', '1', '2' ] },
                    '1,1' => { guess => [ '5', '2', '1', '2' ] }
                },
                '2,1' => { guess => [ '4', '2', '1', '2' ] }
            },
            '1,1' => {
                guess => [ '3', '3', '1', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '2', '5' ],
                    '1,2' => { guess => [ '0', '5', '5', '2' ] },
                    '2,1' => { guess => [ '0', '5', '2', '2' ] },
                    '3,0' => { guess => [ '0', '5', '2', '5' ] }
                },
                '0,1' => {
                    guess => [ '0', '4', '2', '5' ],
                    '0,2' => { guess => [ '5', '0', '5', '3' ] },
                    '1,1' => { guess => [ '5', '0', '3', '5' ] },
                    '1,3' => { guess => [ '0', '5', '4', '2' ] },
                    '2,1' => { guess => [ '0', '4', '4', '2' ] },
                    '2,2' => { guess => [ '0', '4', '5', '2' ] },
                    '3,0' => { guess => [ '0', '4', '2', '2' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '5', '3' ],
                    '0,2' => { guess => [ '2', '5', '3', '1' ] },
                    '1,2' => { guess => [ '4', '0', '3', '5' ] },
                    '2,0' => { guess => [ '4', '0', '4', '3' ] },
                    '2,1' => { guess => [ '5', '0', '4', '3' ] },
                    '3,0' => { guess => [ '4', '0', '5', '3' ] }
                },
                '0,3' => { guess => [ '2', '4', '3', '1' ] },
                '1,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '0', '4', '2', '4' ] },
                    '1,1' => { guess => [ '0', '5', '2', '4' ] },
                    '2,0' => { guess => [ '0', '3', '5', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '5', '4' ],
                    '0,0' => { guess => [ '3', '2', '2', '1' ] },
                    '0,1' => { guess => [ '2', '5', '1', '3' ] },
                    '1,0' => { guess => [ '3', '2', '5', '1' ] },
                    '1,2' => { guess => [ '0', '3', '4', '5' ] },
                    '2,0' => { guess => [ '4', '0', '3', '4' ] },
                    '2,1' => { guess => [ '5', '0', '3', '4' ] }
                },
                '1,2' => {
                    guess => [ '0', '3', '0', '3' ],
                    '0,1' => { guess => [ '3', '2', '4', '1' ] },
                    '0,2' => { guess => [ '3', '5', '3', '1' ] },
                    '1,0' => { guess => [ '2', '4', '1', '3' ] },
                    '1,1' => { guess => [ '5', '3', '3', '1' ] }
                },
                '1,3' => {
                    guess => [ '0', '3', '0', '0' ],
                    '0,1' => { guess => [ '3', '4', '3', '1' ] },
                    '1,0' => { guess => [ '4', '3', '3', '1' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '4', '2' ],
                    '0,1' => { guess => [ '3', '2', '1', '5' ] },
                    '1,0' => { guess => [ '3', '2', '1', '2' ] },
                    '1,1' => { guess => [ '0', '3', '5', '4' ] },
                    '2,0' => { guess => [ '0', '3', '4', '4' ] }
                },
                '2,1' => {
                    guess => [ '0', '3', '0', '1' ],
                    '0,2' => { guess => [ '3', '5', '1', '3' ] },
                    '1,1' => { guess => [ '5', '3', '1', '3' ] },
                    '2,0' => { guess => [ '3', '3', '3', '1' ] }
                },
                '2,2' => {
                    guess => [ '0', '3', '0', '0' ],
                    '0,1' => { guess => [ '3', '4', '1', '3' ] },
                    '1,0' => { guess => [ '4', '3', '1', '3' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,0' => { guess => [ '3', '3', '1', '3' ] },
                    '0,1' => { guess => [ '3', '2', '1', '4' ] }
                }
            },
            '1,2' => {
                guess => [ '2', '0', '3', '4' ],
                '0,3' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '0', '3', '2', '2' ] },
                    '1,1' => { guess => [ '0', '3', '5', '2' ] },
                    '2,0' => { guess => [ '0', '3', '2', '5' ] }
                },
                '0,4' => { guess => [ '0', '3', '4', '2' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '2', '3', '1', '3' ] },
                    '1,0' => { guess => [ '3', '0', '5', '3' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,0' => { guess => [ '3', '0', '4', '3' ] },
                    '1,1' => { guess => [ '5', '0', '2', '3' ] }
                },
                '1,3' => {
                    guess => [ '0', '0', '0', '3' ],
                    '1,1' => { guess => [ '0', '3', '2', '4' ] },
                    '2,0' => { guess => [ '4', '0', '2', '3' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '2', '3', '3', '1' ] },
                    '1,0' => { guess => [ '3', '0', '3', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '2', '2' ],
                    '1,1' => { guess => [ '2', '0', '5', '3' ] },
                    '2,0' => { guess => [ '5', '0', '3', '2' ] },
                    '2,1' => { guess => [ '2', '0', '2', '3' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,1' => { guess => [ '2', '0', '4', '3' ] },
                    '2,0' => { guess => [ '4', '0', '3', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,0' => { guess => [ '3', '0', '3', '4' ] },
                    '1,1' => { guess => [ '2', '0', '3', '5' ] },
                    '2,0' => { guess => [ '2', '0', '3', '2' ] }
                }
            },
            '1,3' => {
                guess => [ '0', '0', '0', '2' ],
                '1,1' => { guess => [ '3', '0', '2', '3' ] },
                '2,0' => { guess => [ '3', '0', '3', '2' ] }
            },
            '2,0' => {
                guess => [ '0', '3', '0', '4' ],
                '0,1' => {
                    guess => [ '0', '0', '1', '5' ],
                    '0,1' => { guess => [ '2', '2', '3', '1' ] },
                    '0,2' => { guess => [ '5', '2', '3', '1' ] },
                    '1,0' => { guess => [ '2', '2', '1', '3' ] },
                    '1,1' => { guess => [ '5', '2', '1', '3' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '4', '2', '1', '3' ] },
                    '1,0' => { guess => [ '4', '2', '3', '1' ] }
                },
                '1,0' => {
                    guess => [ '0', '0', '2', '5' ],
                    '1,2' => { guess => [ '0', '2', '5', '2' ] },
                    '2,0' => { guess => [ '0', '2', '2', '2' ] },
                    '2,1' => { guess => [ '0', '2', '5', '5' ] },
                    '3,0' => { guess => [ '0', '2', '2', '5' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '3', '5' ],
                    '1,0' => { guess => [ '0', '2', '4', '2' ] },
                    '1,2' => { guess => [ '0', '5', '5', '3' ] },
                    '2,0' => { guess => [ '0', '2', '4', '5' ] },
                    '3,0' => { guess => [ '0', '5', '3', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '5', '3' ],
                    '1,2' => { guess => [ '0', '4', '3', '5' ] },
                    '2,0' => { guess => [ '0', '4', '4', '3' ] },
                    '2,1' => { guess => [ '0', '5', '4', '3' ] },
                    '3,0' => { guess => [ '0', '4', '5', '3' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '2', '5' ],
                    '1,1' => { guess => [ '0', '2', '4', '4' ] },
                    '1,2' => { guess => [ '0', '2', '5', '4' ] },
                    '2,0' => { guess => [ '0', '2', '2', '4' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '0', '4', '3', '4' ] },
                    '1,1' => { guess => [ '0', '5', '3', '4' ] }
                }
            },
            '2,1' => {
                guess => [ '0', '3', '0', '4' ],
                '0,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '3', '2', '1', '3' ] },
                    '1,0' => { guess => [ '3', '2', '3', '1' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '3', '0', '3', '3' ] },
                    '1,1' => { guess => [ '5', '0', '3', '3' ] }
                },
                '0,3' => { guess => [ '4', '0', '3', '3' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,1' => { guess => [ '0', '5', '2', '3' ] },
                    '2,0' => { guess => [ '0', '5', '3', '2' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,1' => { guess => [ '0', '4', '2', '3' ] },
                    '2,0' => { guess => [ '0', '4', '3', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '1,1' => { guess => [ '0', '3', '3', '5' ] },
                    '2,0' => { guess => [ '0', '3', '5', '3' ] }
                },
                '2,1' => { guess => [ '0', '3', '4', '3' ] },
                '3,0' => { guess => [ '0', '3', '3', '4' ] }
            },
            '2,2' => {
                guess => [ '0', '0', '2', '3' ],
                '1,2' => { guess => [ '0', '3', '3', '2' ] },
                '2,1' => { guess => [ '2', '0', '3', '3' ] },
                '3,0' => { guess => [ '0', '3', '2', '3' ] }
            },
            '3,0' => {
                guess => [ '0', '2', '2', '4' ],
                '1,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '0', '3', '3', '3' ] },
                    '1,1' => { guess => [ '0', '5', '3', '3' ] }
                },
                '1,1' => { guess => [ '0', '4', '3', '3' ] },
                '2,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '1,1' => { guess => [ '0', '2', '3', '5' ] },
                    '2,0' => { guess => [ '0', '2', '5', '3' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,1' => { guess => [ '0', '2', '4', '3' ] },
                    '2,0' => { guess => [ '0', '2', '3', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '3' ],
                    '1,1' => { guess => [ '0', '2', '3', '4' ] },
                    '2,0' => { guess => [ '0', '2', '2', '3' ] }
                }
            }
        },
        '1,1' => {
            guess => [ '0', '0', '2', '3' ],
            '0,0' => {
                guess => [ '1', '4', '1', '4' ],
                '0,2' => { guess => [ '5', '1', '5', '1' ] },
                '0,3' => {
                    guess => [ '0', '0', '4', '0' ],
                    '0,1' => { guess => [ '4', '1', '5', '1' ] },
                    '1,0' => { guess => [ '5', '1', '4', '1' ] }
                },
                '0,4' => { guess => [ '4', '1', '4', '1' ] },
                '1,1' => {
                    guess => [ '0', '1', '0', '5' ],
                    '0,2' => { guess => [ '1', '5', '5', '1' ] },
                    '1,1' => { guess => [ '1', '1', '5', '1' ] },
                    '2,0' => { guess => [ '5', '1', '1', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '1', '4', '1' ] },
                    '0,1' => { guess => [ '1', '5', '4', '1' ] },
                    '1,0' => { guess => [ '4', '1', '1', '5' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '5', '5' ],
                    '1,0' => { guess => [ '1', '1', '1', '5' ] },
                    '1,1' => { guess => [ '1', '5', '1', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '5', '1', '1', '4' ] },
                    '1,0' => { guess => [ '1', '4', '5', '1' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '4', '1', '1', '4' ] },
                    '1,0' => { guess => [ '1', '4', '4', '1' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '1', '1', '4' ] },
                    '0,1' => { guess => [ '1', '5', '1', '4' ] },
                    '1,0' => { guess => [ '1', '4', '1', '5' ] }
                }
            },
            '0,1' => {
                guess => [ '0', '1', '3', '4' ],
                '0,1' => {
                    guess => [ '0', '0', '1', '5' ],
                    '0,2' => { guess => [ '1', '2', '5', '1' ] },
                    '1,0' => { guess => [ '1', '2', '1', '2' ] },
                    '1,1' => { guess => [ '1', '5', '1', '2' ] },
                    '2,0' => { guess => [ '1', '2', '1', '5' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '1', '5' ],
                    '0,1' => { guess => [ '1', '2', '4', '1' ] },
                    '0,2' => { guess => [ '1', '3', '5', '1' ] },
                    '0,3' => { guess => [ '5', '5', '0', '1' ] },
                    '1,0' => { guess => [ '1', '4', '1', '2' ] },
                    '1,2' => { guess => [ '5', '5', '1', '0' ] },
                    '2,0' => { guess => [ '1', '3', '1', '5' ] }
                },
                '0,3' => {
                    guess => [ '0', '5', '0', '1' ],
                    '0,2' => { guess => [ '4', '4', '1', '0' ] },
                    '0,3' => { guess => [ '5', '4', '1', '0' ] },
                    '1,0' => { guess => [ '1', '3', '4', '1' ] },
                    '1,2' => { guess => [ '4', '5', '1', '0' ] },
                    '2,0' => { guess => [ '4', '4', '0', '1' ] },
                    '2,1' => { guess => [ '5', '4', '0', '1' ] },
                    '3,0' => { guess => [ '4', '5', '0', '1' ] }
                },
                '1,0' => {
                    guess => [ '1', '0', '1', '5' ],
                    '0,3' => { guess => [ '2', '1', '5', '1' ] },
                    '1,1' => { guess => [ '2', '1', '1', '2' ] },
                    '1,2' => { guess => [ '5', '1', '1', '2' ] },
                    '2,0' => { guess => [ '1', '1', '1', '2' ] },
                    '2,1' => { guess => [ '2', '1', '1', '5' ] }
                },
                '1,1' => {
                    guess => [ '1', '3', '1', '5' ],
                    '0,2' => { guess => [ '2', '1', '4', '1' ] },
                    '0,4' => { guess => [ '3', '1', '5', '1' ] },
                    '1,1' => { guess => [ '4', '1', '1', '2' ] },
                    '1,3' => { guess => [ '1', '5', '3', '1' ] },
                    '2,0' => { guess => [ '1', '2', '1', '4' ] },
                    '2,1' => { guess => [ '1', '3', '3', '1' ] },
                    '2,2' => { guess => [ '3', '1', '1', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '3', '1' ],
                    '0,2' => { guess => [ '1', '3', '1', '4' ] },
                    '1,1' => { guess => [ '3', '1', '4', '1' ] },
                    '2,0' => { guess => [ '1', '4', '3', '1' ] }
                },
                '2,0' => {
                    guess => [ '1', '0', '3', '3' ],
                    '0,1' => { guess => [ '2', '1', '1', '4' ] },
                    '1,1' => { guess => [ '5', '1', '3', '1' ] },
                    '1,2' => { guess => [ '3', '1', '3', '1' ] },
                    '2,0' => { guess => [ '1', '1', '3', '1' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '0,1' => { guess => [ '3', '1', '1', '4' ] },
                    '1,0' => { guess => [ '4', '1', '3', '1' ] }
                }
            },
            '0,2' => {
                guess => [ '2', '2', '4', '1' ],
                '0,1' => {
                    guess => [ '0', '3', '0', '3' ],
                    '0,2' => { guess => [ '3', '5', '1', '0' ] },
                    '1,1' => { guess => [ '5', '3', '1', '0' ] },
                    '1,2' => { guess => [ '3', '3', '1', '0' ] }
                },
                '0,2' => {
                    guess => [ '0', '3', '0', '0' ],
                    '0,1' => { guess => [ '3', '1', '1', '2' ] },
                    '1,0' => { guess => [ '1', '3', '1', '2' ] },
                    '1,1' => { guess => [ '3', '4', '1', '0' ] },
                    '2,0' => { guess => [ '4', '3', '1', '0' ] }
                },
                '1,0' => {
                    guess => [ '0', '3', '0', '3' ],
                    '1,1' => { guess => [ '3', '5', '0', '1' ] },
                    '2,0' => { guess => [ '5', '3', '0', '1' ] },
                    '2,1' => { guess => [ '3', '3', '0', '1' ] }
                },
                '1,1' => {
                    guess => [ '2', '3', '0', '1' ],
                    '0,3' => { guess => [ '5', '2', '1', '0' ] },
                    '1,2' => { guess => [ '2', '5', '1', '0' ] },
                    '2,1' => { guess => [ '3', '4', '0', '1' ] },
                    '3,0' => { guess => [ '4', '3', '0', '1' ] }
                },
                '1,2' => {
                    guess => [ '0', '2', '0', '0' ],
                    '1,1' => { guess => [ '2', '4', '1', '0' ] },
                    '2,0' => { guess => [ '4', '2', '1', '0' ] }
                },
                '2,0' => {
                    guess => [ '0', '2', '0', '0' ],
                    '0,1' => { guess => [ '2', '1', '3', '1' ] },
                    '1,0' => { guess => [ '1', '2', '3', '1' ] },
                    '1,1' => { guess => [ '2', '5', '0', '1' ] },
                    '2,0' => { guess => [ '5', '2', '0', '1' ] }
                },
                '2,1' => {
                    guess => [ '0', '2', '0', '1' ],
                    '1,2' => { guess => [ '2', '2', '1', '0' ] },
                    '2,1' => { guess => [ '2', '4', '0', '1' ] },
                    '3,0' => { guess => [ '4', '2', '0', '1' ] }
                },
                '3,0' => { guess => [ '2', '2', '0', '1' ] }
            },
            '0,3' => {
                guess => [ '0', '2', '0', '1' ],
                '0,3' => { guess => [ '2', '3', '1', '0' ] },
                '1,2' => { guess => [ '3', '2', '1', '0' ] },
                '2,1' => { guess => [ '2', '3', '0', '1' ] },
                '3,0' => { guess => [ '3', '2', '0', '1' ] }
            },
            '1,0' => {
                guess => [ '0', '1', '2', '4' ],
                '0,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '3', '1', '3' ] },
                    '0,1' => { guess => [ '1', '5', '1', '3' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '1', '4', '1', '3' ] },
                    '1,0' => { guess => [ '1', '0', '5', '5' ] }
                },
                '0,3' => { guess => [ '1', '0', '4', '5' ] },
                '1,0' => {
                    guess => [ '1', '0', '0', '5' ],
                    '0,1' => { guess => [ '3', '1', '1', '3' ] },
                    '0,2' => { guess => [ '5', '1', '1', '3' ] },
                    '1,0' => { guess => [ '1', '1', '1', '3' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '1', '5' ],
                    '0,1' => { guess => [ '1', '2', '2', '1' ] },
                    '0,2' => { guess => [ '1', '5', '2', '1' ] },
                    '1,0' => { guess => [ '4', '1', '1', '3' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '4', '2', '1' ] },
                    '1,0' => { guess => [ '1', '0', '4', '4' ] },
                    '1,1' => { guess => [ '1', '0', '5', '4' ] }
                },
                '2,0' => {
                    guess => [ '1', '0', '0', '5' ],
                    '0,1' => { guess => [ '2', '1', '2', '1' ] },
                    '0,2' => { guess => [ '5', '1', '2', '1' ] },
                    '1,0' => { guess => [ '1', '1', '2', '1' ] },
                    '1,2' => { guess => [ '0', '1', '5', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '4', '1', '2', '1' ] },
                    '1,0' => { guess => [ '0', '1', '4', '5' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '0', '1', '4', '4' ] },
                    '1,1' => { guess => [ '0', '1', '5', '4' ] }
                }
            },
            '1,1' => {
                guess => [ '0', '4', '0', '5' ],
                '0,0' => {
                    guess => [ '0', '1', '1', '3' ],
                    '0,3' => { guess => [ '1', '3', '2', '1' ] },
                    '1,2' => { guess => [ '3', '1', '2', '1' ] },
                    '2,1' => { guess => [ '1', '2', '1', '3' ] },
                    '3,0' => { guess => [ '2', '1', '1', '3' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '4' ],
                    '1,0' => { guess => [ '1', '0', '5', '2' ] },
                    '1,1' => { guess => [ '1', '0', '4', '2' ] },
                    '2,0' => { guess => [ '1', '0', '3', '4' ] }
                },
                '0,3' => {
                    guess => [ '0', '0', '0', '4' ],
                    '1,1' => { guess => [ '5', '0', '5', '0' ] },
                    '1,2' => { guess => [ '4', '0', '4', '0' ] }
                },
                '0,4' => {
                    guess => [ '0', '0', '4', '0' ],
                    '2,1' => { guess => [ '4', '0', '5', '0' ] },
                    '3,0' => { guess => [ '5', '0', '4', '0' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '2', '4' ],
                    '1,0' => { guess => [ '1', '0', '3', '5' ] },
                    '1,1' => { guess => [ '0', '1', '5', '2' ] },
                    '1,2' => { guess => [ '0', '1', '4', '2' ] },
                    '2,0' => { guess => [ '0', '1', '3', '4' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '4' ],
                    '1,1' => { guess => [ '0', '5', '5', '0' ] },
                    '2,1' => { guess => [ '5', '0', '0', '0' ] },
                    '2,2' => { guess => [ '4', '0', '0', '0' ] },
                    '3,0' => { guess => [ '4', '0', '0', '4' ] }
                },
                '1,3' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '5', '4', '0' ] },
                    '2,0' => { guess => [ '5', '0', '0', '4' ] }
                },
                '2,0' => { guess => [ '0', '1', '3', '5' ] },
                '2,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '4', '4', '0' ] },
                    '2,0' => { guess => [ '5', '0', '0', '5' ] },
                    '2,1' => { guess => [ '0', '5', '0', '0' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '0', '4' ],
                    '1,2' => { guess => [ '0', '4', '5', '0' ] },
                    '2,1' => { guess => [ '4', '0', '0', '5' ] },
                    '3,0' => { guess => [ '0', '5', '0', '4' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '2,0' => { guess => [ '0', '5', '0', '5' ] },
                    '2,2' => { guess => [ '0', '4', '0', '0' ] },
                    '3,0' => { guess => [ '0', '4', '0', '4' ] }
                }
            },
            '1,2' => {
                guess => [ '0', '2', '0', '4' ],
                '0,2' => {
                    guess => [ '0', '0', '3', '3' ],
                    '1,2' => { guess => [ '3', '0', '5', '0' ] },
                    '2,0' => { guess => [ '1', '0', '3', '2' ] },
                    '2,1' => { guess => [ '5', '0', '3', '0' ] },
                    '2,2' => { guess => [ '3', '0', '3', '0' ] }
                },
                '0,3' => {
                    guess => [ '0', '0', '3', '0' ],
                    '2,0' => { guess => [ '2', '0', '5', '0' ] },
                    '2,1' => { guess => [ '3', '0', '4', '0' ] },
                    '3,0' => { guess => [ '4', '0', '3', '0' ] }
                },
                '0,4' => { guess => [ '2', '0', '4', '0' ] },
                '1,1' => {
                    guess => [ '0', '1', '5', '2' ],
                    '0,1' => { guess => [ '3', '0', '0', '0' ] },
                    '0,2' => { guess => [ '3', '0', '0', '5' ] },
                    '1,0' => { guess => [ '0', '3', '3', '0' ] },
                    '1,1' => { guess => [ '0', '5', '3', '0' ] },
                    '2,0' => { guess => [ '0', '3', '5', '0' ] },
                    '3,0' => { guess => [ '0', '1', '3', '2' ] }
                },
                '1,2' => {
                    guess => [ '5', '3', '4', '0' ],
                    '0,1' => { guess => [ '2', '0', '0', '2' ] },
                    '0,2' => { guess => [ '2', '0', '0', '5' ] },
                    '1,0' => { guess => [ '2', '0', '0', '0' ] },
                    '1,1' => { guess => [ '5', '0', '0', '2' ] },
                    '1,2' => { guess => [ '0', '4', '3', '0' ] },
                    '3,0' => { guess => [ '0', '3', '4', '0' ] }
                },
                '1,3' => { guess => [ '4', '0', '0', '2' ] },
                '2,0' => {
                    guess => [ '0', '0', '0', '0' ],
                    '2,0' => { guess => [ '0', '3', '0', '5' ] },
                    '3,0' => { guess => [ '0', '3', '0', '0' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,2' => { guess => [ '0', '2', '5', '0' ] },
                    '2,0' => { guess => [ '3', '0', '0', '4' ] },
                    '3,0' => { guess => [ '0', '5', '0', '2' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,2' => { guess => [ '0', '2', '4', '0' ] },
                    '2,1' => { guess => [ '2', '0', '0', '4' ] },
                    '3,0' => { guess => [ '0', '4', '0', '2' ] }
                },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '2,0' => { guess => [ '0', '3', '0', '4' ] },
                    '2,1' => { guess => [ '0', '2', '0', '5' ] },
                    '2,2' => { guess => [ '0', '2', '0', '0' ] },
                    '3,0' => { guess => [ '0', '2', '0', '2' ] }
                }
            },
            '1,3' => {
                guess => [ '0', '1', '0', '2' ],
                '0,3' => { guess => [ '2', '0', '3', '0' ] },
                '1,2' => { guess => [ '0', '2', '3', '0' ] },
                '2,1' => { guess => [ '3', '0', '0', '2' ] },
                '3,0' => { guess => [ '0', '3', '0', '2' ] }
            },
            '2,0' => {
                guess => [ '0', '1', '2', '4' ],
                '0,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '1', '0', '3', '3' ] },
                    '1,1' => { guess => [ '1', '0', '5', '3' ] }
                },
                '0,3' => { guess => [ '1', '0', '4', '3' ] },
                '1,2' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,1' => { guess => [ '1', '0', '2', '5' ] },
                    '2,0' => { guess => [ '1', '0', '2', '2' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '0', '1', '3', '3' ] },
                    '1,1' => { guess => [ '0', '1', '5', '3' ] }
                },
                '2,1' => { guess => [ '0', '1', '4', '3' ] },
                '2,2' => { guess => [ '1', '0', '2', '4' ] },
                '3,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '1,1' => { guess => [ '0', '1', '2', '5' ] },
                    '2,0' => { guess => [ '0', '1', '2', '2' ] }
                }
            },
            '2,1' => {
                guess => [ '0', '2', '0', '4' ],
                '0,3' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,1' => { guess => [ '2', '0', '2', '0' ] },
                    '1,2' => { guess => [ '5', '0', '2', '0' ] }
                },
                '0,4' => { guess => [ '4', '0', '2', '0' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '2,0' => { guess => [ '3', '0', '0', '3' ] },
                    '2,1' => { guess => [ '5', '0', '0', '3' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '5', '2', '0' ] },
                    '2,0' => { guess => [ '4', '0', '0', '3' ] }
                },
                '1,3' => { guess => [ '0', '4', '2', '0' ] },
                '2,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '2,0' => { guess => [ '0', '3', '0', '3' ] },
                    '2,1' => { guess => [ '0', '5', '0', '3' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '2', '2', '0' ] },
                    '2,0' => { guess => [ '0', '4', '0', '3' ] }
                }
            },
            '2,2' => {
                guess => [ '0', '1', '0', '3' ],
                '0,3' => { guess => [ '3', '0', '2', '0' ] },
                '1,2' => { guess => [ '0', '3', '2', '0' ] },
                '2,1' => { guess => [ '2', '0', '0', '3' ] },
                '3,0' => { guess => [ '0', '2', '0', '3' ] }
            },
            '3,0' => {
                guess => [ '0', '1', '0', '0' ],
                '0,2' => { guess => [ '1', '0', '2', '3' ] },
                '2,0' => { guess => [ '0', '1', '2', '3' ] }
            }
        },
        '1,2' => {
            guess => [ '0', '1', '0', '2' ],
            '0,2' => {
                guess => [ '0', '3', '0', '4' ],
                '0,1' => { guess => [ '1', '5', '1', '0' ] },
                '0,2' => { guess => [ '1', '4', '1', '0' ] },
                '1,1' => { guess => [ '1', '3', '1', '0' ] }
            },
            '0,3' => {
                guess => [ '0', '0', '3', '4' ],
                '0,1' => { guess => [ '1', '2', '1', '0' ] },
                '1,1' => { guess => [ '1', '0', '5', '0' ] },
                '1,2' => { guess => [ '1', '0', '4', '0' ] },
                '2,1' => { guess => [ '1', '0', '3', '0' ] }
            },
            '0,4' => { guess => [ '1', '0', '2', '0' ] },
            '1,1' => {
                guess => [ '0', '1', '3', '4' ],
                '0,2' => { guess => [ '1', '5', '0', '1' ] },
                '0,3' => {
                    guess => [ '0', '0', '0', '3' ],
                    '1,0' => { guess => [ '1', '4', '0', '1' ] },
                    '1,1' => { guess => [ '1', '3', '0', '1' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,1' => { guess => [ '1', '1', '1', '0' ] },
                    '0,2' => { guess => [ '5', '1', '1', '0' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '3' ],
                    '0,1' => { guess => [ '4', '1', '1', '0' ] },
                    '0,2' => { guess => [ '3', '1', '1', '0' ] }
                }
            },
            '1,2' => {
                guess => [ '0', '0', '0', '3' ],
                '0,1' => { guess => [ '2', '1', '1', '0' ] },
                '1,0' => { guess => [ '1', '2', '0', '1' ] },
                '2,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '2,0' => { guess => [ '1', '0', '0', '5' ] },
                    '3,0' => { guess => [ '1', '0', '0', '4' ] }
                },
                '2,1' => { guess => [ '1', '0', '0', '0' ] },
                '3,0' => { guess => [ '1', '0', '0', '3' ] }
            },
            '2,0' => {
                guess => [ '0', '0', '3', '4' ],
                '0,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '1', '1', '0', '1' ] },
                    '1,1' => { guess => [ '5', '1', '0', '1' ] }
                },
                '0,2' => {
                    guess => [ '0', '0', '0', '3' ],
                    '1,0' => { guess => [ '4', '1', '0', '1' ] },
                    '1,1' => { guess => [ '3', '1', '0', '1' ] }
                }
            },
            '2,1' => {
                guess => [ '0', '0', '3', '4' ],
                '0,1' => { guess => [ '2', '1', '0', '1' ] },
                '1,1' => { guess => [ '0', '1', '5', '0' ] },
                '1,2' => { guess => [ '0', '1', '4', '0' ] },
                '2,1' => { guess => [ '0', '1', '3', '0' ] }
            },
            '2,2' => {
                guess => [ '0', '0', '0', '1' ],
                '1,2' => { guess => [ '0', '1', '2', '0' ] },
                '2,1' => { guess => [ '1', '0', '0', '2' ] }
            },
            '3,0' => {
                guess => [ '0', '0', '0', '3' ],
                '2,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '2,0' => { guess => [ '0', '1', '0', '5' ] },
                    '3,0' => { guess => [ '0', '1', '0', '4' ] }
                },
                '2,1' => { guess => [ '0', '1', '0', '0' ] },
                '3,0' => { guess => [ '0', '1', '0', '3' ] }
            }
        },
        '2,0' => {
            guess => [ '0', '1', '2', '3' ],
            '0,1' => {
                guess => [ '1', '4', '0', '4' ],
                '0,1' => { guess => [ '5', '5', '1', '1' ] },
                '0,2' => { guess => [ '4', '5', '1', '1' ] },
                '1,0' => { guess => [ '1', '5', '1', '1' ] },
                '1,1' => { guess => [ '5', '4', '1', '1' ] },
                '1,2' => { guess => [ '4', '4', '1', '1' ] },
                '2,0' => { guess => [ '1', '4', '1', '1' ] }
            },
            '0,2' => {
                guess => [ '0', '2', '1', '4' ],
                '0,2' => { guess => [ '5', '0', '5', '1' ] },
                '0,3' => {
                    guess => [ '0', '0', '4', '4' ],
                    '1,1' => { guess => [ '4', '0', '5', '1' ] },
                    '2,0' => { guess => [ '5', '0', '4', '1' ] },
                    '2,1' => { guess => [ '4', '0', '4', '1' ] }
                },
                '1,0' => {
                    guess => [ '1', '3', '0', '3' ],
                    '0,2' => { guess => [ '3', '5', '1', '1' ] },
                    '1,1' => { guess => [ '5', '3', '1', '1' ] },
                    '1,2' => { guess => [ '3', '3', '1', '1' ] },
                    '2,0' => { guess => [ '1', '3', '1', '1' ] }
                },
                '1,1' => {
                    guess => [ '0', '3', '0', '1' ],
                    '0,2' => { guess => [ '5', '0', '1', '5' ] },
                    '1,0' => { guess => [ '2', '5', '1', '1' ] },
                    '1,1' => { guess => [ '3', '4', '1', '1' ] },
                    '2,0' => { guess => [ '4', '3', '1', '1' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '2', '4', '1', '1' ] },
                    '1,0' => { guess => [ '4', '0', '1', '5' ] }
                },
                '2,0' => {
                    guess => [ '1', '0', '0', '5' ],
                    '0,1' => { guess => [ '2', '2', '1', '1' ] },
                    '0,2' => { guess => [ '5', '2', '1', '1' ] },
                    '1,0' => { guess => [ '1', '2', '1', '1' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '4', '2', '1', '1' ] },
                    '1,0' => { guess => [ '4', '0', '1', '4' ] },
                    '1,1' => { guess => [ '5', '0', '1', '4' ] }
                }
            },
            '0,3' => {
                guess => [ '0', '2', '1', '4' ],
                '0,2' => {
                    guess => [ '0', '0', '3', '3' ],
                    '1,1' => { guess => [ '3', '0', '5', '1' ] },
                    '2,0' => { guess => [ '5', '0', '3', '1' ] },
                    '2,1' => { guess => [ '3', '0', '3', '1' ] }
                },
                '0,3' => {
                    guess => [ '0', '0', '3', '0' ],
                    '1,0' => { guess => [ '2', '0', '5', '1' ] },
                    '1,1' => { guess => [ '3', '0', '4', '1' ] },
                    '2,0' => { guess => [ '4', '0', '3', '1' ] }
                },
                '0,4' => { guess => [ '2', '0', '4', '1' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '0' ],
                    '0,0' => { guess => [ '2', '3', '1', '1' ] },
                    '1,0' => { guess => [ '3', '0', '1', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '2', '0', '1', '2' ] },
                    '1,1' => { guess => [ '5', '0', '1', '2' ] },
                    '2,0' => { guess => [ '2', '0', '1', '5' ] }
                },
                '1,3' => { guess => [ '4', '0', '1', '2' ] },
                '2,0' => { guess => [ '3', '2', '1', '1' ] },
                '2,1' => { guess => [ '3', '0', '1', '4' ] },
                '2,2' => { guess => [ '2', '0', '1', '4' ] }
            },
            '0,4' => {
                guess => [ '0', '0', '0', '1' ],
                '1,1' => { guess => [ '3', '0', '1', '2' ] },
                '2,0' => { guess => [ '2', '0', '3', '1' ] }
            },
            '1,0' => {
                guess => [ '0', '2', '0', '4' ],
                '0,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '0,0' => { guess => [ '1', '1', '1', '1' ] },
                    '0,1' => { guess => [ '5', '1', '1', '1' ] }
                },
                '0,1' => { guess => [ '4', '1', '1', '1' ] },
                '1,1' => {
                    guess => [ '0', '0', '0', '0' ],
                    '2,0' => { guess => [ '0', '0', '5', '5' ] },
                    '3,0' => { guess => [ '0', '0', '5', '0' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '0' ],
                    '2,0' => { guess => [ '0', '0', '4', '5' ] },
                    '3,0' => { guess => [ '0', '0', '4', '0' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '0' ],
                    '3,0' => { guess => [ '0', '0', '0', '5' ] }
                },
                '2,1' => {
                    guess => [ '0', '0', '0', '5' ],
                    '2,0' => { guess => [ '0', '0', '4', '4' ] },
                    '2,1' => { guess => [ '0', '0', '5', '4' ] }
                },
                '3,0' => { guess => [ '0', '0', '0', '4' ] }
            },
            '1,1' => {
                guess => [ '1', '0', '4', '5' ],
                '0,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '0,0' => { guess => [ '3', '1', '1', '1' ] },
                    '0,1' => { guess => [ '2', '1', '1', '1' ] }
                },
                '0,3' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '4', '1', '4' ] },
                    '2,0' => { guess => [ '0', '5', '5', '1' ] }
                },
                '0,4' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '5', '1', '4' ] },
                    '2,0' => { guess => [ '0', '4', '5', '1' ] }
                },
                '1,0' => {
                    guess => [ '0', '0', '0', '1' ],
                    '2,1' => { guess => [ '0', '0', '3', '0' ] },
                    '3,0' => { guess => [ '0', '0', '0', '2' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '2' ],
                    '2,0' => { guess => [ '0', '0', '3', '4' ] },
                    '3,0' => { guess => [ '0', '0', '5', '2' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '5', '1', '5' ] },
                    '2,0' => { guess => [ '0', '4', '4', '1' ] }
                },
                '1,3' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '4', '1', '5' ] },
                    '2,0' => { guess => [ '0', '5', '4', '1' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '2' ],
                    '2,0' => { guess => [ '0', '0', '3', '5' ] },
                    '3,0' => { guess => [ '0', '0', '4', '2' ] }
                }
            },
            '1,2' => {
                guess => [ '0', '1', '3', '4' ],
                '0,2' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '2', '0', '2', '1' ] },
                    '1,1' => { guess => [ '5', '0', '2', '1' ] }
                },
                '0,3' => {
                    guess => [ '0', '0', '1', '5' ],
                    '1,1' => { guess => [ '4', '0', '2', '1' ] },
                    '2,0' => { guess => [ '3', '0', '1', '3' ] },
                    '2,1' => { guess => [ '5', '0', '1', '3' ] }
                },
                '0,4' => { guess => [ '4', '0', '1', '3' ] },
                '1,1' => {
                    guess => [ '0', '0', '1', '5' ],
                    '1,2' => { guess => [ '0', '2', '5', '1' ] },
                    '2,0' => { guess => [ '0', '2', '1', '2' ] },
                    '2,1' => { guess => [ '0', '5', '1', '2' ] },
                    '3,0' => { guess => [ '0', '2', '1', '5' ] }
                },
                '1,2' => {
                    guess => [ '0', '0', '1', '2' ],
                    '1,1' => { guess => [ '0', '3', '5', '1' ] },
                    '1,2' => { guess => [ '0', '2', '4', '1' ] },
                    '2,0' => { guess => [ '0', '3', '1', '5' ] },
                    '3,0' => { guess => [ '0', '4', '1', '2' ] }
                },
                '1,3' => { guess => [ '0', '3', '4', '1' ] },
                '2,0' => { guess => [ '0', '0', '3', '2' ] },
                '2,1' => {
                    guess => [ '0', '0', '1', '5' ],
                    '1,1' => { guess => [ '0', '3', '3', '1' ] },
                    '1,2' => { guess => [ '0', '5', '3', '1' ] },
                    '2,0' => { guess => [ '0', '2', '1', '4' ] }
                },
                '2,2' => {
                    guess => [ '0', '0', '0', '1' ],
                    '1,1' => { guess => [ '0', '3', '1', '4' ] },
                    '2,0' => { guess => [ '0', '4', '3', '1' ] }
                }
            },
            '1,3' => {
                guess => [ '0', '2', '1', '2' ],
                '0,3' => { guess => [ '3', '0', '2', '1' ] },
                '1,2' => { guess => [ '2', '0', '1', '3' ] },
                '2,1' => { guess => [ '0', '2', '3', '1' ] },
                '3,0' => { guess => [ '0', '3', '1', '2' ] }
            },
            '2,0' => {
                guess => [ '0', '4', '2', '5' ],
                '1,0' => {
                    guess => [ '0', '0', '0', '0' ],
                    '2,0' => { guess => [ '0', '0', '3', '3' ] },
                    '3,0' => { guess => [ '0', '0', '0', '3' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '4' ],
                    '2,0' => { guess => [ '0', '0', '5', '3' ] },
                    '2,1' => { guess => [ '0', '0', '4', '3' ] }
                },
                '2,0' => {
                    guess => [ '0', '0', '0', '0' ],
                    '2,0' => { guess => [ '0', '0', '2', '2' ] },
                    '3,0' => { guess => [ '0', '0', '2', '0' ] }
                },
                '2,1' => { guess => [ '0', '0', '2', '4' ] },
                '3,0' => { guess => [ '0', '0', '2', '5' ] }
            },
            '2,1' => {
                guess => [ '2', '4', '1', '5' ],
                '0,2' => { guess => [ '0', '2', '2', '1' ] },
                '0,3' => { guess => [ '0', '5', '2', '1' ] },
                '1,0' => { guess => [ '0', '3', '1', '3' ] },
                '1,1' => { guess => [ '0', '5', '1', '3' ] },
                '1,2' => { guess => [ '0', '4', '2', '1' ] },
                '2,0' => { guess => [ '0', '4', '1', '3' ] }
            },
            '2,2' => {
                guess => [ '0', '0', '0', '1' ],
                '1,1' => { guess => [ '0', '2', '1', '3' ] },
                '2,0' => { guess => [ '0', '3', '2', '1' ] }
            },
            '3,0' => { guess => [ '0', '0', '2', '3' ] }
        },
        '2,1' => {
            guess => [ '0', '1', '1', '2' ],
            '0,2' => {
                guess => [ '3', '0', '0', '4' ],
                '2,0' => { guess => [ '5', '0', '0', '1' ] },
                '2,1' => { guess => [ '4', '0', '0', '1' ] },
                '3,0' => { guess => [ '3', '0', '0', '1' ] }
            },
            '0,3' => {
                guess => [ '1', '0', '3', '4' ],
                '1,1' => { guess => [ '2', '0', '0', '1' ] },
                '2,0' => { guess => [ '1', '0', '5', '1' ] },
                '2,1' => { guess => [ '1', '0', '4', '1' ] },
                '3,0' => { guess => [ '1', '0', '3', '1' ] }
            },
            '0,4' => { guess => [ '1', '0', '2', '1' ] },
            '1,1' => {
                guess => [ '3', '4', '0', '1' ],
                '0,2' => { guess => [ '5', '0', '1', '0' ] },
                '0,3' => { guess => [ '4', '0', '1', '0' ] },
                '1,2' => { guess => [ '3', '0', '1', '0' ] },
                '2,0' => { guess => [ '0', '5', '0', '1' ] },
                '2,1' => { guess => [ '0', '3', '0', '1' ] },
                '3,0' => { guess => [ '0', '4', '0', '1' ] }
            },
            '1,2' => {
                guess => [ '1', '0', '3', '4' ],
                '0,2' => { guess => [ '0', '2', '0', '1' ] },
                '1,1' => { guess => [ '2', '0', '1', '0' ] },
                '2,0' => { guess => [ '1', '0', '1', '5' ] },
                '2,1' => { guess => [ '1', '0', '1', '3' ] },
                '3,0' => { guess => [ '1', '0', '1', '4' ] }
            },
            '2,0' => {
                guess => [ '0', '3', '0', '4' ],
                '1,1' => { guess => [ '0', '5', '1', '0' ] },
                '1,2' => { guess => [ '0', '4', '1', '0' ] },
                '2,1' => { guess => [ '0', '3', '1', '0' ] }
            },
            '2,1' => {
                guess => [ '0', '1', '3', '4' ],
                '1,1' => { guess => [ '0', '2', '1', '0' ] },
                '2,0' => { guess => [ '0', '1', '5', '1' ] },
                '2,1' => { guess => [ '0', '1', '4', '1' ] },
                '3,0' => { guess => [ '0', '1', '3', '1' ] }
            },
            '2,2' => {
                guess => [ '0', '0', '0', '1' ],
                '1,1' => { guess => [ '1', '0', '1', '2' ] },
                '2,0' => { guess => [ '0', '1', '2', '1' ] }
            },
            '3,0' => {
                guess => [ '0', '0', '3', '4' ],
                '1,0' => { guess => [ '0', '1', '1', '5' ] },
                '1,1' => { guess => [ '0', '1', '1', '3' ] },
                '2,0' => { guess => [ '0', '1', '1', '4' ] }
            }
        },
        '2,2' => {
            guess => [ '0', '1', '0', '2' ],
            '0,3' => { guess => [ '1', '0', '1', '0' ] },
            '1,2' => { guess => [ '1', '0', '0', '1' ] },
            '2,1' => { guess => [ '0', '1', '1', '0' ] },
            '3,0' => { guess => [ '0', '1', '0', '1' ] }
        },
        '3,0' => {
            guess => [ '0', '1', '1', '2' ],
            '1,1' => {
                guess => [ '0', '0', '0', '3' ],
                '2,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '2,0' => { guess => [ '0', '0', '5', '1' ] },
                    '2,1' => { guess => [ '0', '0', '4', '1' ] }
                },
                '2,1' => { guess => [ '0', '0', '3', '1' ] },
                '3,0' => { guess => [ '0', '0', '0', '1' ] }
            },
            '1,2' => {
                guess => [ '0', '0', '3', '4' ],
                '1,0' => {
                    guess => [ '0', '0', '0', '5' ],
                    '1,0' => { guess => [ '1', '0', '1', '1' ] },
                    '1,1' => { guess => [ '5', '0', '1', '1' ] }
                },
                '1,1' => {
                    guess => [ '0', '0', '0', '3' ],
                    '1,0' => { guess => [ '4', '0', '1', '1' ] },
                    '1,1' => { guess => [ '3', '0', '1', '1' ] }
                },
                '2,0' => { guess => [ '0', '0', '2', '1' ] }
            },
            '1,3' => { guess => [ '2', '0', '1', '1' ] },
            '2,0' => {
                guess => [ '0', '0', '0', '3' ],
                '2,0' => {
                    guess => [ '0', '0', '0', '4' ],
                    '2,0' => { guess => [ '0', '0', '1', '5' ] },
                    '3,0' => { guess => [ '0', '0', '1', '4' ] }
                },
                '2,1' => { guess => [ '0', '0', '1', '0' ] },
                '3,0' => { guess => [ '0', '0', '1', '3' ] }
            },
            '2,1' => {
                guess => [ '0', '3', '0', '4' ],
                '1,0' => { guess => [ '0', '5', '1', '1' ] },
                '1,1' => { guess => [ '0', '4', '1', '1' ] },
                '2,0' => { guess => [ '0', '3', '1', '1' ] }
            },
            '2,2' => { guess => [ '0', '2', '1', '1' ] },
            '3,0' => {
                guess => [ '0', '0', '0', '0' ],
                '1,0' => { guess => [ '0', '1', '1', '1' ] },
                '2,0' => { guess => [ '0', '0', '1', '2' ] }
            }
        }
    };
}
