package XMMS::InfoPipe::Playlist;
use strict;
use warnings;

our $VERSION = '0.01';
our $PIPE    = '/tmp/xmms-playlist-info';

=head1 NAME

XMMS::InfoPipe::Playlist -  A small module to gather the information produced
by the playlist-infopipe plugin for XMMS

=head1 SYNOPSIS

        use XMMS::InfoPipe::Playlist;
	
        my $pl = XMMS::InfoPipe::Playlist->new();
        
        for ($pl->get_playlist()) {
            print "Position: $_->{position}\n";
            print "Title: $_->{title}\n";
            print "File: $_->{file}\n\n";
        }

=head1 DESCRIPTION

This module was written to provide a way to snag the current XMMS playlist
from the pipe produced by the xmms-playlist-infopipe plugin for XMMS.

B<Nota Bene:> If the XMMS plugin isn't enabled, then this module will NOT
return results as expected (if it even works).

=head1 METHODS

=head2 new

    my $pl = XMMS::InfoPipe::Playlist->new();

Creates a new XMMS::InfoPipe::Playlist instance.  By default this parses the
file before returning the object.  This will undoubtedly cause some initial
slowdown (the bottleneck of XMMS::InfoPipe::Playlist is when it must grab 
information from the named pipe the XMMS plugin provides), and so you may
disable this first parsing by specifying a false value to ForceParse.  For
example:

    my $pl = XMMS::InfoPipe::Playlist->new(ForceParse => 0);

will create the object and immediately return it, without first populating
it with the information from XMMS.  This means that before trying to obtain
this information, you should first call C<$pl-E<gt>update_playlist()>.

=cut


sub new {
    my $class = shift;
    my %args = @_;
    
    $args{ForceParse} = 1 unless exists $args{ForceParse};
    
    my $self = bless {
        error       => '',
        playlist    => [],
        _force_parse=> $args{ForceParse}
    }, $class;
    
    $self->update_playlist if $self->{_force_parse};
    return $self;
}

=head2 get_playlist

    @list = $pl->get_playlist();

Returns an array of hashrefs that represents the current playlist.

=cut

sub get_playlist {
    my $self = shift;
    return @{$self->{playlist}};
}

=head2 xmms_is_running

    $pl->xmms_is_running()

Returns 1 if XMMS is running and 0 if not.  This relies on the fact that
the named pipe does not exist if XMMS is not running.  If the infopipe
plugin isn't enabled, this will also return 0.

=cut

sub is_running {
    my $self = shift;
    return $self->{playlist} ? 1 : 0;
}

=head2 update_playlist

    $pl->update_playlist()
    
Updates C<$pl-E<gt>{playlist}>.

=cut

sub update_playlist {
    my $self = shift;
    open DATA, $PIPE or return [];
    
    while (<DATA>) {
        next if /^#/;
        chomp;
        
        my ($pos, $title, $file) = split /\t/, $_, 3;
        push @{$self->{playlist}}, {
            position => $pos,
            title => $title,
            file => $file,
        };
    }
    close DATA;
}

=head1 VARIABLES

=head2 $XMMS::InfoPipe::Playlist::PIPE

    $XMMS::InfoPipe::Playlist::PIPE = '/tmp/other-name';

This variable defaults to C</tmp/xmms-playlist-info> which should be a
symlink (created by xmms-playlist-infopipe) to the real named pipe
(something like C</tmp/xmms-playlist-info_user.0>). If for whatever reason
you need to change it (maybe you have a file generated by something else
that follows the same format as xmms-playlist-infopipe), just set it before
C<update_playlist> is called (by default that means before C<new> is called)
for the right file to be used.

=head1 LICENSE

This module is free software, and may be distributed under the same
terms as Perl itself.

=head1 AUTHOR

Copyright (C) 2003, Thomas R. Sibley C<tsibley@cpan.org>

=cut

1;
