#! /usr/bin/env false

use v6.d;

unit module IO::Path::XDG;

#| Retrieve the value of XDG_CONFIG_DIRS as a sequence of IO::Path objects. If
#| this variable is not set, it will use the default value of /etc/xdg. These
#| directories should be read in order to retrieve configuration files.
sub xdg-config-dirs (
	#| When set to true, this sequence will start with the XDG_CONFIG_HOME
	#| value (or its default).
	Bool:D :$home = True,

	--> Iterable
) is export {
	(
		(xdg-config-home if $home),
		(%*ENV<XDG_CONFIG_DIRS> // '/etc/xdg')
			.split(':')
			.map(*.IO)
			.Slip,
	)
}

#| Returns an IO::Path for $XDG_CONFIG_HOME, if it exists as environment
#| variable. Otherwise, return the default value, $HOME/.config. This directory
#| should contain user-specific configuration files.
sub xdg-config-home (
	--> IO::Path
) is export {
	return %*ENV<XDG_CONFIG_HOME>.IO if %*ENV<XDG_CONFIG_HOME>:exists;

	$*HOME.add(".config");
}

#| Returns an IO::Path for $XDG_CACHE_HOME, if it exists as environment
#| variable. Otherwise, return the default value, $HOME/.cache. This directory
#| should contain user-specific, non-essential (cached) data.
sub xdg-cache-home (
	--> IO::Path
) is export {
	return %*ENV<XDG_CACHE_HOME>.IO if %*ENV<XDG_CACHE_HOME>:exists;

	$*HOME.IO.add(".cache");
}

#| Retrieve the value of XDG_DATA_DIRS as a sequence of IO::Path objects. If
#| this variable is not set, it will use the default values of /usr/local/share
#| and /usr/share. These directories should be read in order to retrieve user
#| specific data files.
sub xdg-data-dirs (
	#| When set to true, this sequence will start with the XDG_DATA_HOME
	#| value (or its default).
	Bool:D :$home = True,

	--> Iterable
) is export {
	(
		(xdg-data-home if $home),
		(%*ENV<XDG_DATA_DIRS> // '/usr/local/share:/usr/share')
			.split(':')
			.map(*.IO)
			.Slip
	)
}

#| Returns an IO::Path for $XDG_DATA_HOME, if it exists as environment
#| variable. Otherwise, return the default value, $HOME/.local/share. This
#| directory should contain user-specific data files.
sub xdg-data-home (
	--> IO::Path
) is export {
	return %*ENV<XDG_DATA_HOME>.IO if %*ENV<XDG_DATA_HOME>:exists;

	$*HOME.add(".local/share");
}

#| Returns an IO::Path for $XDG_RUNTIME_DIR, if it exists as environment
#| variable. Otherwise, return an IO::Path to a temporary directory. This
#| directory should contain user-specific runtime files and other file objects.
sub xdg-runtime-dir (
	--> IO::Path
) is export {
	return %*ENV<XDG_RUNTIME_DIR>.IO if %*ENV<XDG_RUNTIME_DIR>:exists;

	# XDG_RUNTIME_DIR is the only XDG basedir variant that does not come with
	# defaults. However, there are a number of de facto standard locations to
	# make use of. Try them, and return whichever hits first that also seems to
	# exist on the user's system.
	[
		"/var/run".IO.add(+$*USER),
		"/var/run".IO.add(~$*USER),
		$*HOME.add(".local/run"),
		$*TMPDIR,
		$*CWD,
	]
	.grep(*.d)
	.first
	.add($*PID)
}

=begin pod

=NAME    IO::Path::XDG
=AUTHOR  Patrick Spek <p.spek@tyil.work>
=VERSION 0.2.0

=head1 Synopsis

=head1 Description

=head1 Examples

=head1 See also

=end pod

# vim: ft=raku noet
